/**
 * Shared utilities for membase commands
 */

import { readFileSync } from 'node:fs';
import { homedir } from 'node:os';
import { join } from 'node:path';

export interface MembaseConfig {
  endpoint: string;
  account: string;
  secretKey: string;
  agentName: string;
  workspaceDir: string;
}

/**
 * Load config from multiple sources (priority order):
 * 1. Environment variables
 * 2. openclaw.json (if exists)
 * 3. Default values
 */
export function loadConfig(): MembaseConfig {
  //Try to load from openclaw.json
  let openclawConfig: any = {};
  try {
    const configPath = join(homedir(), '.openclaw', 'openclaw.json');
    const config = JSON.parse(readFileSync(configPath, 'utf-8'));
    openclawConfig = config.plugins?.entries?.['memory-membase']?.config || {};
  } catch (error) {
    // File doesn't exist or invalid, use defaults
  }

  // Merge with environment variables (env vars have priority)
  return {
    endpoint: process.env.MEMBASE_ENDPOINT || openclawConfig.endpoint || 'https://testnet.hub.membase.io',
    account: process.env.MEMBASE_ACCOUNT || openclawConfig.account || '',
    secretKey: process.env.MEMBASE_SECRET_KEY || openclawConfig.secretKey || '',
    agentName: process.env.MEMBASE_AGENT_NAME || openclawConfig.agentName || 'openclaw-agent',
    workspaceDir: process.env.MEMBASE_WORKSPACE_DIR || openclawConfig.workspaceDir || join(homedir(), '.openclaw', 'workspace')
  };
}

/**
 * Parse command line arguments
 */
export function parseArgs(args: string[]): Record<string, any> {
  const options: Record<string, any> = {};
  const positional: string[] = [];

  for (let i = 0; i < args.length; i++) {
    const arg = args[i];

    if (arg.startsWith('--')) {
      const key = arg.substring(2);

      // Check if next arg is a value or another flag
      if (i + 1 < args.length && !args[i + 1].startsWith('-')) {
        options[key] = args[++i];
      } else {
        // Boolean flag
        options[key] = true;
      }
    } else if (arg.startsWith('-') && arg.length === 2) {
      // Short flag
      const key = arg.substring(1);

      if (i + 1 < args.length && !args[i + 1].startsWith('-')) {
        options[key] = args[++i];
      } else {
        options[key] = true;
      }
    } else {
      // Positional argument
      positional.push(arg);
    }
  }

  options._positional = positional;
  return options;
}
