/**
 * Status command - Show backup status and statistics
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const manager = new BackupManager(client, config.workspaceDir, config.agentName);

    // Get status
    const status = await manager.getStatus();

    console.log('\n[STATS] Backup Status\n');

    console.log('Local:');
    console.log(`  Files: ${status.local.fileCount}`);
    console.log(`  Size: ${Math.round(status.local.totalSize / 1024)} KB`);

    console.log('\nRemote:');
    console.log(`  Backups: ${status.remote.backupCount}`);

    console.log('\nConfiguration:');
    console.log(`  Endpoint: ${config.endpoint}`);
    console.log(`  Agent: ${config.agentName}`);
    console.log(`  Workspace: ${config.workspaceDir}`);

    // Output machine-readable JSON for agent parsing
    if (!options['no-json']) {
      console.log('\n---JSON_OUTPUT---');
      console.log(JSON.stringify({ status, config }, null, 2));
      console.log('---END_JSON---');
    }

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] Status check failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
