/**
 * Restore command - Restore memories from Membase backup
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  // Get backup ID from positional argument
  const backupId = options._positional[0];

  if (!backupId) {
    console.error('[ERROR] Error: Backup ID is required');
    console.error('   Usage: node membase.ts restore <backup-id> --password <password>');
    process.exit(1);
  }

  // Get password
  const password = options.password || options.p || process.env.MEMBASE_BACKUP_PASSWORD || '';

  if (!password) {
    console.error('[ERROR] Error: Decryption password is required');
    console.error('   Set MEMBASE_BACKUP_PASSWORD or use --password option');
    process.exit(1);
  }

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const manager = new BackupManager(client, config.workspaceDir, config.agentName);

    // Execute restore
    console.log(`[SYNC] Downloading backup: ${backupId}`);
    const result = await manager.restore(backupId, password);

    // Output results
    console.log('\n[OK] Restore completed');
    console.log(`  Files restored: ${result.fileCount}`);
    console.log(`  Total size: ${Math.round(result.totalSize / 1024)} KB`);
    console.log(`  Agent: ${result.agentName}`);
    console.log(`  Backup date: ${result.timestamp}`);
    console.log(`  Location: ${config.workspaceDir}`);

    // Output machine-readable JSON for agent parsing
    if (!options['no-json']) {
      console.log('\n---JSON_OUTPUT---');
      console.log(JSON.stringify(result, null, 2));
      console.log('---END_JSON---');
    }

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] Restore failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
