/**
 * List command - List all available backups
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const manager = new BackupManager(client, config.workspaceDir, config.agentName);

    // List backups
    const backups = await manager.listBackups();

    if (backups.length === 0) {
      console.log('No backups found.');
      process.exit(0);
    }

    console.log('\nAvailable backups:\n');
    console.log('ID                            Timestamp              Files  Size');
    console.log('─'.repeat(70));

    for (const backup of backups) {
      const date = backup.timestamp;
      const size = backup.size ? `${Math.round(backup.size / 1024)} KB` : 'N/A';
      const files = String(backup.fileCount).padEnd(7);

      console.log(`${backup.id.padEnd(30)} ${date.padEnd(22)} ${files} ${size}`);
    }

    // Output machine-readable JSON for agent parsing
    if (!options['no-json']) {
      console.log('\n---JSON_OUTPUT---');
      console.log(JSON.stringify(backups, null, 2));
      console.log('---END_JSON---');
    }

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] List failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
