/**
 * Diff command - Compare two backups
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  // Get backup IDs from positional arguments
  const [backupId1, backupId2] = options._positional;

  if (!backupId1 || !backupId2) {
    console.error('[ERROR] Error: Two backup IDs are required');
    console.error('   Usage: node membase.ts diff <backup-id-1> <backup-id-2> --password <password>');
    process.exit(1);
  }

  // Get password
  const password = options.password || options.p || process.env.MEMBASE_BACKUP_PASSWORD || '';

  if (!password) {
    console.error('[ERROR] Error: Decryption password is required');
    console.error('   Set MEMBASE_BACKUP_PASSWORD or use --password option');
    process.exit(1);
  }

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const manager = new BackupManager(client, config.workspaceDir, config.agentName);

    // Execute diff
    console.log(`\n[SEARCH] Comparing backups:\n`);
    console.log(`  ${backupId1}`);
    console.log(`  ${backupId2}\n`);

    const diff = await manager.diffBackups(backupId1, backupId2, password);

    // Output results
    if (diff.added.length > 0) {
      console.log(`Added files (${diff.added.length}):`);
      for (const file of diff.added) {
        console.log(`  + ${file}`);
      }
      console.log('');
    }

    if (diff.removed.length > 0) {
      console.log(`Removed files (${diff.removed.length}):`);
      for (const file of diff.removed) {
        console.log(`  - ${file}`);
      }
      console.log('');
    }

    if (diff.modified.length > 0) {
      console.log(`Modified files (${diff.modified.length}):`);
      for (const file of diff.modified) {
        console.log(`  ~ ${file}`);
      }
      console.log('');
    }

    if (diff.added.length === 0 && diff.removed.length === 0 && diff.modified.length === 0) {
      console.log('No differences found.');
    }

    // Output machine-readable JSON for agent parsing
    if (!options['no-json']) {
      console.log('\n---JSON_OUTPUT---');
      console.log(JSON.stringify(diff, null, 2));
      console.log('---END_JSON---');
    }

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] Diff failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
