/**
 * Cleanup command - Clean up old backups
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  const keepLast = parseInt(options['keep-last'] || '10');
  const dryRun = options['dry-run'] || false;

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const manager = new BackupManager(client, config.workspaceDir, config.agentName);

    // List backups
    const backups = await manager.listBackups();

    if (backups.length <= keepLast) {
      console.log(`\n[INFO]  Only ${backups.length} backups exist (keeping last ${keepLast})`);
      console.log('Nothing to clean up.');
      process.exit(0);
    }

    const toDelete = backups.slice(keepLast);
    console.log(`\n[CLEANUP] Found ${backups.length} backups, will ${dryRun ? 'would delete' : `keep last ${keepLast}`}`);
    console.log(`\nBackups to delete (${toDelete.length}):\n`);

    for (const backup of toDelete) {
      console.log(`  - ${backup.id} (${backup.timestamp})`);
    }

    if (dryRun) {
      console.log('\n(Dry run - no backups deleted)');
      process.exit(0);
    }

    console.log('\n[WARNING]  Note: Membase doesn\'t currently support delete API.');
    console.log('   Please manually clean up via Membase Hub UI.');
    console.log(`   URL: ${config.endpoint}/account/${config.account}`);

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] Cleanup failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
