/**
 * Backup command - Backup memories to Membase
 */

import { BackupManager } from '../lib/backup-manager.js';
import { MembaseClient } from '../lib/membase-client.js';
import { MemoryEncryption } from '../lib/encryption.js';
import { loadConfig, parseArgs } from './utils.js';

export async function run(args: string[]) {
  const options = parseArgs(args);
  const config = loadConfig();

  // Get password
  const password = options.password || options.p || process.env.MEMBASE_BACKUP_PASSWORD || '';

  if (!password) {
    console.error('[ERROR] Error: Backup password is required');
    console.error('   Set MEMBASE_BACKUP_PASSWORD or use --password option');
    process.exit(1);
  }

  // Validate password strength (unless disabled)
  if (options.validate !== false && !process.env.CI && !process.env.TEST) {
    try {
      MemoryEncryption.validatePassword(password);
    } catch (error) {
      console.error(`[ERROR] Error: ${error instanceof Error ? error.message : 'Invalid password'}`);
      console.error('   Password must be at least 12 characters with uppercase, lowercase, and numbers');
      console.error('   Use --no-validate to skip password strength check');
      process.exit(1);
    }
  }

  // Validate Membase credentials
  if (!config.account || !config.secretKey) {
    console.error('[ERROR] Error: Membase credentials not configured');
    console.error('   Set MEMBASE_ACCOUNT and MEMBASE_SECRET_KEY environment variables');
    process.exit(1);
  }

  try {
    // Initialize client and manager
    const client = new MembaseClient(config.endpoint, config.account, config.secretKey);
    const workspaceDir = options.workspace || config.workspaceDir;
    const manager = new BackupManager(client, workspaceDir, config.agentName);

    // Execute backup
    console.log('[SEARCH] Scanning memory files...');
    const result = await manager.backup(password, {
      incremental: options.incremental || options.i || false
    });

    // Output results
    console.log('\n[OK] Backup completed');
    console.log(`  Backup ID: ${result.backupId}`);
    console.log(`  Files: ${result.fileCount}`);

    if (result.incremental && result.skippedFiles) {
      console.log(`  Skipped: ${result.skippedFiles} unchanged files`);
    }

    console.log(`  Size: ${Math.round(result.totalSize / 1024)} KB`);
    console.log(`  Timestamp: ${result.timestamp}`);

    if (result.incremental) {
      console.log('  Type: Incremental');
    }

    console.log('\n[WARNING]  Save your backup ID and password securely!');

    // Output machine-readable JSON for agent parsing
    if (!options['no-json']) {
      console.log('\n---JSON_OUTPUT---');
      console.log(JSON.stringify(result, null, 2));
      console.log('---END_JSON---');
    }

    process.exit(0);
  } catch (error) {
    console.error(`\n[ERROR] Backup failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}
