---
name: membase
description: Manage agent memory with Membase - a decentralized, encrypted memory backup and restore system. Provides backup, restore, list, diff, status, and cleanup operations for agent memories.
license: MIT
metadata:
  author: Unibase
  version: "1.0.0"
  category: memory-management
  tags: [backup, encryption, membase, storage, decentralized, memory]
allowed-tools:
  - bash
---

# Membase Memory Management

Membase provides secure, decentralized memory storage for AI agents with end-to-end encryption.

## When to Use This Skill

Activate this skill when the user:
- Asks to backup their memories, conversations, or workspace
- Wants to restore previous memories or conversations
- Wants to see available backups
- Asks to compare different backup versions
- Wants to check backup status
- Mentions "membase" or "backup memories"

## Overview

All Membase operations go through a single command:

```bash
node membase.ts <command> [options]
```

Available commands:
- `backup` - Backup memories to Membase
- `restore` - Restore memories from a backup
- `list` - List all available backups
- `diff` - Compare two backups
- `status` - Show backup status and statistics
- `cleanup` - Clean up old backups

## Configuration

### Environment Variables

```bash
export MEMBASE_ACCOUNT=your-account-address
export MEMBASE_SECRET_KEY=your-secret-key
export MEMBASE_BACKUP_PASSWORD=your-backup-password
export MEMBASE_ENDPOINT=https://testnet.hub.membase.io
```

Check if configured:
```bash
echo $MEMBASE_ACCOUNT
echo $MEMBASE_SECRET_KEY
echo $MEMBASE_BACKUP_PASSWORD
```

## Commands

### 1. backup - Backup Memories

Backs up agent memory files (MEMORY.md, memory/**/*.md) to Membase with AES-256-GCM encryption.

**Usage:**
```bash
node membase.ts backup [options]
```

**Options:**
- `--password <pwd>` or `-p <pwd>` - Encryption password (required if not in env)
- `--incremental` or `-i` - Only backup changed files since last backup
- `--workspace <path>` - Custom workspace directory
- `--no-validate` - Skip password strength validation
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "Please backup my memories"

You should:
1. Check for MEMBASE_BACKUP_PASSWORD:
   ```bash
   echo $MEMBASE_BACKUP_PASSWORD
   ```

2. If not set, ask: "Please provide a backup password for encryption (at least 12 characters with uppercase, lowercase, and numbers):"

3. Run backup:
   ```bash
   cd skills/membase
   node membase.ts backup --password "<password>"
   ```

4. Show result to user:
   ```
   [OK] Backup completed
   Backup ID: backup-2026-02-02T10-30-45-123Z
   Files: 15
   Size: 234 KB

   [WARNING]  Save your backup ID and password securely!
   ```

**Incremental backup (faster):**
```bash
node membase.ts backup --password "<password>" --incremental
```

### 2. restore - Restore Memories

Restores memories from a Membase backup.

**Usage:**
```bash
node membase.ts restore <backup-id> [options]
```

**Options:**
- `<backup-id>` - The backup ID to restore (required)
- `--password <pwd>` or `-p <pwd>` - Decryption password (required if not in env)
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "Restore my memories from backup-2026-02-02T10-30-45-123Z"

You should:
1. Check for password:
   ```bash
   echo $MEMBASE_BACKUP_PASSWORD
   ```

2. Run restore:
   ```bash
   cd skills/membase
   node membase.ts restore backup-2026-02-02T10-30-45-123Z --password "<password>"
   ```

3. Show result:
   ```
   [OK] Restore completed
   Files restored: 15
   Total size: 234 KB
   Location: ~/.openclaw/workspace/
   ```

### 3. list - List Backups

Lists all available backups for this agent.

**Usage:**
```bash
node membase.ts list [options]
```

**Options:**
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "Show me my backups" or "List my backups"

You should:
```bash
cd skills/membase
node membase.ts list
```

Output will show:
```
Available backups:

ID                            Timestamp              Files  Size
──────────────────────────────────────────────────────────────────
backup-2026-02-02T10-30-45-123Z 2026-02-02 10:30:45    15     234 KB
backup-2026-02-01T15-20-10-456Z 2026-02-01 15:20:10    12     198 KB
```

### 4. diff - Compare Backups

Compares two backups to see what changed.

**Usage:**
```bash
node membase.ts diff <backup-id-1> <backup-id-2> [options]
```

**Options:**
- `<backup-id-1>` - First backup ID (required)
- `<backup-id-2>` - Second backup ID (required)
- `--password <pwd>` or `-p <pwd>` - Decryption password (required if not in env)
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "What changed between my last two backups?"

You should:
1. Get the two most recent backup IDs:
   ```bash
   cd skills/membase
   node membase.ts list
   ```

2. Run diff with the two IDs:
   ```bash
   node membase.ts diff backup-2026-02-02T10-30-45-123Z backup-2026-02-01T15-20-10-456Z --password "<password>"
   ```

3. Show result:
   ```
   Added files (2):
     + memory/conversation-new.md
     + memory/notes.md

   Modified files (1):
     ~ MEMORY.md
   ```

### 5. status - Show Status

Shows backup status and statistics.

**Usage:**
```bash
node membase.ts status [options]
```

**Options:**
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "What's my backup status?" or "Check backup status"

You should:
```bash
cd skills/membase
node membase.ts status
```

Output shows:
```
[STATS] Backup Status

Local:
  Files: 15
  Size: 234 KB

Remote:
  Backups: 10

Configuration:
  Endpoint: https://testnet.hub.membase.io
  Agent: my-agent
  Workspace: ~/.openclaw/workspace
```

### 6. cleanup - Clean Up Old Backups

Lists old backups that could be deleted (Membase doesn't support delete API yet).

**Usage:**
```bash
node membase.ts cleanup [options]
```

**Options:**
- `--keep-last <n>` - Keep last N backups (default: 10)
- `--dry-run` - Show what would be deleted without deleting
- `--no-json` - Don't output JSON for agent parsing

**Example conversation:**

User: "Clean up old backups, keep the last 5"

You should:
```bash
cd skills/membase
node membase.ts cleanup --keep-last 5
```

Note: Will show which backups should be deleted, but user needs to delete manually via Membase Hub UI.

## Security Notes

- All data is encrypted **client-side** with AES-256-GCM
- Password is derived using PBKDF2 with 100,000 iterations
- Your password **never** leaves the local machine
- Membase storage is decentralized and zero-knowledge
- Only you can decrypt your backups

## Password Requirements

- At least 12 characters
- Must contain uppercase letters
- Must contain lowercase letters
- Must contain numbers
- Recommended: Use a password manager

## Error Handling

### Missing credentials
If you see "Membase credentials not configured":
```bash
# User needs to set environment variables:
export MEMBASE_ACCOUNT=your-account
export MEMBASE_SECRET_KEY=your-key
```

### Missing password
If you see "Backup password is required":
- Ask user for password
- Or suggest setting MEMBASE_BACKUP_PASSWORD env var

### Invalid password
If you see "Invalid password" or "Decryption failed":
- User provided wrong password
- Ask for correct password

### No backups found
If list shows "No backups found":
- No backups exist yet
- Suggest creating first backup

### Network error
If connection fails:
- Check internet connection
- Verify MEMBASE_ENDPOINT is correct
- Try again later

## Tips for Agents

1. **Always check for password first** before asking user
2. **Show the backup ID** clearly so user can save it
3. **Parse JSON output** if available (between ---JSON_OUTPUT--- and ---END_JSON---)
4. **Be clear about security** - emphasize that password is required for restore
5. **Suggest incremental backups** for speed after first backup
6. **Remember backup IDs** from list command to help user with restore/diff

## Examples

### Complete backup workflow
```bash
# 1. Check status
node membase.ts status

# 2. First backup (full)
node membase.ts backup --password "MySecure123Pass"

# 3. Later: incremental backup
node membase.ts backup --password "MySecure123Pass" --incremental

# 4. List all backups
node membase.ts list

# 5. Compare recent backups
node membase.ts diff backup-id-1 backup-id-2 --password "MySecure123Pass"

# 6. Restore if needed
node membase.ts restore backup-id-1 --password "MySecure123Pass"
```

## Troubleshooting

### Command not found
Make sure you're in the skills/membase directory:
```bash
cd skills/membase
pwd  # Should show .../skills/membase
```

### Module not found
The lib folder needs to be linked to compiled source:
```bash
cd skills/membase
ln -sf ../../dist/lib lib
```

### Permission denied
Make membase.ts executable:
```bash
chmod +x membase.ts
```

## Learn More

- [Membase Documentation](https://github.com/unibaseio/membase)
- [AgentSkills Specification](https://agentskills.io)
- [OpenClaw Skills Guide](https://docs.openclaw.ai/tools/skills)
