#!/usr/bin/env npx ts-node
import { readFileSync, writeFileSync } from "fs";
import { dirname, join } from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);
const PATTERNS = JSON.parse(readFileSync(join(__dirname, "patterns.json"), "utf-8"));

const replacePhrases = (text: string): [string, string[]] => {
  const changes: string[] = [];
  for (const [old, replacement] of Object.entries(PATTERNS.replacements)) {
    const flags = "gi";
    const pattern = old.includes(" ") || old.endsWith(",")
      ? new RegExp(old.replace(/[.*+?^${}()|[\]\\]/g, "\\$&"), flags)
      : new RegExp(`\\b${old.replace(/[.*+?^${}()|[\]\\]/g, "\\$&")}\\b`, flags);
    const matches = text.match(pattern);
    if (matches) {
      changes.push(`"${old}" → ${replacement ? `"${replacement}"` : "(removed)"} (${matches.length}x)`);
      text = text.replace(pattern, replacement as string);
    }
  }
  return [text, changes];
};

const fixCurlyQuotes = (text: string): [string, boolean] => {
  const original = text;
  text = text.replace(/[""]/g, '"').replace(/['']/g, "'");
  return [text, text !== original];
};

const removeChatbotArtifacts = (text: string): [string, string[]] => {
  const changes: string[] = [];
  for (const artifact of PATTERNS.chatbot_artifacts) {
    const escaped = artifact.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    const pattern = new RegExp(`[^.!?\\n]*${escaped}[^.!?\\n]*[.!?]?\\s*`, "gi");
    if (pattern.test(text)) {
      changes.push(`Removed sentence with "${artifact}"`);
      text = text.replace(pattern, "");
    }
  }
  return [text, changes];
};

const reduceEmDashes = (text: string): [string, number] => {
  const count = (text.match(/—/g) || []).length;
  text = text.replace(/\s*—\s*/g, ", ").replace(/--/g, ", ");
  return [text, count];
};

const cleanWhitespace = (text: string): string => {
  return text.replace(/ +/g, " ").replace(/\n{3,}/g, "\n\n").replace(/^\s+/gm, "").trim();
};

const fixCapitalization = (text: string): string => {
  return text
    .replace(/(^|[.!?]\s+)([a-z])/g, (_, pre, char) => pre + char.toUpperCase())
    .replace(/^\s*([a-z])/, (_, char) => char.toUpperCase());
};

interface Options {
  fixDashes?: boolean;
  removeArtifacts?: boolean;
}

const transform = (text: string, opts: Options = {}): [string, string[]] => {
  const { fixDashes = false, removeArtifacts = true } = opts;
  const allChanges: string[] = [];

  let changes: string[];
  [text, changes] = replacePhrases(text);
  allChanges.push(...changes);

  let fixed: boolean;
  [text, fixed] = fixCurlyQuotes(text);
  if (fixed) allChanges.push("Fixed curly quotes → straight quotes");

  if (removeArtifacts) {
    [text, changes] = removeChatbotArtifacts(text);
    allChanges.push(...changes);
  }

  if (fixDashes) {
    let count: number;
    [text, count] = reduceEmDashes(text);
    if (count) allChanges.push(`Replaced ${count} em dashes with commas`);
  }

  text = cleanWhitespace(text);
  text = fixCapitalization(text);

  return [text, allChanges];
};

const main = () => {
  const args = process.argv.slice(2);
  const quiet = args.includes("-q") || args.includes("--quiet");
  const fixDashes = args.includes("--fix-dashes");
  const keepArtifacts = args.includes("--keep-artifacts");
  const outputIdx = args.findIndex((a) => a === "-o" || a === "--output");
  const outputFile = outputIdx !== -1 ? args[outputIdx + 1] : null;
  const inputFile = args.find((a) => !a.startsWith("-") && a !== outputFile);

  let text: string;
  if (inputFile) {
    text = readFileSync(inputFile, "utf-8");
  } else {
    text = readFileSync(0, "utf-8");
  }

  const [result, changes] = transform(text, { fixDashes, removeArtifacts: !keepArtifacts });

  if (!quiet && changes.length) {
    console.error("CHANGES MADE:");
    changes.forEach((c) => console.error(`  • ${c}`));
    console.error();
  }

  if (outputFile) {
    writeFileSync(outputFile, result);
    if (!quiet) console.error(`Written to ${outputFile}`);
  } else {
    console.log(result);
  }
};

main();
