#!/usr/bin/env npx ts-node
import { readFileSync } from "fs";
import { dirname, join } from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);
const PATTERNS = JSON.parse(readFileSync(join(__dirname, "patterns.json"), "utf-8"));

type Match = [string, number];
interface Results {
  ai_words: Match[];
  puffery: Match[];
  chatbot_artifacts: Match[];
  hedging: Match[];
  em_dashes: number;
  curly_quotes: number;
  replaceable: Match[];
  total_issues: number;
}

const findMatches = (text: string, patterns: string[]): Match[] => {
  const lower = text.toLowerCase();
  return patterns
    .map((p): Match => [p, (lower.match(new RegExp(p.toLowerCase(), "g")) || []).length])
    .filter(([, c]) => c > 0)
    .sort((a, b) => b[1] - a[1]);
};

const analyze = (text: string): Results => {
  const results: Results = {
    ai_words: findMatches(text, PATTERNS.ai_words),
    puffery: findMatches(text, PATTERNS.puffery),
    chatbot_artifacts: findMatches(text, PATTERNS.chatbot_artifacts),
    hedging: findMatches(text, PATTERNS.hedging_phrases),
    em_dashes: (text.match(/—|--/g) || []).length,
    curly_quotes: (text.match(/[""]/g) || []).length,
    replaceable: Object.keys(PATTERNS.replacements)
      .map((k): Match => [k, (text.toLowerCase().match(new RegExp(k.toLowerCase(), "g")) || []).length])
      .filter(([, c]) => c > 0),
    total_issues: 0,
  };
  results.total_issues =
    results.ai_words.reduce((s, [, c]) => s + c, 0) +
    results.puffery.reduce((s, [, c]) => s + c, 0) +
    results.chatbot_artifacts.reduce((s, [, c]) => s + c, 0) +
    results.hedging.reduce((s, [, c]) => s + c, 0) +
    results.em_dashes +
    results.curly_quotes +
    results.replaceable.reduce((s, [, c]) => s + c, 0);
  return results;
};

const printReport = (r: Results) => {
  console.log(`\n${"=".repeat(50)}`);
  console.log(`AI DETECTION SCAN - ${r.total_issues} issues found`);
  console.log(`${"=".repeat(50)}\n`);

  if (r.ai_words.length) {
    console.log("AI VOCABULARY:");
    r.ai_words.forEach(([w, c]) => console.log(`  • ${w}: ${c}x`));
    console.log();
  }
  if (r.puffery.length) {
    console.log("PUFFERY/PROMOTIONAL:");
    r.puffery.forEach(([w, c]) => console.log(`  • ${w}: ${c}x`));
    console.log();
  }
  if (r.chatbot_artifacts.length) {
    console.log("CHATBOT ARTIFACTS:");
    r.chatbot_artifacts.forEach(([p, c]) => console.log(`  • "${p}": ${c}x`));
    console.log();
  }
  if (r.hedging.length) {
    console.log("EXCESSIVE HEDGING:");
    r.hedging.forEach(([p, c]) => console.log(`  • "${p}": ${c}x`));
    console.log();
  }
  if (r.replaceable.length) {
    console.log("AUTO-FIXABLE:");
    r.replaceable.forEach(([p, c]) => {
      const repl = PATTERNS.replacements[p];
      console.log(`  • "${p}" → ${repl ? `"${repl}"` : "(remove)"}: ${c}x`);
    });
    console.log();
  }
  if (r.em_dashes > 2) console.log(`EM DASHES: ${r.em_dashes} (consider reducing)\n`);
  if (r.curly_quotes) console.log(`CURLY QUOTES: ${r.curly_quotes} (replace with straight)\n`);
  if (r.total_issues === 0) console.log("✓ Text appears human-written.\n");
};

const main = () => {
  const args = process.argv.slice(2);
  const jsonOutput = args.includes("--json");
  const file = args.find((a) => !a.startsWith("-"));

  let text: string;
  if (file) {
    text = readFileSync(file, "utf-8");
  } else {
    text = readFileSync(0, "utf-8");
  }

  const results = analyze(text);
  if (jsonOutput) {
    console.log(JSON.stringify(results, null, 2));
  } else {
    printReport(results);
  }
};

main();
