#!/usr/bin/env python3
"""Ultrahuman daily summary via mcporter.

Usage:
  ultrahuman_summary.py [--date YYYY-MM-DD] [--yesterday] [--mcporter-config PATH]

Reads Ultrahuman MCP data via mcporter stdio server, then prints a concise
human-readable summary.

Requires:
- mcporter in PATH
- mcporter config that defines the `ultrahuman` MCP server
- env vars: ULTRAHUMAN_AUTH_TOKEN, ULTRAHUMAN_USER_EMAIL
"""

from __future__ import annotations

import argparse
import datetime as dt
import json
import os
import subprocess
import sys
from typing import Any, Dict, Optional

DEFAULT_MCPORTER_CONFIG = "config/mcporter.json"


def _sg_today() -> dt.date:
    # Asia/Singapore is UTC+8 with no DST.
    now_utc = dt.datetime.now(dt.timezone.utc)
    now_sg = now_utc.astimezone(dt.timezone(dt.timedelta(hours=8)))
    return now_sg.date()


def _pick_date(date_str: Optional[str], yesterday: bool) -> str:
    if date_str and yesterday:
        raise SystemExit("Pass either --date or --yesterday, not both")

    if date_str:
        # basic validation
        try:
            dt.date.fromisoformat(date_str)
        except ValueError:
            raise SystemExit("--date must be YYYY-MM-DD")
        return date_str

    if yesterday:
        d = _sg_today() - dt.timedelta(days=1)
        return d.isoformat()

    # default: yesterday (most common)
    return (_sg_today() - dt.timedelta(days=1)).isoformat()


def _resolve_mcporter_config(path: str) -> str:
    # Accept explicit path (absolute or relative).
    if os.path.isabs(path) and os.path.exists(path):
        return path

    # Try relative to current working directory.
    cwd_path = os.path.abspath(path)
    if os.path.exists(cwd_path):
        return cwd_path

    # Try OpenClaw workspace, if provided.
    ws = os.environ.get("OPENCLAW_WORKSPACE")
    if ws:
        ws_path = os.path.join(ws, path)
        if os.path.exists(ws_path):
            return ws_path

    # Try default OpenClaw workspace location.
    home_ws = os.path.expanduser("~/.openclaw/workspace")
    home_ws_path = os.path.join(home_ws, path)
    if os.path.exists(home_ws_path):
        return home_ws_path

    raise FileNotFoundError(
        "Could not find mcporter config. Set --mcporter-config or MCPORTER_CONFIG, "
        "or run from your OpenClaw workspace so ./config/mcporter.json exists."
    )


def _mcporter_call(date_str: str, config_path: str) -> Dict[str, Any]:
    config_path = _resolve_mcporter_config(config_path)

    cmd = [
        "mcporter",
        "--config",
        config_path,
        "call",
        "ultrahuman.ultrahuman_metrics",
        f"date={date_str}",
        "--output",
        "json",
    ]

    p = subprocess.run(cmd, capture_output=True, text=True)
    if p.returncode != 0:
        raise RuntimeError(
            f"mcporter failed (exit {p.returncode}).\nSTDOUT:\n{p.stdout}\nSTDERR:\n{p.stderr}"
        )

    # mcporter sometimes prints non-json banners; attempt to locate the JSON object.
    out = p.stdout.strip()
    first_brace = out.find("{")
    if first_brace == -1:
        raise RuntimeError(f"Unexpected mcporter output (no JSON found):\n{out}")
    out = out[first_brace:]

    return json.loads(out)


def _find_item(items: list[dict[str, Any]], type_name: str) -> Optional[dict[str, Any]]:
    for it in items:
        if it.get("type") == type_name:
            return it.get("object")
    return None


def _get(obj: Optional[dict[str, Any]], *path: str) -> Any:
    cur: Any = obj
    for k in path:
        if not isinstance(cur, dict) or k not in cur:
            return None
        cur = cur[k]
    return cur


def _fmt_duration(seconds: Optional[int]) -> str:
    if not seconds and seconds != 0:
        return "—"
    m = int(seconds) // 60
    h = m // 60
    mm = m % 60
    return f"{h}h {mm}m" if h else f"{mm}m"


def _fmt_int(x: Any) -> str:
    if x is None:
        return "—"
    try:
        return str(int(x))
    except Exception:
        return str(x)


def _fmt_float(x: Any, ndigits: int = 2) -> str:
    if x is None:
        return "—"
    try:
        return f"{float(x):.{ndigits}f}"
    except Exception:
        return str(x)


def summarize(date_str: str, payload: Dict[str, Any]) -> str:
    data = payload.get("data") or {}
    items = data.get("items") or data.get("metric_data") or []

    sleep = _find_item(items, "Sleep")
    steps = _find_item(items, "steps")

    recovery = _find_item(items, "recovery_index")
    movement = _find_item(items, "movement_index")
    active_minutes = _find_item(items, "active_minutes")
    vo2 = _find_item(items, "vo2_max")
    sleep_rhr = _find_item(items, "sleep_rhr")
    avg_sleep_hrv = _find_item(items, "avg_sleep_hrv")
    night_rhr = _find_item(items, "night_rhr")

    # Sleep breakdown
    sleep_score = _get(sleep, "sleep_score", "score")
    total_sleep_sec = _get(sleep, "total_sleep", "seconds")
    time_in_bed_min = _get(sleep, "time_in_bed", "minutes")
    efficiency_pct = _get(sleep, "sleep_efficiency", "percentage")
    restorative_pct = _get(sleep, "restorative_sleep", "percentage")

    deep_min = _get(sleep, "deep_sleep", "minutes")
    rem_min = _get(sleep, "rem_sleep", "minutes")

    steps_total = _get(steps, "total")

    # Some are nested as { value: n }
    rec_val = _get(recovery, "value")
    mov_val = _get(movement, "value")
    active_val = _get(active_minutes, "value")
    vo2_val = _get(vo2, "value")
    sleep_rhr_val = _get(sleep_rhr, "value")
    avg_hrv_val = _get(avg_sleep_hrv, "value")

    # night_rhr is a series; use avg if present
    night_rhr_avg = _get(night_rhr, "avg")

    lines: list[str] = []
    lines.append(f"Ultrahuman summary — {date_str} (SG)")

    if sleep_score is not None or total_sleep_sec is not None:
        lines.append(
            "Sleep: "
            + f"score { _fmt_int(sleep_score) } | "
            + f"total { _fmt_duration(total_sleep_sec) } | "
            + (f"in bed {int(time_in_bed_min//60)}h {int(time_in_bed_min%60)}m | " if isinstance(time_in_bed_min, (int, float)) else "in bed — | ")
            + f"eff { _fmt_int(efficiency_pct) }% | "
            + f"restorative { _fmt_int(restorative_pct) }%"
        )
        if deep_min is not None or rem_min is not None:
            lines.append(f"  Stages: deep {_fmt_int(deep_min)}m | REM {_fmt_int(rem_min)}m")
    else:
        lines.append("Sleep: —")

    lines.append(f"Steps: {_fmt_int(steps_total)}")

    lines.append(
        "Recovery/Activity: "
        + f"recovery {_fmt_int(rec_val)} | "
        + f"movement {_fmt_int(mov_val)} | "
        + f"active min {_fmt_int(active_val)}"
    )

    lines.append(
        "Cardio: "
        + f"VO₂ max {_fmt_int(vo2_val)} | "
        + f"HRV (sleep avg) {_fmt_int(avg_hrv_val)} | "
        + f"RHR (sleep) {_fmt_int(sleep_rhr_val)} | "
        + f"RHR (night avg) {_fmt_int(night_rhr_avg)}"
    )

    return "\n".join(lines)


def main() -> int:
    ap = argparse.ArgumentParser()
    ap.add_argument("--date", help="Date in YYYY-MM-DD")
    ap.add_argument("--yesterday", action="store_true", help="Use yesterday in Asia/Singapore")
    ap.add_argument(
        "--mcporter-config",
        default=os.environ.get("MCPORTER_CONFIG", DEFAULT_MCPORTER_CONFIG),
        help=(
            "Path to mcporter config. Accepts absolute path, or relative paths "
            "(resolved against CWD, OPENCLAW_WORKSPACE, and ~/.openclaw/workspace). "
            f"Default: {DEFAULT_MCPORTER_CONFIG}"
        ),
    )

    args = ap.parse_args()

    # Ensure we have creds in env for mcporter substitution.
    # Prefer process env; fallback to OpenClaw config env.vars.
    if not os.environ.get("ULTRAHUMAN_AUTH_TOKEN") or not os.environ.get("ULTRAHUMAN_USER_EMAIL"):
        cfg_path = os.path.expanduser("~/.openclaw/openclaw.json")
        try:
            with open(cfg_path, "r", encoding="utf-8") as f:
                cfg = json.load(f)
            vars_ = (cfg.get("env") or {}).get("vars") or {}
            if not os.environ.get("ULTRAHUMAN_AUTH_TOKEN") and vars_.get("ULTRAHUMAN_AUTH_TOKEN"):
                os.environ["ULTRAHUMAN_AUTH_TOKEN"] = vars_["ULTRAHUMAN_AUTH_TOKEN"]
            if not os.environ.get("ULTRAHUMAN_USER_EMAIL") and vars_.get("ULTRAHUMAN_USER_EMAIL"):
                os.environ["ULTRAHUMAN_USER_EMAIL"] = vars_["ULTRAHUMAN_USER_EMAIL"]
        except Exception:
            pass

    if not os.environ.get("ULTRAHUMAN_AUTH_TOKEN"):
        print("Missing ULTRAHUMAN_AUTH_TOKEN (env var or ~/.openclaw/openclaw.json env.vars)", file=sys.stderr)
        return 2
    if not os.environ.get("ULTRAHUMAN_USER_EMAIL"):
        print("Missing ULTRAHUMAN_USER_EMAIL (env var or ~/.openclaw/openclaw.json env.vars)", file=sys.stderr)
        return 2

    date_str = _pick_date(args.date, args.yesterday)

    payload = _mcporter_call(date_str, args.mcporter_config)
    print(summarize(date_str, payload))
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
