# Accessibility (WCAG 2.2) Compliance Guide

Complete guide to building accessible web applications that meet WCAG 2.2 standards.

---

## Table of Contents

1. [WCAG Overview](#wcag-overview)
2. [Contrast Ratios](#contrast-ratios)
3. [Keyboard Navigation](#keyboard-navigation)
4. [Screen Reader Support](#screen-reader-support)
5. [Focus Management](#focus-management)
6. [ARIA Attributes](#aria-attributes)
7. [Form Accessibility](#form-accessibility)
8. [Interactive Elements](#interactive-elements)
9. [Testing Tools](#testing-tools)
10. [Quick Checklist](#quick-checklist)

---

## WCAG Overview

**WCAG 2.2** (Web Content Accessibility Guidelines) has three conformance levels:
- **Level A:** Minimum accessibility (basic)
- **Level AA:** Mid-range accessibility (target for most sites)
- **Level AAA:** Highest accessibility (specialized needs)

**Target:** WCAG 2.2 Level AA compliance for all projects.

**Four Principles (POUR):**
1. **Perceivable:** Information must be presentable to users
2. **Operable:** Interface components must be operable
3. **Understandable:** Information and UI must be understandable
4. **Robust:** Content must work with assistive technologies

---

## Contrast Ratios

### Requirements

**Text Contrast:**
- **Normal text (< 24px):** 4.5:1 minimum
- **Large text (≥ 24px or 19px bold):** 3:1 minimum

**UI Components:**
- **Buttons, inputs, icons:** 3:1 minimum vs background
- **Focus indicators:** 3:1 minimum vs unfocused state

### Testing Contrast

**Tools:**
- WebAIM Contrast Checker: https://webaim.org/resources/contrastchecker/
- Chrome DevTools: Inspect element → Styles → Color picker shows contrast ratio
- Figma/Sketch: Built-in contrast checkers

**Examples:**

```css
/* ✅ Good: 7:1 ratio */
color: #000000;
background: #ffffff;

/* ✅ Good: 4.5:1 ratio */
color: #595959;
background: #ffffff;

/* ❌ Bad: 2.5:1 ratio (fails AA) */
color: #999999;
background: #ffffff;
```

### Common Failures

1. **Light gray text on white:** #999 on white = 2.9:1 (fails)
2. **Blue links on dark backgrounds:** Check contrast carefully
3. **Placeholder text:** Often too light, make it darker
4. **Disabled buttons:** Exempt from contrast rules, but should still be visible

---

## Keyboard Navigation

### Requirements

**Every interactive element must be keyboard-accessible:**
- Tab/Shift+Tab: Navigate between elements
- Enter/Space: Activate buttons and links
- Escape: Close modals and dropdowns
- Arrow keys: Navigate within components (menus, radios, tabs)

### Tab Order

**Logical tab order** matches visual layout:

```jsx
// ✅ Good: Natural DOM order
<nav>
  <a href="/">Home</a>
  <a href="/about">About</a>
  <a href="/contact">Contact</a>
</nav>

// ❌ Bad: Using tabIndex to override
<nav>
  <a href="/" tabIndex={3}>Home</a>
  <a href="/about" tabIndex={1}>About</a>
  <a href="/contact" tabIndex={2}>Contact</a>
</nav>
```

**Only use tabIndex in three cases:**
- `tabIndex={0}`: Add to tab order (for custom interactive elements)
- `tabIndex={-1}`: Remove from tab order (programmatic focus only)
- `tabIndex={1+}`: Override order (avoid unless absolutely necessary)

### Focus Indicators

**All interactive elements must have visible focus:**

```css
/* ✅ Good: Visible focus outline */
button:focus-visible {
  outline: 3px solid #3b82f6;
  outline-offset: 2px;
}

/* ❌ Bad: Removing focus without replacement */
button:focus {
  outline: none;
}

/* ✅ Good: Custom focus style */
button:focus-visible {
  outline: none;
  box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.5);
}
```

**Focus indicators must:**
- Be visible (3:1 contrast vs unfocused)
- Be consistent across site
- Not rely on color alone

### Skip Links

Provide skip links for keyboard users:

```jsx
<a 
  href="#main-content" 
  className="sr-only focus:not-sr-only focus:absolute focus:top-4 focus:left-4 focus:z-50 focus:px-4 focus:py-2 focus:bg-blue-600 focus:text-white"
>
  Skip to main content
</a>

<main id="main-content">
  {/* Page content */}
</main>
```

---

## Screen Reader Support

### Semantic HTML

**Always use semantic elements:**

```jsx
// ✅ Good: Semantic HTML
<nav>
  <ul>
    <li><a href="/">Home</a></li>
  </ul>
</nav>
<main>
  <article>
    <h1>Title</h1>
    <p>Content</p>
  </article>
</main>

// ❌ Bad: Div soup
<div class="nav">
  <div class="nav-item">
    <div class="link" onclick="navigate()">Home</div>
  </div>
</div>
<div class="main">
  <div class="article">
    <div class="title">Title</div>
    <div class="content">Content</div>
  </div>
</div>
```

### Headings Hierarchy

**Use proper heading levels (h1-h6):**

```jsx
// ✅ Good: Logical hierarchy
<h1>Page Title</h1>
  <h2>Section 1</h2>
    <h3>Subsection 1.1</h3>
  <h2>Section 2</h2>
    <h3>Subsection 2.1</h3>

// ❌ Bad: Skipping levels
<h1>Page Title</h1>
  <h4>Section 1</h4>
```

**One h1 per page** (page title).

### Alt Text

**Every image must have alt text:**

```jsx
// ✅ Good: Descriptive alt text
<img src="logo.png" alt="Acme Corporation logo" />
<img src="chart.png" alt="Bar chart showing 50% increase in sales from 2024 to 2025" />

// ❌ Bad: Generic or missing alt
<img src="logo.png" alt="image" />
<img src="chart.png" />

// ✅ Good: Decorative images (empty alt)
<img src="decorative-line.png" alt="" />
```

**Alt text rules:**
- Describe content/function, not appearance
- Keep under 125 characters
- Don't start with "image of" or "picture of"
- Use empty alt (`alt=""`) for decorative images

---

## Focus Management

### Modal Dialogs

**Trap focus inside modals:**

```jsx
"use client"

import { useEffect, useRef } from "react"
import { Dialog, DialogContent } from "@/components/ui/dialog"

export function AccessibleDialog({ isOpen, onClose, children }) {
  const closeButtonRef = useRef(null)
  
  useEffect(() => {
    if (isOpen && closeButtonRef.current) {
      closeButtonRef.current.focus()
    }
  }, [isOpen])
  
  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent onEscapeKeyDown={onClose}>
        <button 
          ref={closeButtonRef}
          onClick={onClose}
          aria-label="Close dialog"
        >
          ×
        </button>
        {children}
      </DialogContent>
    </Dialog>
  )
}
```

### Dynamic Content

**Announce dynamic changes:**

```jsx
// Live region for status updates
<div role="status" aria-live="polite" aria-atomic="true">
  {statusMessage}
</div>

// Alert for errors
<div role="alert" aria-live="assertive">
  {errorMessage}
</div>
```

**aria-live values:**
- `polite`: Announce when screen reader is idle
- `assertive`: Interrupt immediately
- `off`: Don't announce

---

## ARIA Attributes

### Common ARIA Attributes

**Labels:**
```jsx
// aria-label: Provides label when no visible text
<button aria-label="Close menu">
  <XIcon />
</button>

// aria-labelledby: References visible label
<h2 id="dialog-title">Delete Account</h2>
<div role="dialog" aria-labelledby="dialog-title">
  ...
</div>

// aria-describedby: Additional description
<input 
  id="password" 
  aria-describedby="password-requirements"
/>
<p id="password-requirements">
  Password must be at least 8 characters
</p>
```

**States:**
```jsx
// aria-expanded: Dropdown/accordion state
<button aria-expanded={isOpen}>
  Menu
</button>

// aria-selected: Selected item
<div role="tab" aria-selected={isActive}>
  Tab 1
</div>

// aria-checked: Checkbox/radio state
<div role="checkbox" aria-checked={isChecked}>
  Accept terms
</div>

// aria-disabled: Disabled state
<button aria-disabled="true">
  Submit
</button>
```

**Relationships:**
```jsx
// aria-controls: Element controls another
<button aria-controls="menu-panel" aria-expanded={isOpen}>
  Open Menu
</button>
<div id="menu-panel">
  Menu content
</div>

// aria-owns: Parent-child relationship
<div role="listbox" aria-owns="option1 option2 option3">
  <div id="option1" role="option">Option 1</div>
  <div id="option2" role="option">Option 2</div>
  <div id="option3" role="option">Option 3</div>
</div>
```

### ARIA Roles

**Only use ARIA roles when semantic HTML isn't available:**

```jsx
// ✅ Good: Semantic HTML (no ARIA needed)
<button>Click Me</button>
<nav>...</nav>

// ⚠️ Acceptable: When no semantic option exists
<div role="button" tabIndex={0} onClick={handleClick} onKeyDown={handleKeyDown}>
  Custom Button
</div>

// ❌ Bad: Redundant ARIA
<button role="button">Click Me</button>
```

---

## Form Accessibility

### Labels

**Every input must have a label:**

```jsx
// ✅ Good: Explicit label association
<label htmlFor="email">Email Address</label>
<input id="email" type="email" />

// ✅ Good: Wrapping label
<label>
  Email Address
  <input type="email" />
</label>

// ❌ Bad: Placeholder as label
<input type="email" placeholder="Email Address" />
```

### Error Messages

**Associate errors with inputs:**

```jsx
<div>
  <label htmlFor="email">Email</label>
  <input 
    id="email"
    type="email"
    aria-invalid={hasError}
    aria-describedby={hasError ? "email-error" : undefined}
  />
  {hasError && (
    <p id="email-error" role="alert" className="text-red-600">
      Please enter a valid email address
    </p>
  )}
</div>
```

### Required Fields

**Indicate required fields:**

```jsx
<label htmlFor="name">
  Name <span aria-label="required">*</span>
</label>
<input id="name" required aria-required="true" />
```

---

## Interactive Elements

### Buttons vs Links

**Use buttons for actions, links for navigation:**

```jsx
// ✅ Good: Button for action
<button onClick={handleSubmit}>Submit Form</button>

// ✅ Good: Link for navigation
<a href="/about">About Us</a>

// ❌ Bad: Link styled as button for action
<a href="#" onClick={handleSubmit}>Submit Form</a>

// ❌ Bad: Button for navigation
<button onClick={() => navigate('/about')}>About Us</button>
```

### Custom Interactive Elements

**Make divs keyboard-accessible:**

```jsx
// ❌ Bad: Div as button (not keyboard accessible)
<div onClick={handleClick}>Click Me</div>

// ✅ Good: Proper keyboard support
<div 
  role="button"
  tabIndex={0}
  onClick={handleClick}
  onKeyDown={(e) => {
    if (e.key === 'Enter' || e.key === ' ') {
      e.preventDefault()
      handleClick()
    }
  }}
>
  Click Me
</div>

// ✅ Better: Just use a button
<button onClick={handleClick}>Click Me</button>
```

---

## Testing Tools

### Automated Testing

1. **axe DevTools (Chrome Extension)**
   - https://www.deque.com/axe/devtools/
   - Scans page for WCAG violations
   - Most accurate automated tool

2. **Lighthouse (Chrome DevTools)**
   - Built into Chrome
   - Run: DevTools → Lighthouse → Accessibility
   - Provides score + recommendations

3. **WAVE (WebAIM)**
   - https://wave.webaim.org/
   - Visual feedback overlay
   - Chrome extension available

### Manual Testing

1. **Keyboard Navigation**
   - Unplug mouse, navigate site with keyboard only
   - Tab through all interactive elements
   - Verify logical tab order

2. **Screen Reader**
   - **macOS:** VoiceOver (Cmd+F5)
   - **Windows:** NVDA (free) or JAWS
   - Navigate and ensure content is announced correctly

3. **Zoom Test**
   - Zoom to 200% (Cmd/Ctrl + +)
   - Verify no horizontal scrolling
   - Check text doesn't overflow containers

4. **Color Blindness Simulator**
   - Chrome DevTools → Rendering → Emulate vision deficiencies
   - Test all color combinations

---

## Quick Checklist

**Before Launch:**

- [ ] All images have alt text
- [ ] Contrast ratios meet 4.5:1 (text) and 3:1 (UI)
- [ ] Keyboard navigation works (Tab, Enter, Escape)
- [ ] Focus indicators visible on all interactive elements
- [ ] Forms have proper labels
- [ ] Error messages associated with inputs
- [ ] Headings in logical order (h1-h6)
- [ ] Skip link provided
- [ ] ARIA labels on icon-only buttons
- [ ] Screen reader tested on key flows
- [ ] Lighthouse accessibility score > 95
- [ ] axe DevTools reports 0 violations

**Common Gotchas:**

- Custom dropdowns often lack keyboard support
- Modal focus trapping frequently broken
- Icon buttons missing aria-labels
- Form validation errors not announced
- Loading states not communicated
- Color used as only indicator

---

## Resources

- **WCAG 2.2 Guidelines:** https://www.w3.org/WAI/WCAG22/quickref/
- **WebAIM Checklist:** https://webaim.org/standards/wcag/checklist
- **A11y Project:** https://www.a11yproject.com/
- **Inclusive Components:** https://inclusive-components.design/
- **Deque University:** https://dequeuniversity.com/

---

**Remember:** Accessibility is not optional. It's a legal requirement (ADA, Section 508) and ethical responsibility. 1 in 4 adults in the US has a disability.

**Test early, test often.** Don't wait until launch to check accessibility.
