# 唤醒词功能实现方案

## 方案概述

实现**条件语音回复**：只有当用户消息包含唤醒词时，才生成并播放语音，否则只返回文字。

---

## 三种实现方案

### 方案 1: TTS 标签触发（最简单）⭐推荐

**原理**：使用 OpenClaw 的 `tagged` 模式，只有包含 `[[tts]]` 标签时才生成语音。

**配置**：
```json
{
  "messages": {
    "tts": {
      "auto": "tagged"
    }
  }
}
```

**使用方式**：
- 用户说："用语音回复" → AI 自动添加 `[[tts]]` 标签 → 生成语音
- 用户说："你好" → AI 不加标签 → 只返回文字

**优点**：
- ✅ 配置简单，无需额外脚本
- ✅ OpenClaw 原生支持
- ✅ 准确率高

**缺点**：
- ❌ 需要 AI 理解何时添加标签
- ❌ 依赖模型判断

---

### 方案 2: 文件名标记检测

**原理**：AI 在回复时根据唤醒词在文件名中添加标记，脚本检测标记决定是否播放。

**流程**：
```
用户："用语音告诉我天气"
  ↓
AI 回复：[[tts:voice=weather]] 今天天气晴朗...
  ↓
生成文件：C:\tmp\openclaw\tts-weather-xxx\voice-xxx.mp3
  ↓
脚本检测：文件名包含"weather" → 播放
```

**脚本逻辑**：
```powershell
# 检测文件名中的标记
if ($file.Directory.Name -match "voice|speak|read") {
    # 播放语音
} else {
    # 跳过播放
}
```

**优点**：
- ✅ 灵活可控
- ✅ 可以自定义多种标记

**缺点**：
- ❌ 需要修改 TTS 生成逻辑
- ❌ 实现复杂度中等

---

### 方案 3: 会话历史检测（最智能）⭐⭐⭐

**原理**：脚本监控聊天记录，检测用户消息是否包含唤醒词，然后决定是否播放对应语音。

**流程**：
```
1. 脚本读取最近会话历史
2. 检测用户最后一条消息是否包含唤醒词
3. 如果有唤醒词 + 新语音文件 → 播放
4. 如果没有唤醒词 → 跳过播放
```

**唤醒词示例**：
```powershell
$wakeWords = @(
    "语音", "念出来", "读出来", 
    "播放语音", "用语音", "说出来",
    "voice", "speak", "read it"
)
```

**优点**：
- ✅ 最智能，用户体验最好
- ✅ 无需修改 OpenClaw 核心
- ✅ 可以扩展更多触发条件

**缺点**：
- ❌ 需要访问会话历史
- ❌ 实现复杂度较高

---

## 推荐实现：方案 1 + 方案 3 结合

### 步骤 1: 配置 TTS 为 tagged 模式

```json
{
  "messages": {
    "tts": {
      "auto": "tagged"
    }
  }
}
```

### 步骤 2: 创建唤醒词检测脚本

脚本功能：
1. 监控用户输入（通过日志文件或 API）
2. 检测是否包含唤醒词
3. 如果包含，在 AI 回复后自动触发 TTS
4. 播放生成的语音

### 步骤 3: AI 技能配置

创建技能让 AI 理解何时使用语音：

```markdown
# Voice Response Skill

当用户消息包含以下关键词时，使用语音回复：
- "语音"、"念"、"读"、"说"、"播放"
- "voice"、"speak"、"read"

在回复中添加 [[tts]] 标签触发语音生成。
```

---

## 唤醒词列表（可自定义）

### 中文唤醒词
```
语音
用语音
念出来
读出来
播放语音
说出来
讲出来
念给我听
读给我听
说给我听
```

### 英文唤醒词
```
voice
speak
read it
say it
read aloud
speak out
play voice
```

### 自定义唤醒词
```
爪爪（名字唤醒）
助手
Hey ZhaoZhao
```

---

## 使用示例

### 场景 1: 文字回复（默认）
```
用户：今天天气怎么样？
AI: [文字] 今天杭州晴朗，气温 25 度...
```

### 场景 2: 语音回复（触发唤醒词）
```
用户：用语音告诉我天气
AI: [语音] 今天杭州晴朗，气温 25 度...
```

### 场景 3: 混合模式
```
用户：念一下这篇文章
AI: [语音] 好的，这篇文章的主要内容是...
```

---

## 配置文件示例

### openclaw.json（tagged 模式）
```json
{
  "messages": {
    "tts": {
      "auto": "tagged",
      "provider": "edge",
      "edge": {
        "voice": "zh-CN-XiaoxiaoNeural"
      }
    }
  }
}
```

### wake-words.json（唤醒词配置）
```json
{
  "enabled": true,
  "words": [
    "语音", "念", "读", "说",
    "voice", "speak", "read"
  ],
  "mode": "keyword",  // keyword | exact | regex
  "caseSensitive": false
}
```

---

## 实现代码

### 简化版脚本（方案 1）

```powershell
# 使用 tagged 模式，AI 自动判断是否添加 [[tts]] 标签
# 无需额外脚本，OpenClaw 自动处理

# 只需配置：
# messages.tts.auto = "tagged"
```

### 完整版脚本（方案 3）

见文件：`tts-autoplay-wakeword.ps1`

---

## 测试方法

### 测试 1: 文字回复
```
发送：你好
预期：只看到文字，没有语音播放
```

### 测试 2: 语音回复
```
发送：用语音回复我
预期：听到语音播放
```

### 测试 3: 混合测试
```
发送：今天天气如何？
预期：文字回复

发送：念给我听
预期：语音回复
```

---

## 注意事项

1. **重启 Gateway**：修改配置后需要重启
2. **唤醒词准确性**：避免常用词作为唤醒词
3. **性能影响**：会话检测会略微增加资源占用
4. **隐私考虑**：会话历史检测需要访问聊天记录

---

## 进阶功能

### 1. 多用户识别
不同用户有不同的唤醒词偏好

### 2. 时间段控制
```powershell
# 夜间自动关闭语音
$hour = (Get-Date).Hour
if ($hour -lt 8 -or $hour -ge 23) {
    $ttsEnabled = $false
}
```

### 3. 音量渐变
```powershell
# 早上音量大，晚上音量小
$volume = if ($hour -lt 10) { 0.8 } else { 0.5 }
```

### 4. 语音类型选择
```powershell
# 根据内容选择不同语音
if ($content -match "新闻") {
    $voice = "zh-CN-YunjianNeural"  # 体育解说风格
}
```

---

## 总结

**推荐方案**：使用 `auto: "tagged"` 模式 + AI 技能配置

**理由**：
- ✅ 最简单，无需额外开发
- ✅ OpenClaw 原生支持
- ✅ 用户体验好（自然语言触发）
- ✅ 易于维护和扩展

**实现步骤**：
1. 修改配置为 `tagged` 模式
2. 创建 AI 技能理解唤醒词
3. 测试并调整唤醒词列表
4. 可选：添加高级功能（时间段、多用户等）

---

**作者**: 爪爪 (ZhaoZhao)  
**时间**: 2026-02-27  
**版本**: 1.0.0
