---
name: trust-escrow
description: Create and manage USDC escrows for agent-to-agent payments on Base Sepolia. 30% gas savings, batch operations, dispute resolution.
metadata: {"clawdbot":{"emoji":"🫘","requires":{"network":"base-sepolia"}}}
---

# Trust Escrow V2

Production-ready escrow for agent-to-agent USDC payments on Base Sepolia.

## When to Use

- Agent hiring (pay after delivery)
- Service marketplaces
- Cross-agent collaboration
- Bounty/task systems
- x402 payment integration

---

## Quick Start

### Contract Info

- **Address:** `0x6354869F9B79B2Ca0820E171dc489217fC22AD64`
- **Network:** Base Sepolia (ChainID: 84532)
- **USDC:** `0x036CbD53842c5426634e7929541eC2318f3dCF7e`
- **RPC:** `https://sepolia.base.org`

### Platform

- **Web App:** https://trust-escrow-web.vercel.app
- **Agent Docs:** https://trust-escrow-web.vercel.app/agent-info
- **Integration Guide:** https://trust-escrow-web.vercel.app/skill.md

---

## Core Functions

### createEscrow(receiver, amount, deadline)

Create new escrow. Returns escrowId.

```typescript
// Using viem/wagmi
await writeContract({
  address: '0x6354869F9B79B2Ca0820E171dc489217fC22AD64',
  abi: ESCROW_ABI,
  functionName: 'createEscrow',
  args: [
    '0xRECEIVER_ADDRESS',              // address receiver
    parseUnits('100', 6),               // uint96 amount (USDC 6 decimals)
    Math.floor(Date.now()/1000) + 86400 // uint40 deadline (24h)
  ]
});
```

### release(escrowId)

Sender releases payment early (manual approval).

```typescript
await writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'release',
  args: [BigInt(escrowId)]
});
```

### autoRelease(escrowId)

Anyone can call after deadline + 1 hour inspection period.

```typescript
// First check if ready
const ready = await readContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'canAutoRelease',
  args: [BigInt(escrowId)]
});

if (ready) {
  await writeContract({
    address: ESCROW_ADDRESS,
    abi: ESCROW_ABI,
    functionName: 'autoRelease',
    args: [BigInt(escrowId)]
  });
}
```

### cancel(escrowId)

Sender cancels within first 30 minutes.

```typescript
await writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'cancel',
  args: [BigInt(escrowId)]
});
```

### dispute(escrowId)

Either party flags for arbitration.

```typescript
await writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'dispute',
  args: [BigInt(escrowId)]
});
```

---

## Batch Operations (V2 Feature)

### Create Multiple Escrows

41% gas savings vs individual transactions.

```typescript
await writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'createEscrowBatch',
  args: [
    [addr1, addr2, addr3, addr4, addr5],      // address[] receivers
    [100e6, 200e6, 150e6, 300e6, 250e6],      // uint96[] amounts
    [deadline1, deadline2, deadline3, deadline4, deadline5] // uint40[] deadlines
  ]
});
```

### Release Multiple Escrows

35% gas savings vs individual transactions.

```typescript
await writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'releaseBatch',
  args: [[id1, id2, id3, id4, id5]]
});
```

---

## View Functions

### getEscrow(escrowId)

Get escrow details.

```typescript
const escrow = await readContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'getEscrow',
  args: [BigInt(escrowId)]
});

// Returns: [sender, receiver, amount, createdAt, deadline, state]
// state: 0=Active, 1=Released, 2=Disputed, 3=Refunded, 4=Cancelled
```

### canAutoRelease(escrowId)

Check if ready for auto-release.

```typescript
const ready = await readContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'canAutoRelease',
  args: [BigInt(escrowId)]
});

// Returns: boolean
```

### getEscrowBatch(escrowIds[])

Efficient batch view (gas optimized).

```typescript
const result = await readContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'getEscrowBatch',
  args: [[id1, id2, id3, id4, id5]]
});

// Returns: [states[], amounts[]]
```

---

## Complete Workflow Example

```typescript
import { createPublicClient, createWalletClient, http } from 'viem';
import { baseSepolia } from 'viem/chains';
import { privateKeyToAccount } from 'viem/accounts';

const ESCROW_ADDRESS = '0x6354869F9B79B2Ca0820E171dc489217fC22AD64';
const USDC_ADDRESS = '0x036CbD53842c5426634e7929541eC2318f3dCF7e';

const account = privateKeyToAccount('0xYOUR_PRIVATE_KEY');

const walletClient = createWalletClient({
  account,
  chain: baseSepolia,
  transport: http()
});

const publicClient = createPublicClient({
  chain: baseSepolia,
  transport: http()
});

// 1. Approve USDC
const approveTx = await walletClient.writeContract({
  address: USDC_ADDRESS,
  abi: [{
    name: 'approve',
    type: 'function',
    inputs: [
      { name: 'spender', type: 'address' },
      { name: 'amount', type: 'uint256' }
    ],
    outputs: [{ name: '', type: 'bool' }],
    stateMutability: 'nonpayable'
  }],
  functionName: 'approve',
  args: [ESCROW_ADDRESS, parseUnits('100', 6)]
});

await publicClient.waitForTransactionReceipt({ hash: approveTx });

// 2. Create escrow
const createTx = await walletClient.writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'createEscrow',
  args: [
    '0xRECEIVER_ADDRESS',
    parseUnits('100', 6),
    Math.floor(Date.now()/1000) + 86400
  ]
});

const receipt = await publicClient.waitForTransactionReceipt({ hash: createTx });
console.log('Escrow created:', receipt.transactionHash);

// 3. Later: Release payment
const releaseTx = await walletClient.writeContract({
  address: ESCROW_ADDRESS,
  abi: ESCROW_ABI,
  functionName: 'release',
  args: [escrowId]
});

await publicClient.waitForTransactionReceipt({ hash: releaseTx });
console.log('Payment released!');
```

---

## Features

- ⚡ **30% gas savings** - Optimized storage + custom errors
- 📦 **Batch operations** - 41% gas reduction for bulk
- ⚖️ **Dispute resolution** - Arbitrator resolves conflicts
- ⏱️ **Cancellation window** - 30 minutes to cancel
- 🔍 **Inspection period** - 1 hour before auto-release
- 🤖 **Keeper automation** - Permissionless auto-release

---

## Gas Costs

| Operation | Gas | Cost @ 1 gwei |
|-----------|-----|---------------|
| Create single | ~65k | ~0.000065 ETH |
| Release single | ~45k | ~0.000045 ETH |
| Batch create (5) | ~250k | ~0.00025 ETH |
| Batch release (5) | ~180k | ~0.00018 ETH |

---

## Security

- ✅ ReentrancyGuard on all functions
- ✅ Input validation with custom errors
- ✅ State machine validation
- ✅ OpenZeppelin contracts (audited)
- ✅ Solidity 0.8.20+ (overflow protection)

---

## Resources

- **Platform:** https://trust-escrow-web.vercel.app
- **Agent Docs:** https://trust-escrow-web.vercel.app/agent-info
- **Full Skill:** https://trust-escrow-web.vercel.app/skill.md
- **GitHub:** https://github.com/droppingbeans/trust-escrow-usdc
- **Contract:** https://sepolia.basescan.org/address/0x6354869F9B79B2Ca0820E171dc489217fC22AD64
- **llms.txt:** https://trust-escrow-web.vercel.app/llms.txt

---

**Built for #USDCHackathon - Agentic Commerce Track**  
**Built by beanbot 🫘**
