---
name: truenas-skill
description: >
  Manage TrueNAS SCALE via API. Check pool health, manage datasets and snapshots,
  monitor alerts, control services, manage apps, orchestrate Dockge container stacks,
  and manage bookmarks. Use when the user asks about their NAS, storage, backups,
  containers, bookmarks, or homelab services.
license: MIT
homepage: https://github.com/anotb/truenas-skill
compatibility: Requires curl, jq, and Node.js 18+. Network access to TrueNAS instance.
metadata: {"author": "anotb", "version": "1.2.0", "openclaw": {"requires": {"env": ["TRUENAS_URL", "TRUENAS_API_KEY"], "bins": ["curl", "jq", "node"]}, "primaryEnv": "TRUENAS_API_KEY"}}
---

# TrueNAS SCALE Skill

Manage a TrueNAS SCALE server and its apps via the TrueNAS API and Dockge Socket.IO.

## Setup

### Required Environment Variables

```
TRUENAS_URL       — TrueNAS base URL (e.g., https://10.0.0.5:444)
TRUENAS_API_KEY   — API key from TrueNAS UI → API Keys
```

### Optional: TLS Configuration

```
TRUENAS_VERIFY_TLS  — Set to "1" to enforce TLS certificate validation (default: skip for self-signed certs)
```

### Optional: Dockge (Docker Compose UI)

```
DOCKGE_URL        — Dockge URL (e.g., http://10.0.0.5:5001)
DOCKGE_USER       — Dockge login username
DOCKGE_PASS       — Dockge login password
```

### Optional: Homelab Service API Keys

See the `references/` directory for per-service env vars. Common ones:

```
SONARR_URL, SONARR_API_KEY           — TV show management
RADARR_URL, RADARR_API_KEY           — Movie management
PROWLARR_URL, PROWLARR_API_KEY       — Indexer management
OVERSEERR_URL, OVERSEERR_API_KEY     — Media request UI
PLEX_URL                             — Media server (no auth on LAN)
TAUTULLI_URL, TAUTULLI_API_KEY       — Plex analytics
QBITTORRENT_URL                      — Torrent client (no auth)
SABNZBD_URL, SABNZBD_API_KEY         — Usenet client
AUDIOBOOKSHELF_URL, AUDIOBOOKSHELF_API_KEY
NTFY_URL                             — Push notifications
SYNCTHING_URL, SYNCTHING_API_KEY     — File sync
N8N_URL, N8N_API_KEY                 — Workflow automation
NOCODB_URL, NOCODB_API_KEY           — Database
CHANGEDETECTION_URL, CHANGEDETECTION_API_KEY
CRAFTY_URL, CRAFTY_API_KEY           — Game servers
LAZYLIBRARIAN_URL, LAZYLIBRARIAN_API_KEY
METUBE_URL                           — YouTube downloader
KARAKEEP_URL, KARAKEEP_API_KEY       — Bookmarks with AI tagging
```

## API Notes

**HTTPS REQUIRED:** TrueNAS auto-revokes API keys used over HTTP.

> **REST API Deprecation Notice:** The REST API (`/api/v2.0/`) is deprecated in TrueNAS 25.04
> and **fully removed in 26.04**. Use the WebSocket API (via `scripts/truenas-ws.mjs`) as
> the forward-compatible method. REST examples below still work on 24.10 and 25.x.

### REST API (Legacy)

```bash
curl -sk "$TRUENAS_URL/api/v2.0/[endpoint]" \
  -H "Authorization: Bearer $TRUENAS_API_KEY"
```

The `-k` flag is needed for self-signed certificates (common on home servers).

### WebSocket API (Recommended)

The WebSocket API uses a DDP-like protocol (Meteor style). REST paths become dot notation:
`/api/v2.0/app` → `app.query`, `/api/v2.0/system/info` → `system.info`.

```javascript
// Connect: wss://<host>/websocket (rejectUnauthorized: false for self-signed)

// 1. Handshake
send: {"msg": "connect", "version": "1", "support": ["1"]}
recv: {"msg": "connected", "session": "..."}

// 2. Authenticate
send: {"id": "1", "msg": "method", "method": "auth.login_with_api_key", "params": ["API_KEY"]}
recv: {"id": "1", "msg": "result", "result": true}

// 3. Call methods
send: {"id": "2", "msg": "method", "method": "system.info", "params": []}
send: {"id": "3", "msg": "method", "method": "app.query", "params": []}
```

Use the helper script for WebSocket calls: `node scripts/truenas-ws.mjs <method> [params_json]`

## Security Notes

- **Self-signed certificates:** TLS verification is skipped by default (`curl -k`, `rejectUnauthorized: false`) because homelab servers typically use self-signed certs. Set `TRUENAS_VERIFY_TLS=1` to enforce strict TLS validation.
- **API key scope:** Use a read-only or least-privilege API key when possible. TrueNAS lets you scope keys to specific endpoints.
- **Credentials stay local:** All env vars are read at runtime and sent only to the configured service endpoints. Nothing is phoned home.

## Core Operations

### System Info

```bash
curl -sk "$TRUENAS_URL/api/v2.0/system/info" -H "Authorization: Bearer $TRUENAS_API_KEY"
```

### Pool Health

```bash
# All pools with health status
curl -sk "$TRUENAS_URL/api/v2.0/pool" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, healthy}'

# Or via WebSocket
node scripts/truenas-ws.mjs pool.query '[]'
```

The API returns a `.healthy` boolean per pool. For deeper status, inspect the full pool object.

### Active Alerts

```bash
curl -sk "$TRUENAS_URL/api/v2.0/alert/list" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {level, formatted}'
```

### Running Services

```bash
curl -sk "$TRUENAS_URL/api/v2.0/service" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | select(.state == "RUNNING") | .service'
```

## Dataset Management

### List Datasets

```bash
curl -sk "$TRUENAS_URL/api/v2.0/pool/dataset" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, type, used: .used.parsed, available: .available.parsed}'
```

### Create Dataset

```bash
curl -sk -X POST "$TRUENAS_URL/api/v2.0/pool/dataset" \
  -H "Authorization: Bearer $TRUENAS_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"name": "pool/path/new-dataset"}'
```

### Delete Dataset

```bash
# Destructive — confirm with user first
curl -sk -X DELETE "$TRUENAS_URL/api/v2.0/pool/dataset/id/DATASET_ID" \
  -H "Authorization: Bearer $TRUENAS_API_KEY"
```

## Snapshots & Replication

### List Snapshots

```bash
# WebSocket (required on 25.10+, /api/v2.0/zfs/snapshot returns 404)
node scripts/truenas-ws.mjs zfs.snapshot.query '[]'
```

### Create Snapshot

```bash
node scripts/truenas-ws.mjs zfs.snapshot.create '[{"dataset": "pool/dataset", "name": "manual-YYYY-MM-DD"}]'
```

### Snapshot Task Status

```bash
curl -sk "$TRUENAS_URL/api/v2.0/pool/snapshottask" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {dataset, schedule, enabled}'
```

### Replication Health

```bash
curl -sk "$TRUENAS_URL/api/v2.0/replication" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, state: .state.state}'
```

## App Management

TrueNAS Apps are the official marketplace for installing containerized services.

### List Installed Apps

```bash
curl -sk "$TRUENAS_URL/api/v2.0/app" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, state, version}'
```

### Check for Updates

```bash
curl -sk "$TRUENAS_URL/api/v2.0/app" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | select(.upgrade_available) | .name'
```

### Install / Update Apps

See `references/app-installation.md` for the full installation guide covering:
- Checking app templates and storage requirements
- Creating datasets with proper ACLs
- Installing with correct storage mappings
- Handling apps with multiple storage mounts

### App Status

```bash
curl -sk "$TRUENAS_URL/api/v2.0/app?name=APP_NAME" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[0] | {name, state, portals}'
```

## Dockge (Docker Compose Stacks)

Dockge is a companion UI for Docker Compose stacks not in the TrueNAS Apps catalog.
It uses Socket.IO, not REST. Use the provided scripts.

### Prerequisites

```bash
npm install   # in this skill's root directory
```

### List Stacks

```bash
node scripts/dockge-list.mjs
```

### Update Stacks

```bash
# Update all running stacks
node scripts/dockge-update.mjs

# Update specific stacks
node scripts/dockge-update.mjs mystack1 mystack2
```

### Socket.IO Protocol Details

Dockge uses Socket.IO with WebSocket transport.

**Status codes:**
- 1 = inactive/exited
- 3 = running
- 4 = updating

**Key events:**
- `login` — authenticate with username/password
- `stackList` — get all stacks (received via `agent` event)
- `agent`, "", "updateStack", stackName — trigger pull + restart

**Note:** Stacks prefixed with `ix-` are TrueNAS-managed apps visible to Dockge — skip those when updating.

## Monitoring Checklist

Run these commands for a quick health overview:

```bash
# Pool health
curl -sk "$TRUENAS_URL/api/v2.0/pool" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, healthy}'

# Active alerts
curl -sk "$TRUENAS_URL/api/v2.0/alert/list" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {level, formatted}'

# Running services
curl -sk "$TRUENAS_URL/api/v2.0/service" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | select(.state == "RUNNING") | .service'

# App updates available
curl -sk "$TRUENAS_URL/api/v2.0/app" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | select(.upgrade_available) | .name'

# Replication status
curl -sk "$TRUENAS_URL/api/v2.0/replication" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | {name, state: .state.state}'
```

## Homelab Services

This skill includes reference files for common homelab service categories. Each covers
API patterns, env vars, and common agent tasks for services that typically run alongside
TrueNAS:

| Reference | Services | File |
|-----------|----------|------|
| Media management | Overseerr, Sonarr, Radarr, Prowlarr, Plex, Tautulli | `references/media-management.md` |
| App installation | TrueNAS native app install guide | `references/app-installation.md` |
| Download clients | qBittorrent, SABnzbd, FlareSolverr | `references/downloads.md` |
| Homelab services | ntfy, Syncthing, n8n, NocoDB, ChangeDetection, Crafty | `references/homelab-services.md` |
| Books & media | Audiobookshelf, LazyLibrarian, Calibre-Web, MeTube | `references/books-and-media.md` |
| Bookmarks | Karakeep (AI-powered bookmark manager) | `references/bookmarks.md` |

Load the relevant reference file when the user asks about a specific service category.

## Common Tasks

### "Check NAS health"

Run the monitoring checklist above. Summarize pool states, alerts, and any pending updates.

### "What's running?"

```bash
# TrueNAS apps
curl -sk "$TRUENAS_URL/api/v2.0/app" -H "Authorization: Bearer $TRUENAS_API_KEY" \
  | jq '.[] | select(.state == "RUNNING") | .name'

# Dockge stacks (if configured)
node scripts/dockge-list.mjs
```

### "Install an app"

Follow the guide in `references/app-installation.md`:
1. Check app template for storage requirements
2. Create dataset(s) under the apps pool
3. Set ACL with apps preset
4. Install app with correct storage mappings

### "Take a snapshot"

```bash
node scripts/truenas-ws.mjs zfs.snapshot.create '[{"dataset": "pool/dataset", "name": "manual-snapshot-name"}]'
```

### "What's downloading?"

See `references/downloads.md` for qBittorrent and SABnzbd API commands.

### "Add a movie/show"

See `references/media-management.md` for Overseerr request workflow.
