/**
 * TreeListy Pattern Definitions
 * Extracted from treeplexity.html - the 21 hierarchical patterns
 *
 * Copyright (c) 2024-2026 Prairie2Cloud LLC
 * Licensed under Apache-2.0
 */

const PATTERNS = {
  generic: {
    name: 'Generic Project',
    icon: '📋',
    description: 'Universal structure for any project',
    levels: ['Project', 'Phase', 'Item', 'Task'],
    phaseSubtitles: ['Pre-Seed', 'Seed', 'Build'],
    types: ['Land', 'Engineering', 'Equipment', 'Infrastructure', 'Corporate', 'Professional', 'Contingency'],
    fields: {
      cost: { type: 'number', label: 'Budget', description: 'Budget allocated for this item' },
      alternateSource: { type: 'text', label: 'Alternate Source', description: 'Backup vendor or alternative solution' },
      leadTime: { type: 'text', label: 'Lead Time', description: 'Procurement/delivery timeline' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  sales: {
    name: 'Sales Pipeline',
    icon: '💼',
    description: 'Track sales opportunities through quarters',
    levels: ['Pipeline', 'Quarter', 'Deal', 'Action'],
    phaseSubtitles: ['Q1', 'Q2', 'Q3', 'Q4'],
    types: ['Inbound Lead', 'Outbound Prospect', 'Partnership', 'Account Expansion', 'Renewal', 'Upsell', 'Cross-sell', 'Enterprise Deal'],
    fields: {
      dealValue: { type: 'number', label: 'Deal Value', description: 'Potential revenue' },
      expectedCloseDate: { type: 'date', label: 'Expected Close', description: 'Target close date' },
      leadSource: { type: 'text', label: 'Lead Source', description: 'Lead origin' },
      contactPerson: { type: 'text', label: 'Contact', description: 'Main decision maker' },
      stageProbability: { type: 'number', label: 'Probability %', description: 'Likelihood of closing (0-100)', min: 0, max: 100 },
      competitorInfo: { type: 'textarea', label: 'Competitors', description: 'Other vendors' }
    },
    includeTracking: true,
    includeDependencies: false
  },

  thesis: {
    name: 'Academic Writing',
    icon: '🎓',
    description: 'Structure academic papers and dissertations',
    levels: ['Thesis', 'Chapter', 'Section', 'Point'],
    phaseSubtitles: ['Introduction', 'Body', 'Conclusion'],
    types: ['Literature Review', 'Methodology', 'Analysis', 'Discussion', 'Theory', 'Evidence', 'Argument', 'Conclusion'],
    fields: {
      wordCount: { type: 'number', label: 'Word Count', description: 'Current word count' },
      targetWordCount: { type: 'number', label: 'Target Words', description: 'Goal word count' },
      draftStatus: { type: 'select', label: 'Draft Status', options: ['Outline', 'First Draft', 'Revision', 'Final'] },
      citations: { type: 'textarea', label: 'Citations', description: 'Key sources' },
      keyArgument: { type: 'textarea', label: 'Key Argument', description: 'Central claim' },
      evidenceType: { type: 'select', label: 'Evidence Type', options: ['Empirical', 'Theoretical', 'Mixed', 'N/A'] }
    },
    includeTracking: false,
    includeDependencies: true
  },

  roadmap: {
    name: 'Product Roadmap',
    icon: '🚀',
    description: 'Plan product features across quarters',
    levels: ['Product', 'Quarter', 'Feature', 'Story'],
    phaseSubtitles: ['Q1', 'Q2', 'Q3', 'Q4'],
    types: ['Core Feature', 'Enhancement', 'Bug Fix', 'Technical Debt', 'Research/Spike', 'Platform', 'Integration', 'UX Improvement'],
    fields: {
      storyPoints: { type: 'number', label: 'Story Points', description: 'Effort estimate (Fibonacci)' },
      engineeringEstimate: { type: 'text', label: 'Estimate', description: 'Time estimate' },
      userImpact: { type: 'select', label: 'User Impact', options: ['High', 'Medium', 'Low'] },
      technicalRisk: { type: 'select', label: 'Technical Risk', options: ['Low', 'Medium', 'High', 'Unknown'] },
      featureFlag: { type: 'text', label: 'Feature Flag', description: 'Flag for gradual rollout' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  book: {
    name: 'Book Writing',
    icon: '📚',
    description: 'Organize books into parts, chapters, and scenes',
    levels: ['Book', 'Part', 'Chapter', 'Scene'],
    phaseSubtitles: ['Act I', 'Act II', 'Act III'],
    types: ['Narrative', 'Dialogue', 'Description', 'Action', 'Reflection', 'Transition', 'Climax', 'Exposition'],
    fields: {
      wordCount: { type: 'number', label: 'Word Count', description: 'Current word count' },
      targetWordCount: { type: 'number', label: 'Target Words', description: 'Goal word count' },
      draftStatus: { type: 'select', label: 'Draft Status', options: ['Outline', 'First Draft', 'Revision', 'Final'] },
      povCharacter: { type: 'text', label: 'POV Character', description: 'Point-of-view character' },
      sceneSetting: { type: 'textarea', label: 'Setting', description: 'Location, time, mood' },
      plotFunction: { type: 'select', label: 'Plot Function', options: ['Setup', 'Conflict', 'Resolution', 'Transition'] }
    },
    includeTracking: false,
    includeDependencies: true
  },

  event: {
    name: 'Event Planning',
    icon: '🎉',
    description: 'Plan events from prep to execution to follow-up',
    levels: ['Event', 'Stage', 'Activity', 'Task'],
    phaseSubtitles: ['Pre-Event', 'Event Day', 'Post-Event'],
    types: ['Logistics', 'Catering', 'Entertainment', 'Venue', 'Marketing', 'Registration', 'Follow-up', 'AV/Tech'],
    fields: {
      budget: { type: 'number', label: 'Budget', description: 'Budget for activity' },
      vendor: { type: 'text', label: 'Vendor', description: 'External vendor' },
      bookingDeadline: { type: 'date', label: 'Booking Deadline', description: 'Last date to book' },
      guestCount: { type: 'number', label: 'Guest Count', description: 'Expected attendees' },
      location: { type: 'text', label: 'Location', description: 'Venue location' },
      responsiblePerson: { type: 'text', label: 'Responsible', description: 'Team member' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  fitness: {
    name: 'Fitness Program',
    icon: '💪',
    description: 'Structure training programs with periodization',
    levels: ['Program', 'Phase', 'Workout', 'Exercise'],
    phaseSubtitles: ['Foundation', 'Build', 'Peak'],
    types: ['Strength Training', 'Cardio', 'Flexibility', 'Recovery', 'Nutrition', 'Assessment', 'Conditioning', 'Mobility'],
    fields: {
      sets: { type: 'number', label: 'Sets', description: 'Number of sets' },
      reps: { type: 'text', label: 'Reps', description: 'Repetitions per set' },
      duration: { type: 'text', label: 'Duration', description: 'Time for exercise' },
      intensity: { type: 'select', label: 'Intensity', options: ['Light', 'Moderate', 'High', 'Max'] },
      equipment: { type: 'text', label: 'Equipment', description: 'Required equipment' },
      formCues: { type: 'textarea', label: 'Form Cues', description: 'Technique reminders' },
      restPeriod: { type: 'text', label: 'Rest Period', description: 'Rest between sets' }
    },
    includeTracking: false,
    includeDependencies: true
  },

  strategy: {
    name: 'Strategic Plan',
    icon: '📊',
    description: 'Organize business strategy into pillars and initiatives',
    levels: ['Strategy', 'Pillar', 'Initiative', 'Action'],
    phaseSubtitles: ['Planning', 'Execution', 'Review'],
    types: ['Market Expansion', 'Operational Excellence', 'Financial', 'HR', 'Technology', 'Risk Management', 'Innovation', 'Customer Experience'],
    fields: {
      investment: { type: 'number', label: 'Investment', description: 'Capital investment' },
      keyMetric: { type: 'text', label: 'Key Metric', description: 'Success measurement' },
      targetValue: { type: 'text', label: 'Target Value', description: 'Goal to achieve' },
      responsibleExecutive: { type: 'text', label: 'Executive', description: 'Executive sponsor' },
      strategicTheme: { type: 'select', label: 'Theme', options: ['Growth', 'Efficiency', 'Innovation', 'Transformation', 'Risk Mitigation'] },
      riskLevel: { type: 'select', label: 'Risk Level', options: ['Low', 'Medium', 'High'] }
    },
    includeTracking: true,
    includeDependencies: true
  },

  course: {
    name: 'Course Design',
    icon: '📖',
    description: 'Build educational curricula with units and lessons',
    levels: ['Course', 'Unit', 'Lesson', 'Exercise'],
    phaseSubtitles: ['Beginning', 'Middle', 'Advanced'],
    types: ['Lecture', 'Lab/Practical', 'Discussion', 'Assessment', 'Reading', 'Project', 'Workshop', 'Field Work'],
    fields: {
      learningObjectives: { type: 'textarea', label: 'Learning Objectives', description: 'Expected outcomes' },
      duration: { type: 'text', label: 'Duration', description: 'Class time needed' },
      difficultyLevel: { type: 'select', label: 'Difficulty', options: ['Beginner', 'Intermediate', 'Advanced'] },
      prerequisites: { type: 'textarea', label: 'Prerequisites', description: 'Prior knowledge' },
      assessmentType: { type: 'select', label: 'Assessment', options: ['Quiz', 'Assignment', 'Project', 'Discussion', 'Exam', 'None'] },
      resourcesNeeded: { type: 'textarea', label: 'Resources', description: 'Required materials' },
      homework: { type: 'textarea', label: 'Homework', description: 'Out-of-class work' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  film: {
    name: 'AI Video Production',
    icon: '🎬',
    description: 'Create films using AI video generation (Sora, Veo, Runway, Pika)',
    levels: ['Film', 'Act', 'Scene', 'Shot'],
    phaseSubtitles: ['Act I - Setup', 'Act II - Conflict', 'Act III - Resolution'],
    types: ['Establishing Shot', 'Character Introduction', 'Dialogue Scene', 'Action Sequence', 'Montage', 'Transition', 'Climax', 'Resolution'],
    fields: {
      aiPlatform: { type: 'select', label: 'AI Platform', options: ['Sora', 'Veo 3', 'Runway Gen-3', 'Pika 2.0', 'Kling AI', 'Luma', 'Haiper', 'Testing Multiple'] },
      videoPrompt: { type: 'textarea', label: 'Video Prompt', description: 'Text-to-video prompt' },
      visualStyle: { type: 'select', label: 'Visual Style', options: ['Photorealistic', 'Cinematic', 'Documentary', 'Anime', 'Pixar 3D', 'Noir', 'Vintage Film'] },
      duration: { type: 'select', label: 'Duration', options: ['2s', '4s', '6s', '10s', '20s', 'Extended'] },
      aspectRatio: { type: 'select', label: 'Aspect Ratio', options: ['16:9', '9:16', '1:1', '2.39:1', '4:3'] },
      cameraMovement: { type: 'select', label: 'Camera Movement', options: ['Static', 'Pan', 'Dolly', 'Tracking', 'Crane', 'Handheld', 'Orbiting'] },
      motionIntensity: { type: 'select', label: 'Motion', options: ['Minimal', 'Subtle', 'Moderate', 'Dynamic', 'Intense'] },
      lightingMood: { type: 'select', label: 'Lighting', options: ['Golden Hour', 'Overcast', 'Night', 'Neon', 'Dramatic', 'Studio', 'Natural'] },
      iterationNotes: { type: 'textarea', label: 'Notes', description: 'Generation insights' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  veo3: {
    name: 'Veo3 (Google)',
    icon: '🎥',
    description: 'Google Veo 3.1 AI video generation with Flow workflow',
    levels: ['Project', 'Sequence', 'Scene', 'Shot'],
    phaseSubtitles: ['Opening Sequence', 'Development', 'Climax Sequence'],
    types: ['Ingredients to Video', 'Frames to Video', 'Extend Video', 'Standard Generation'],
    fields: {
      flowMode: { type: 'select', label: 'Flow Mode', options: ['Ingredients', 'Frames', 'Extend', 'Standard'] },
      videoPrompt: { type: 'textarea', label: 'Veo Prompt', description: 'Veo 3 prompt' },
      ingredientImages: { type: 'textarea', label: 'Ingredient Images', description: 'Reference image URLs' },
      startFrame: { type: 'text', label: 'Start Frame', description: 'Starting frame' },
      endFrame: { type: 'text', label: 'End Frame', description: 'Ending frame' },
      extendDuration: { type: 'select', label: 'Extend Duration', options: ['10s', '20s', '30s', '60+ seconds'] },
      duration: { type: 'select', label: 'Duration', options: ['4s', '6s', '8s @ 24 FPS'] },
      resolution: { type: 'select', label: 'Resolution', options: ['720p', '1080p'] },
      audioType: { type: 'select', label: 'Audio', options: ['Dialogue', 'Sound Effects', 'Ambience', 'Mixed', 'Silent'] },
      cinematicStyle: { type: 'select', label: 'Style', options: ['Realistic', 'Dramatic', 'Documentary', 'Commercial', 'Artistic', 'Music Video'] }
    },
    includeTracking: true,
    includeDependencies: true
  },

  sora2: {
    name: 'Sora2 (OpenAI)',
    icon: '🎬',
    description: 'OpenAI Sora 2 AI video generation with cameo and remix',
    levels: ['Project', 'Sequence', 'Beat', 'Shot'],
    phaseSubtitles: ['Setup', 'Conflict', 'Resolution'],
    types: ['Cameo Shot', 'Remix Variant', 'Standard Generation', 'Complex Physics'],
    fields: {
      videoPrompt: { type: 'textarea', label: 'Sora Prompt', description: 'Sora 2 prompt' },
      beatType: { type: 'select', label: 'Beat Type', options: ['Setup', 'Conflict', 'Transition', 'Resolution', 'Character Moment'] },
      cameoUsed: { type: 'select', label: 'Cameo', options: ['None', 'Cameo ID 1', 'Cameo ID 2', 'Cameo ID 3'] },
      remixSource: { type: 'text', label: 'Remix Source', description: 'Source shot ID' },
      physicsComplexity: { type: 'select', label: 'Physics', options: ['Simple', 'Medium', 'Complex'] },
      duration: { type: 'select', label: 'Duration', options: ['4s', '8s', '12s'] },
      resolution: { type: 'select', label: 'Resolution', options: ['720p', '1080p'] },
      nleExportFormat: { type: 'select', label: 'NLE Export', options: ['Premiere', 'Final Cut', 'DaVinci', 'After Effects', 'Raw'] },
      audioSync: { type: 'select', label: 'Audio Sync', options: ['Synchronized Dialogue', 'Sound Effects', 'Background Score', 'Silent'] }
    },
    includeTracking: true,
    includeDependencies: true
  },

  philosophy: {
    name: 'Philosophy',
    icon: '🤔',
    description: 'Structure philosophical dialogues, treatises, and arguments',
    levels: ['Dialogue', 'Movement', 'Claim', 'Support'],
    phaseSubtitles: ['Opening Question', 'First Definition', 'Refutation', 'Second Attempt', 'Deeper Inquiry', 'Resolution'],
    types: ['Question', 'Definition', 'Refutation/Elenchus', 'Premise', 'Conclusion', 'Objection', 'Response', 'Example', 'Analogy', 'Distinction', 'Paradox', 'Thought Experiment', 'Aporia'],
    fields: {
      speaker: { type: 'text', label: 'Speaker', description: 'Who makes this claim' },
      argumentType: { type: 'select', label: 'Argument Type', options: ['Deductive', 'Inductive', 'Abductive', 'Dialectical', 'Reductio', 'Socratic Elenchus'] },
      validity: { type: 'select', label: 'Validity', options: ['Valid', 'Invalid', 'Sound', 'Unsound', 'Uncertain'] },
      keyTerms: { type: 'text', label: 'Key Terms', description: 'Central concepts' },
      premise1: { type: 'textarea', label: 'Premise 1', description: 'First premise' },
      premise2: { type: 'textarea', label: 'Premise 2', description: 'Second premise' },
      conclusion: { type: 'textarea', label: 'Conclusion', description: 'Logical conclusion' },
      objection: { type: 'textarea', label: 'Objection', description: 'Main counterargument' },
      response: { type: 'textarea', label: 'Response', description: 'Defense/reply' },
      textualReference: { type: 'text', label: 'Reference', description: 'Stephanus number or citation' },
      philosophicalSchool: { type: 'select', label: 'School', options: ['Pre-Socratic', 'Platonic', 'Aristotelian', 'Stoic', 'Epicurean', 'Skeptic', 'Neoplatonic', 'Medieval', 'Modern', 'Contemporary'] }
    },
    includeTracking: false,
    includeDependencies: true,
    dialecticMode: true
  },

  prompting: {
    name: 'Prompt Engineering',
    icon: '🧠',
    description: 'Design and test AI prompts with best practices',
    levels: ['Prompt Library', 'Category', 'Prompt', 'Test Case'],
    phaseSubtitles: ['Customer Support', 'Content Generation', 'Data Analysis', 'Code Assistance', 'Research', 'Creative Writing'],
    types: ['Task Instruction', 'Few-Shot Examples', 'Chain-of-Thought', 'Structured Output', 'XML-Guided', 'Prefill-Guided', 'Production-Ready', 'Experimental'],
    fields: {
      systemPrompt: { type: 'textarea', label: 'System Prompt', description: 'AI role and behavior', required: true },
      userPromptTemplate: { type: 'textarea', label: 'User Prompt', description: 'Main instruction', required: true },
      fewShotExamples: { type: 'textarea', label: 'Few-Shot Examples', description: '2-3 examples' },
      outputFormat: { type: 'textarea', label: 'Output Format', description: 'Expected structure' },
      chainOfThought: { type: 'textarea', label: 'Chain of Thought', description: 'Step-by-step reasoning' },
      modelTarget: { type: 'select', label: 'Model', options: ['Claude 3.5 Sonnet', 'Claude 3 Opus', 'GPT-4o', 'GPT-4 Turbo', 'Gemini Pro', 'Other'] },
      temperature: { type: 'number', label: 'Temperature', description: 'Creativity level (0-1)', min: 0, max: 1, step: 0.1 },
      testResults: { type: 'textarea', label: 'Test Results', description: 'Performance metrics' },
      testStatus: { type: 'select', label: 'Status', options: ['Draft', 'Testing', 'Validated', 'Production', 'Deprecated'] }
    },
    includeTracking: true,
    includeDependencies: true
  },

  familytree: {
    name: 'Family Tree',
    icon: '👨‍👩‍👧‍👦',
    description: 'Build and document your family genealogy',
    levels: ['Family', 'Generation', 'Person', 'Event'],
    phaseSubtitles: ['Self/Siblings', 'Parents', 'Grandparents', 'Great-Grandparents', 'Great-Great-Grandparents'],
    types: ['Paternal Line', 'Maternal Line', 'Spouse', 'Biological', 'Adopted', 'Step-Family', 'Foster', 'Half-Sibling'],
    fields: {
      fullName: { type: 'text', label: 'Full Name', description: 'Complete name' },
      maidenName: { type: 'text', label: 'Maiden Name', description: 'Birth surname' },
      gender: { type: 'select', label: 'Gender', options: ['Male', 'Female', 'Other', 'Unknown'] },
      birthDate: { type: 'date', label: 'Birth Date', description: 'Date of birth' },
      birthPlace: { type: 'text', label: 'Birth Place', description: 'City, state, country' },
      livingStatus: { type: 'select', label: 'Status', options: ['Living', 'Deceased', 'Unknown'] },
      deathDate: { type: 'date', label: 'Death Date', description: 'Date of death' },
      deathPlace: { type: 'text', label: 'Death Place', description: 'Place of death' },
      marriageDate: { type: 'date', label: 'Marriage Date', description: 'Marriage date' },
      marriagePlace: { type: 'text', label: 'Marriage Place', description: 'Marriage location' },
      spouseName: { type: 'text', label: 'Spouse', description: 'Spouse name' },
      occupation: { type: 'text', label: 'Occupation', description: 'Career' },
      photoURL: { type: 'text', label: 'Photo URL', description: 'Photo link' },
      dnaInfo: { type: 'textarea', label: 'DNA Info', description: 'DNA test results' },
      sources: { type: 'textarea', label: 'Sources', description: 'Documents, certificates' },
      relationshipType: { type: 'select', label: 'Relationship', options: ['Biological', 'Adopted', 'Step', 'Foster', 'In-Law', 'Unknown'] }
    },
    includeTracking: false,
    includeDependencies: true
  },

  dialogue: {
    name: 'Dialogue & Rhetoric',
    icon: '💬',
    description: 'Analyze conversations, debates, and rhetoric',
    levels: ['Conversation', 'Speaker', 'Statement', 'Point'],
    phaseSubtitles: ['Speaker A', 'Speaker B', 'Speaker C', 'Moderator'],
    types: ['Logical Argument', 'Emotional Appeal (Pathos)', 'Ethical Appeal (Ethos)', 'Statistical Evidence', 'Anecdotal Evidence', 'Rhetorical Question', 'Counterargument', 'Deflection/Dodge', 'Concession/Agreement'],
    fields: {
      speaker: { type: 'text', label: 'Speaker', description: 'Who is making this statement' },
      verbatimQuote: { type: 'textarea', label: 'Quote', description: 'Exact words' },
      rhetoricalDevice: { type: 'select', label: 'Device', options: ['Logos', 'Pathos', 'Ethos', 'Kairos', 'Metaphor', 'Anaphora', 'Rhetorical Question', 'Appeal to Authority', 'None'] },
      logicalStructure: { type: 'textarea', label: 'Logic', description: 'Premises and conclusion' },
      fallaciesPresent: { type: 'textarea', label: 'Fallacies', description: 'Logical fallacies' },
      hiddenMotivation: { type: 'textarea', label: 'Hidden Motivation', description: 'Unstated goals' },
      emotionalTone: { type: 'select', label: 'Tone', options: ['Calm', 'Passionate', 'Angry', 'Defensive', 'Confident', 'Hesitant', 'Sarcastic', 'Neutral'] },
      counterargument: { type: 'textarea', label: 'Counter', description: 'Strongest rebuttal' },
      evidenceQuality: { type: 'select', label: 'Evidence Quality', options: ['Strong', 'Moderate', 'Weak', 'None', 'Misleading'] },
      effectivenessRating: { type: 'number', label: 'Effectiveness', description: 'Persuasiveness (1-10)', min: 1, max: 10 }
    },
    includeTracking: false,
    includeDependencies: true,
    dialecticMode: true
  },

  filesystem: {
    name: 'File System',
    icon: '💾',
    description: 'Organize files and folders from local/cloud drives',
    levels: ['Drive', 'Folder', 'File/Folder', 'File'],
    phaseSubtitles: ['Documents', 'Downloads', 'Desktop', 'Pictures', 'Videos', 'Projects'],
    types: ['Folder', 'Shared Folder', 'Cloud Folder', 'PDF', 'Word Doc', 'Spreadsheet', 'Presentation', 'Text File', 'Image', 'Video', 'Audio', 'Code', 'HTML', 'CSS', 'JavaScript', 'Python', 'Archive', 'ZIP', 'Executable', 'Database', 'Unknown'],
    fields: {
      fileSize: { type: 'number', label: 'Size (bytes)', description: 'Size in bytes' },
      fileExtension: { type: 'text', label: 'Extension', description: 'File type' },
      filePath: { type: 'text', label: 'Path', description: 'Complete path' },
      dateModified: { type: 'datetime-local', label: 'Modified', description: 'Last modification' },
      dateCreated: { type: 'datetime-local', label: 'Created', description: 'Creation date' },
      fileOwner: { type: 'text', label: 'Owner', description: 'Owner email' },
      sharedWith: { type: 'textarea', label: 'Shared With', description: 'Users with access' },
      permissions: { type: 'select', label: 'Permissions', options: ['Read Only', 'Read/Write', 'Owner', 'Admin', 'None'] },
      driveType: { type: 'select', label: 'Drive', options: ['Local', 'Google Drive', 'OneDrive', 'Dropbox', 'iCloud', 'S3', 'Other'] },
      mimeType: { type: 'text', label: 'MIME Type', description: 'MIME type' },
      tags: { type: 'text', label: 'Tags', description: 'Custom tags' },
      fileUrl: { type: 'text', label: 'URL', description: 'Cloud URL' },
      isFolder: { type: 'checkbox', label: 'Is Folder', description: 'Is this a folder' }
    },
    includeTracking: false,
    includeDependencies: false,
    isFlexibleDepth: true
  },

  gmail: {
    name: 'Email Workflow',
    icon: '📧',
    description: 'Import and analyze Gmail threads',
    levels: ['Inbox/Campaign', 'Label/Stage', 'Thread', 'Message'],
    phaseSubtitles: ['Inbox', 'Sent', 'Important', 'Archive'],
    types: ['Cold Outreach', 'Newsletter', 'Response', 'Follow-up', 'Internal Update', 'Transactional'],
    fields: {
      recipientEmail: { type: 'text', label: 'To', description: 'Primary recipient' },
      ccEmail: { type: 'text', label: 'CC', description: 'CC recipients' },
      subjectLine: { type: 'text', label: 'Subject', description: 'Email subject' },
      emailBody: { type: 'textarea', label: 'Body', description: 'Main content' },
      sendDate: { type: 'date', label: 'Date', description: 'When sent/received' },
      status: { type: 'select', label: 'Status', options: ['Draft', 'Ready', 'Sent', 'Replied', 'Archived'] },
      threadId: { type: 'text', label: 'Thread ID', description: 'Gmail thread ID' },
      messageCount: { type: 'number', label: 'Messages', description: 'Messages in thread' },
      sender: { type: 'text', label: 'From', description: 'Email sender' },
      labels: { type: 'text', label: 'Labels', description: 'Gmail labels' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  'knowledge-base': {
    name: 'Knowledge Base',
    icon: '📚',
    description: 'Document corpus for knowledge retrieval and AI context',
    levels: ['Knowledge Base', 'Source', 'Section', 'Chunk'],
    phaseSubtitles: ['Documents', 'Web Pages', 'Notes', 'Research'],
    types: ['PDF Document', 'Web Page', 'Plain Text', 'Markdown', 'Personal Note', 'Research Paper', 'Article', 'Transcript'],
    fields: {
      sourceUrl: { type: 'text', label: 'Source URL', description: 'Original source' },
      sourceType: { type: 'select', label: 'Type', options: ['PDF', 'Web Page', 'Plain Text', 'Markdown', 'Note', 'Paper', 'Article', 'Transcript'] },
      importDate: { type: 'date', label: 'Imported', description: 'When imported' },
      author: { type: 'text', label: 'Author', description: 'Content author' },
      wordCount: { type: 'number', label: 'Words', description: 'Total words' },
      chunkIndex: { type: 'number', label: 'Chunk #', description: 'Position in document' },
      relevanceScore: { type: 'number', label: 'Relevance', description: 'How relevant (0-100)', min: 0, max: 100 },
      tags: { type: 'text', label: 'Tags', description: 'Keywords' }
    },
    includeTracking: false,
    includeDependencies: false
  },

  capex: {
    name: 'CAPEX / Angel Pitch',
    icon: '💰',
    description: 'Investor-ready capital expenditure structure for fundraising',
    levels: ['Project', 'Funding Phase', 'Investment', 'Deliverable'],
    phaseSubtitles: ['Seed', 'Series A', 'Series B'],
    types: ['Equipment', 'Infrastructure', 'Validation', 'Development', 'Milestone', 'Risk Mitigation', 'Working Capital', 'Personnel'],
    fields: {
      cost: { type: 'number', label: 'Cost', description: 'Capital expenditure' },
      risk: { type: 'text', label: 'Risk', description: 'Primary risk' },
      mitigation: { type: 'text', label: 'Mitigation', description: 'Risk mitigation strategy' },
      valuationImpact: { type: 'text', label: 'Valuation Impact', description: 'How this affects valuation' },
      leadTime: { type: 'text', label: 'Lead Time', description: 'Time to complete' }
    },
    includeTracking: true,
    includeDependencies: true
  },

  freespeech: {
    name: 'Free Speech',
    icon: '🎙️',
    description: 'Stream-of-consciousness voice capture with psychological pattern analysis',
    levels: ['Session', 'Theme', 'Pattern', 'Evidence'],
    phaseSubtitles: ['Surface Themes', 'Hidden Patterns', 'Contradictions', 'Silences', 'Recurring Structures'],
    types: ['Repetition', 'Emotional Weight', 'Contradiction', 'Avoidance', 'Implicit Belief', 'Named Entity', 'Sentence Structure'],
    fields: {
      frequency: { type: 'number', label: 'Frequency', description: 'Times pattern appeared' },
      emotionalIntensity: { type: 'select', label: 'Intensity', options: ['Low', 'Medium', 'High', 'Peak'] },
      quotedText: { type: 'textarea', label: 'Quote', description: 'Verbatim speech' },
      insight: { type: 'textarea', label: 'Insight', description: 'Psychological interpretation' }
    },
    includeTracking: false,
    includeDependencies: false
  },

  lifetree: {
    name: 'LifeTree',
    icon: '🌳',
    description: 'Biographical timeline for a life story',
    levels: ['Life', 'Decade', 'Event', 'Detail'],
    phaseSubtitles: [], // Auto-generated from birth year
    types: ['Birth', 'Family', 'Education', 'Career', 'Relationship', 'Residence/Move', 'Health', 'Milestone', 'Loss', 'Travel', 'Achievement', 'Memory/Story'],
    fields: {
      eventDate: { type: 'text', label: 'Date', description: 'Natural language date' },
      age: { type: 'number', label: 'Age', description: 'Auto-calculated', computed: true },
      location: { type: 'text', label: 'Location', description: 'Event location' },
      people: { type: 'text', label: 'People', description: 'People involved' },
      emotion: { type: 'select', label: 'Emotion', options: ['Joyful', 'Proud', 'Bittersweet', 'Difficult', 'Routine', 'Milestone'] },
      source: { type: 'text', label: 'Source', description: 'Who contributed memory' },
      confidence: { type: 'select', label: 'Confidence', options: ['Exact', 'Approximate', 'Family legend'] },
      historicalContext: { type: 'textarea', label: 'Historical Context', description: 'World events', computed: true },
      locationContext: { type: 'textarea', label: 'Location Context', description: 'Place description', computed: true },
      mediaUrl: { type: 'text', label: 'Media URL', description: 'Photo/document link' }
    },
    includeTracking: false,
    includeDependencies: false,
    requiresBirthYear: true,
    supportsDeathYear: true,
    autoGenerateDecades: true
  },

  custom: {
    name: 'Custom Names',
    icon: '✏️',
    description: 'Define your own names for all four levels',
    levels: ['Level 0', 'Level 1', 'Level 2', 'Level 3'],
    phaseSubtitles: [],
    types: [],
    fields: {},
    includeTracking: true,
    includeDependencies: true,
    customizable: true
  }
};

/**
 * Get all pattern keys
 */
function getPatternKeys() {
  return Object.keys(PATTERNS);
}

/**
 * Get pattern by key
 */
function getPattern(key) {
  return PATTERNS[key] || null;
}

/**
 * Get pattern summary (for listing)
 */
function getPatternSummaries() {
  return Object.entries(PATTERNS).map(([key, p]) => ({
    key,
    name: p.name,
    icon: p.icon,
    description: p.description,
    levels: p.levels
  }));
}

/**
 * Get pattern with full detail
 */
function getPatternDetail(key) {
  const pattern = PATTERNS[key];
  if (!pattern) return null;

  return {
    key,
    ...pattern,
    fieldCount: Object.keys(pattern.fields || {}).length,
    typeCount: (pattern.types || []).length
  };
}

module.exports = {
  PATTERNS,
  getPatternKeys,
  getPattern,
  getPatternSummaries,
  getPatternDetail
};
