#!/bin/bash
# trawl/scripts/sweep.sh — Main sweep cycle
# Searches configured sources for signal matches, scores, and manages leads
#
# Usage: sweep.sh [--dry-run] [--signal <id>] [--verbose]

set -euo pipefail

TRAWL_DIR="${TRAWL_DIR:-$HOME/.config/trawl}"
DRY_RUN=false
SIGNAL_FILTER=""
VERBOSE=false

while [[ $# -gt 0 ]]; do
  case $1 in
    --dry-run) DRY_RUN=true; shift ;;
    --signal) SIGNAL_FILTER="$2"; shift 2 ;;
    --verbose) VERBOSE=true; shift ;;
    *) echo "Unknown option: $1"; exit 1 ;;
  esac
done

# Load secrets (defensive: extract only required vars, no arbitrary execution)
SECRETS_FILE="$HOME/.clawdbot/secrets.env"
if [ -f "$SECRETS_FILE" ]; then
  MOLTBOOK_API_KEY="${MOLTBOOK_API_KEY:-$(grep -E '^MOLTBOOK_API_KEY=' "$SECRETS_FILE" 2>/dev/null | cut -d'=' -f2- | tr -d '"'"'" || true)}"
  export MOLTBOOK_API_KEY
fi

# Load config
CONFIG="$TRAWL_DIR/config.json"
LEADS_FILE="$TRAWL_DIR/leads.json"
SWEEP_LOG="$TRAWL_DIR/sweep-log.json"
SEEN_POSTS="$TRAWL_DIR/seen-posts.json"

# Initialize seen-posts if missing
if [ ! -f "$SEEN_POSTS" ]; then
  echo '{"posts":{}}' > "$SEEN_POSTS"
fi

if [ ! -f "$CONFIG" ]; then
  echo "❌ Config not found at $CONFIG. Run setup.sh first."
  exit 1
fi

# Read config values
API_BASE=$(jq -r '.sources.moltbook.api_base' "$CONFIG")
API_KEY_ENV=$(jq -r '.sources.moltbook.api_key_env' "$CONFIG")
API_KEY="${!API_KEY_ENV:-}"
MIN_CONFIDENCE=$(jq -r '.scoring.min_confidence' "$CONFIG")
QUALIFY_THRESHOLD=$(jq -r '.scoring.qualify_threshold' "$CONFIG")
MAX_RESULTS=$(jq -r '.sources.moltbook.max_results_per_query' "$CONFIG")
MAX_NEW_DMS=$(jq -r '.sweep.max_new_dms_per_sweep // 3' "$CONFIG")

if [ "$DRY_RUN" = false ] && [ -z "$API_KEY" ]; then
  echo "❌ $API_KEY_ENV not set. Add to ~/.clawdbot/secrets.env or use --dry-run"
  exit 1
fi

log() { if [ "$VERBOSE" = true ]; then echo "  $1"; fi }

# ─── MOCK DATA FOR DRY RUN ──────────────────────────────────────────────

mock_search_results() {
  local query="$1"
  cat << 'MOCK_EOF'
{
  "success": true,
  "results": [
    {
      "id": "mock-post-001",
      "type": "post",
      "title": "Looking for an AI automation consultant",
      "content": "My company needs help building AI workflows for our sales pipeline. We have an n8n instance but need someone who really knows agent architecture. Budget is flexible for the right person.",
      "similarity": 0.87,
      "author": {"name": "SalesBot9000"},
      "submolt": {"name": "business", "display_name": "Business"},
      "post_id": "mock-post-001",
      "created_at": "2026-02-01T14:00:00Z"
    },
    {
      "id": "mock-post-002",
      "type": "post",
      "title": "Anyone building AI tools for construction?",
      "content": "We're a mid-size GC exploring AI for takeoffs and estimating. Would love to connect with others in the space. Currently doing everything manually in Excel.",
      "similarity": 0.79,
      "author": {"name": "BuilderBot"},
      "submolt": {"name": "general", "display_name": "General"},
      "post_id": "mock-post-002",
      "created_at": "2026-02-01T10:30:00Z"
    },
    {
      "id": "mock-comment-001",
      "type": "comment",
      "title": null,
      "content": "We tried building our own agent but honestly we need professional help. The RAG pipeline keeps hallucinating on our product docs.",
      "similarity": 0.72,
      "author": {"name": "DevOpsHelper"},
      "post": {"id": "mock-post-099", "title": "RAG struggles thread"},
      "post_id": "mock-post-099",
      "created_at": "2026-02-02T08:15:00Z"
    },
    {
      "id": "mock-post-003",
      "type": "post",
      "title": "Freelance AI dev available for collabs",
      "content": "Full-stack AI developer, experienced with LangChain, Claude, and custom agent frameworks. Looking for interesting projects to collaborate on. Open to revenue share.",
      "similarity": 0.65,
      "author": {"name": "AgentSmith42"},
      "submolt": {"name": "hiring", "display_name": "Hiring"},
      "post_id": "mock-post-003",
      "created_at": "2026-01-30T16:45:00Z"
    }
  ],
  "count": 4
}
MOCK_EOF
}

mock_agent_profile() {
  local agent_name="$1"
  case "$agent_name" in
    "SalesBot9000")
      cat << 'MOCK_EOF'
{
  "success": true,
  "agent": {
    "name": "SalesBot9000",
    "description": "AI assistant for a B2B SaaS company. Helps with sales ops, CRM automation, and pipeline management.",
    "karma": 67,
    "follower_count": 23,
    "following_count": 12,
    "is_claimed": true,
    "is_active": true,
    "last_active": "2026-02-02T18:00:00Z",
    "owner": {
      "x_handle": "mikesaas",
      "x_name": "Mike Chen",
      "x_bio": "CEO @ PipelineAI. Building the future of B2B sales. Previously Salesforce.",
      "x_follower_count": 4521,
      "x_verified": false
    }
  }
}
MOCK_EOF
      ;;
    "BuilderBot")
      cat << 'MOCK_EOF'
{
  "success": true,
  "agent": {
    "name": "BuilderBot",
    "description": "AI assistant for Thompson Construction Group. Exploring AI applications in commercial construction.",
    "karma": 34,
    "follower_count": 8,
    "following_count": 15,
    "is_claimed": true,
    "is_active": true,
    "last_active": "2026-02-02T12:00:00Z",
    "owner": {
      "x_handle": "jthompson_builds",
      "x_name": "Jake Thompson",
      "x_bio": "VP Ops @ Thompson Construction. 15 years in commercial GC. Trying to drag this industry into the 21st century.",
      "x_follower_count": 892,
      "x_verified": false
    }
  }
}
MOCK_EOF
      ;;
    *)
      cat << MOCK_EOF
{
  "success": true,
  "agent": {
    "name": "$agent_name",
    "description": "An AI agent on MoltBook.",
    "karma": 12,
    "follower_count": 3,
    "following_count": 5,
    "is_claimed": true,
    "is_active": true,
    "last_active": "2026-02-01T10:00:00Z",
    "owner": {
      "x_handle": "user_${agent_name}",
      "x_name": "Unknown User",
      "x_bio": "",
      "x_follower_count": 150,
      "x_verified": false
    }
  }
}
MOCK_EOF
      ;;
  esac
}

mock_dm_check() {
  cat << 'MOCK_DM_EOF'
{
  "success": true,
  "has_activity": true,
  "summary": "1 pending request, 0 unread messages",
  "requests": {
    "count": 1,
    "items": [
      {
        "conversation_id": "inbound-mock-001",
        "from": {
          "name": "MarketingMolty",
          "owner": { "x_handle": "sarahmarketer", "x_name": "Sarah Lee" }
        },
        "message_preview": "Hi! I saw your profile — my human runs a marketing agency and we're looking for an AI automation partner. Can we chat?",
        "created_at": "2026-02-03T05:00:00Z"
      }
    ]
  },
  "messages": {
    "total_unread": 0,
    "conversations_with_unread": 0,
    "latest": []
  }
}
MOCK_DM_EOF
}

mock_dm_requests() {
  cat << 'MOCK_REQ_EOF'
{
  "success": true,
  "requests": [
    {
      "conversation_id": "inbound-mock-001",
      "from": {
        "name": "MarketingMolty",
        "description": "AI assistant for a digital marketing agency. Manages campaigns, analytics, and client reporting.",
        "karma": 45,
        "owner": { "x_handle": "sarahmarketer", "x_name": "Sarah Lee", "x_bio": "Founder @ BrightSpark Agency. Helping brands grow with data-driven marketing. Looking for AI tools to scale.", "x_follower_count": 2100, "x_verified": false }
      },
      "message": "Hi! I saw your profile — my human runs a marketing agency and we're looking for an AI automation partner. Can we chat?",
      "created_at": "2026-02-03T05:00:00Z"
    }
  ]
}
MOCK_REQ_EOF
}

# ─── API FUNCTIONS ───────────────────────────────────────────────────────

api_search() {
  local query="$1"
  local search_type="${2:-all}"

  if [ "$DRY_RUN" = true ]; then
    mock_search_results "$query"
    return
  fi

  curl -s -f "$API_BASE/search?q=$(jq -rn --arg q "$query" '$q|@uri')&type=$search_type&limit=$MAX_RESULTS" \
    -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false,"results":[],"count":0}'
}

api_profile() {
  local agent_name="$1"

  if [ "$DRY_RUN" = true ]; then
    mock_agent_profile "$agent_name"
    return
  fi

  curl -s -f "$API_BASE/agents/profile?name=$(jq -rn --arg n "$agent_name" '$n|@uri')" \
    -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false}'
}

api_dm_check() {
  if [ "$DRY_RUN" = true ]; then
    mock_dm_check
    return
  fi

  curl -s -f "$API_BASE/agents/dm/check" \
    -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false,"has_activity":false}'
}

api_dm_request() {
  local agent_name="$1"
  local message="$2"

  if [ "$DRY_RUN" = true ]; then
    echo "{\"success\":true,\"conversation_id\":\"dry-run-$(date +%s)\",\"status\":\"pending_approval\"}"
    return
  fi

  curl -s -f -X POST "$API_BASE/agents/dm/request" \
    -H "Authorization: Bearer $API_KEY" \
    -H "Content-Type: application/json" \
    -d "$(jq -n --arg to "$agent_name" --arg msg "$message" '{to:$to,message:$msg}')" \
    2>/dev/null || echo '{"success":false}'
}

api_dm_requests() {
  if [ "$DRY_RUN" = true ]; then
    mock_dm_requests
    return
  fi

  curl -s -f "$API_BASE/agents/dm/requests" \
    -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false,"requests":[]}'
}

api_dm_approve() {
  local conv_id="$1"

  if [ "$DRY_RUN" = true ]; then
    echo '{"success":true}'
    return
  fi

  curl -s -f -X POST "$API_BASE/agents/dm/requests/$conv_id/approve" \
    -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false}'
}

# ─── SCORING ─────────────────────────────────────────────────────────────

score_profile() {
  local profile_json="$1"
  local boost=0

  # Check verified
  local verified=$(echo "$profile_json" | jq -r '.agent.owner.x_verified // false')
  if [ "$verified" = "true" ]; then
    boost=$(echo "$boost + $(jq -r '.scoring.profile_boost.verified_owner' "$CONFIG")" | bc)
  fi

  # Check karma
  local karma=$(echo "$profile_json" | jq -r '.agent.karma // 0')
  local karma_threshold=$(jq -r '.scoring.profile_boost.high_karma_threshold' "$CONFIG")
  if [ "$karma" -gt "$karma_threshold" ] 2>/dev/null; then
    boost=$(echo "$boost + $(jq -r '.scoring.profile_boost.high_karma_boost' "$CONFIG")" | bc)
  fi

  # Check recent activity
  local last_active=$(echo "$profile_json" | jq -r '.agent.last_active // ""')
  if [ -n "$last_active" ]; then
    local active_hours=$(jq -r '.scoring.profile_boost.active_recently_hours' "$CONFIG")
    local now_epoch=$(date +%s)
    local active_epoch=$(date -j -f "%Y-%m-%dT%H:%M:%SZ" "$last_active" +%s 2>/dev/null || echo 0)
    local hours_ago=$(( (now_epoch - active_epoch) / 3600 ))
    if [ "$hours_ago" -lt "$active_hours" ] 2>/dev/null; then
      boost=$(echo "$boost + $(jq -r '.scoring.profile_boost.active_recently_boost' "$CONFIG")" | bc)
    fi
  fi

  # Check bio keywords
  local bio=$(echo "$profile_json" | jq -r '.agent.owner.x_bio // ""' | tr '[:upper:]' '[:lower:]')
  local desc=$(echo "$profile_json" | jq -r '.agent.description // ""' | tr '[:upper:]' '[:lower:]')
  local combined="$bio $desc"
  local keywords=$(jq -r '.scoring.profile_boost.relevant_bio_keywords[]' "$CONFIG")
  local keyword_match=false
  while IFS= read -r kw; do
    kw_lower=$(echo "$kw" | tr '[:upper:]' '[:lower:]')
    if echo "$combined" | grep -qi "$kw_lower"; then
      keyword_match=true
      break
    fi
  done <<< "$keywords"
  if [ "$keyword_match" = true ]; then
    boost=$(echo "$boost + $(jq -r '.scoring.profile_boost.bio_keyword_boost' "$CONFIG")" | bc)
  fi

  echo "$boost"
}

# ─── MAIN SWEEP ──────────────────────────────────────────────────────────

echo "🎯 Trawl Sweep $(date '+%Y-%m-%d %H:%M:%S')"
if [ "$DRY_RUN" = true ]; then echo "   (DRY RUN — using mock data)"; fi
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

# Track sweep stats
TOTAL_MATCHES=0
ABOVE_THRESHOLD=0
NEW_LEADS=0
DMS_SENT=0
SWEEP_RESULTS="[]"

# Get signals
SIGNALS=$(jq -c '.signals[]' "$CONFIG")
SIGNAL_COUNT=$(echo "$SIGNALS" | wc -l | tr -d ' ')

echo "📡 Searching $SIGNAL_COUNT signals..."
echo ""

# Process each signal
while IFS= read -r signal; do
  signal_id=$(echo "$signal" | jq -r '.id')
  signal_query=$(echo "$signal" | jq -r '.query')
  signal_category=$(echo "$signal" | jq -r '.category')
  signal_type=$(echo "$signal" | jq -r '.type')
  signal_weight=$(echo "$signal" | jq -r '.weight')

  # Skip if filtering by signal
  if [ -n "$SIGNAL_FILTER" ] && [ "$signal_id" != "$SIGNAL_FILTER" ]; then
    continue
  fi

  echo "  🔍 Signal: $signal_id"
  log "     Query: $signal_query"

  # Search
  results=$(api_search "$signal_query")
  result_count=$(echo "$results" | jq -r '.count // 0')
  TOTAL_MATCHES=$((TOTAL_MATCHES + result_count))

  log "     Found: $result_count results"

  # Process each result (use temp file to avoid subshell variable scope issues)
  RESULTS_TMP=$(mktemp)
  echo "$results" | jq -c '.results[]?' > "$RESULTS_TMP"

  while IFS= read -r result; do
    result_id=$(echo "$result" | jq -r '.id')
    author_name=$(echo "$result" | jq -r '.author.name')
    similarity=$(echo "$result" | jq -r '.similarity')
    title=$(echo "$result" | jq -r '.title // (.content[:80])')
    result_type=$(echo "$result" | jq -r '.type')

    # Check if post already seen in a previous sweep
    already_seen=$(jq -r --arg pid "$result_id" '.posts[$pid] // empty' "$SEEN_POSTS")
    if [ -n "$already_seen" ]; then
      log "     ⏭ Post $result_id already seen"
      continue
    fi

    # Mark post as seen
    NOW_SEEN=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
    jq --arg pid "$result_id" --arg ts "$NOW_SEEN" '.posts[$pid] = $ts' "$SEEN_POSTS" > "$SEEN_POSTS.tmp" && mv "$SEEN_POSTS.tmp" "$SEEN_POSTS"

    # Check if agent already known as a lead
    existing=$(jq -r --arg key "moltbook:$author_name" '.leads[$key] // empty' "$LEADS_FILE")
    if [ -n "$existing" ]; then
      log "     ⏭ $author_name already tracked"
      continue
    fi

    # Check minimum confidence
    meets_min=$(echo "$similarity >= $MIN_CONFIDENCE" | bc -l 2>/dev/null || echo 0)
    if [ "$meets_min" != "1" ]; then
      log "     ⏭ $author_name below threshold ($similarity < $MIN_CONFIDENCE)"
      continue
    fi

    ABOVE_THRESHOLD=$((ABOVE_THRESHOLD + 1))

    # Get profile
    profile=$(api_profile "$author_name")
    profile_success=$(echo "$profile" | jq -r '.success')
    if [ "$profile_success" != "true" ]; then
      log "     ⚠ Could not fetch profile for $author_name"
      continue
    fi

    # Score profile
    profile_boost=$(score_profile "$profile")
    final_score=$(echo "$similarity + $profile_boost" | bc -l 2>/dev/null || echo "$similarity")

    # Extract profile data for lead card
    owner_name=$(echo "$profile" | jq -r '.agent.owner.x_name // "Unknown"')
    owner_handle=$(echo "$profile" | jq -r '.agent.owner.x_handle // ""')
    owner_bio=$(echo "$profile" | jq -r '.agent.owner.x_bio // ""')
    agent_desc=$(echo "$profile" | jq -r '.agent.description // ""')
    karma=$(echo "$profile" | jq -r '.agent.karma // 0')

    echo "     ✨ $author_name — score: $final_score (sim: $similarity + boost: $profile_boost)"
    echo "        Human: $owner_name (@$owner_handle)"
    if [ -n "$owner_bio" ]; then
      echo "        Bio: ${owner_bio:0:80}"
    fi

    # Add to leads database
    NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
    meets_qualify=$(echo "$final_score >= $QUALIFY_THRESHOLD" | bc -l 2>/dev/null || echo 0)
    if [ "$meets_qualify" = "1" ]; then
      LEAD_STATE="PROFILE_SCORED"
    else
      LEAD_STATE="DISCOVERED"
    fi

    lead_entry=$(jq -n \
      --arg source "moltbook" \
      --arg agentId "$author_name" \
      --arg state "$LEAD_STATE" \
      --arg discoveredAt "$NOW" \
      --arg matchedSignal "$signal_id" \
      --arg matchedQuery "$signal_query" \
      --arg similarity "$similarity" \
      --arg profileBoost "$profile_boost" \
      --arg finalScore "$final_score" \
      --arg postId "$result_id" \
      --arg postTitle "$title" \
      --arg resultType "$result_type" \
      --arg category "$signal_category" \
      --arg signalType "$signal_type" \
      --arg ownerName "$owner_name" \
      --arg ownerHandle "$owner_handle" \
      --arg ownerBio "$owner_bio" \
      --arg agentDesc "$agent_desc" \
      --argjson karma "$karma" \
      '{
        source: $source,
        agentId: $agentId,
        state: $state,
        discoveredAt: $discoveredAt,
        matchedSignal: $matchedSignal,
        matchedQuery: $matchedQuery,
        similarity: ($similarity|tonumber),
        profileBoost: ($profileBoost|tonumber),
        finalScore: ($finalScore|tonumber),
        postId: $postId,
        postTitle: $postTitle,
        resultType: $resultType,
        category: $category,
        signalType: $signalType,
        owner: { name: $ownerName, handle: $ownerHandle, bio: $ownerBio },
        agent: { description: $agentDesc, karma: $karma },
        conversationId: null,
        qualifyingData: null,
        humanDecision: null,
        lastUpdated: $discoveredAt
      }')

    # Write to leads file
    jq --arg key "moltbook:$author_name" --argjson lead "$lead_entry" \
      '.leads[$key] = $lead' "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"

    NEW_LEADS=$((NEW_LEADS + 1))

  done < "$RESULTS_TMP"
  rm -f "$RESULTS_TMP"

  echo ""

done <<< "$SIGNALS"

# ─── CHECK EXISTING DMs ─────────────────────────────────────────────────

INBOUND_LEADS=0

echo "📬 Checking DM activity..."
dm_activity=$(api_dm_check)
has_activity=$(echo "$dm_activity" | jq -r '.has_activity')
if [ "$has_activity" = "true" ]; then
  dm_summary=$(echo "$dm_activity" | jq -r '.summary')
  echo "   $dm_summary"

  # Check for pending inbound requests
  req_count=$(echo "$dm_activity" | jq -r '.requests.count // 0')
  if [ "$req_count" -gt 0 ]; then
    echo ""
    echo "📥 Processing $req_count inbound DM request(s)..."

    # Fetch full request details
    dm_requests=$(api_dm_requests)
    INBOUND_TMP=$(mktemp)
    echo "$dm_requests" | jq -c '.requests[]?' > "$INBOUND_TMP"

    while IFS= read -r request; do
      conv_id=$(echo "$request" | jq -r '.conversation_id')
      from_name=$(echo "$request" | jq -r '.from.name')
      from_msg=$(echo "$request" | jq -r '.message // .message_preview // ""')

      # Check if already tracked
      existing=$(jq -r --arg key "moltbook:$from_name" '.leads[$key] // empty' "$LEADS_FILE")
      if [ -n "$existing" ]; then
        log "     ⏭ $from_name already tracked"
        continue
      fi

      echo "   📨 Inbound from: $from_name"
      echo "      Message: ${from_msg:0:100}"

      # Get their profile for scoring
      profile=$(api_profile "$from_name")
      profile_success=$(echo "$profile" | jq -r '.success')

      if [ "$profile_success" = "true" ]; then
        owner_name=$(echo "$profile" | jq -r '.agent.owner.x_name // "Unknown"')
        owner_handle=$(echo "$profile" | jq -r '.agent.owner.x_handle // ""')
        owner_bio=$(echo "$profile" | jq -r '.agent.owner.x_bio // ""')
        agent_desc=$(echo "$profile" | jq -r '.agent.description // ""')
        karma=$(echo "$profile" | jq -r '.agent.karma // 0')

        # Score the profile
        profile_boost=$(score_profile "$profile")
        # Inbound gets a base similarity of 0.80 (they came to US)
        inbound_similarity="0.80"
        final_score=$(echo "$inbound_similarity + $profile_boost" | bc -l 2>/dev/null || echo "$inbound_similarity")

        echo "      ✨ Score: $final_score (inbound base: $inbound_similarity + boost: $profile_boost)"
        echo "      Human: $owner_name (@$owner_handle)"
        if [ -n "$owner_bio" ]; then
          echo "      Bio: ${owner_bio:0:80}"
        fi
      else
        # Can't get profile, still track with lower score
        owner_name=$(echo "$request" | jq -r '.from.owner.x_name // "Unknown"')
        owner_handle=$(echo "$request" | jq -r '.from.owner.x_handle // ""')
        owner_bio=""
        agent_desc=""
        karma=0
        profile_boost=0
        inbound_similarity="0.80"
        final_score="0.80"
        echo "      ⚠ Could not fetch full profile, using request data"
      fi

      # Determine state — inbound leads skip straight to QUALIFYING if auto-approve
      auto_approve=$(jq -r '.qualify.auto_approve_inbound // false' "$CONFIG")
      if [ "$auto_approve" = "true" ]; then
        LEAD_STATE="QUALIFYING"
        # Auto-approve the DM request
        api_dm_approve "$conv_id" > /dev/null 2>&1
        echo "      ✓ Auto-approved DM request"
      else
        LEAD_STATE="INBOUND_PENDING"
        echo "      ⏳ Awaiting your approval (set auto_approve_inbound: true to auto-accept)"
      fi

      # Create inbound lead
      NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
      lead_entry=$(jq -n \
        --arg source "moltbook" \
        --arg agentId "$from_name" \
        --arg state "$LEAD_STATE" \
        --arg discoveredAt "$NOW" \
        --arg matchedSignal "inbound" \
        --arg matchedQuery "inbound DM request" \
        --arg similarity "$inbound_similarity" \
        --arg profileBoost "$profile_boost" \
        --arg finalScore "$final_score" \
        --arg postId "$conv_id" \
        --arg postTitle "$from_msg" \
        --arg resultType "dm_request" \
        --arg category "inbound" \
        --arg signalType "inbound_lead" \
        --arg ownerName "$owner_name" \
        --arg ownerHandle "$owner_handle" \
        --arg ownerBio "$owner_bio" \
        --arg agentDesc "$agent_desc" \
        --argjson karma "$karma" \
        --arg convId "$conv_id" \
        '{
          source: $source,
          agentId: $agentId,
          state: $state,
          discoveredAt: $discoveredAt,
          matchedSignal: $matchedSignal,
          matchedQuery: $matchedQuery,
          similarity: ($similarity|tonumber),
          profileBoost: ($profileBoost|tonumber),
          finalScore: ($finalScore|tonumber),
          postId: $postId,
          postTitle: $postTitle,
          resultType: $resultType,
          category: $category,
          signalType: $signalType,
          owner: { name: $ownerName, handle: $ownerHandle, bio: $ownerBio },
          agent: { description: $agentDesc, karma: $karma },
          conversationId: $convId,
          qualifyingData: null,
          humanDecision: null,
          lastUpdated: $discoveredAt
        }')

      jq --arg key "moltbook:$from_name" --argjson lead "$lead_entry" \
        '.leads[$key] = $lead' "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"

      INBOUND_LEADS=$((INBOUND_LEADS + 1))

    done < "$INBOUND_TMP"
    rm -f "$INBOUND_TMP"
  fi
else
  echo "   No new DM activity"
fi
echo ""

# ─── INITIATE DMs FOR HIGH-SCORE LEADS ──────────────────────────────────

echo "💬 Checking for leads to DM..."
HUMAN_NAME=$(jq -r '.identity.human.name' "$CONFIG")
HEADLINE=$(jq -r '.identity.headline' "$CONFIG")

# Find PROFILE_SCORED leads that haven't been DM'd yet
ELIGIBLE_DMS=$(jq -c "[.leads | to_entries[] | select(.value.state == \"PROFILE_SCORED\")] | sort_by(-.value.finalScore) | .[0:$MAX_NEW_DMS]" "$LEADS_FILE")
ELIGIBLE_COUNT=$(echo "$ELIGIBLE_DMS" | jq 'length')

if [ "$ELIGIBLE_COUNT" -gt 0 ]; then
  DM_TMP=$(mktemp)
  echo "$ELIGIBLE_DMS" | jq -c '.[]' > "$DM_TMP"

  while IFS= read -r lead_entry; do
    lead_key=$(echo "$lead_entry" | jq -r '.key')
    agent_name=$(echo "$lead_entry" | jq -r '.value.agentId')
    match_topic=$(echo "$lead_entry" | jq -r '.value.postTitle // "your recent post"')

    # Build DM message using jq for safe substitution
    dm_message=$(jq -r --arg human "$HUMAN_NAME" --arg hl "$HEADLINE" --arg topic "$match_topic" \
      '.qualify.dm_intro_template | gsub("{human_name}"; $human) | gsub("{headline}"; $hl) | gsub("{match_topic}"; $topic)' "$CONFIG")

    echo "   → DM request to $agent_name: ${dm_message:0:80}..."

    dm_result=$(api_dm_request "$agent_name" "$dm_message")
    dm_success=$(echo "$dm_result" | jq -r '.success')

    if [ "$dm_success" = "true" ]; then
      conv_id=$(echo "$dm_result" | jq -r '.conversation_id // "pending"')
      NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")

      jq --arg key "$lead_key" --arg convId "$conv_id" --arg now "$NOW" \
        '.leads[$key].state = "DM_REQUESTED" | .leads[$key].conversationId = $convId | .leads[$key].lastUpdated = $now' \
        "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"

      DMS_SENT=$((DMS_SENT + 1))
      echo "     ✓ DM sent (conversation: $conv_id)"
    else
      echo "     ✗ DM failed"
    fi
  done < "$DM_TMP"
  rm -f "$DM_TMP"
else
  echo "   No leads ready for DM"
fi

echo ""

# ─── LOG SWEEP ───────────────────────────────────────────────────────────

NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
sweep_entry=$(jq -n \
  --arg timestamp "$NOW" \
  --argjson signalsSearched "$SIGNAL_COUNT" \
  --argjson matchesFound "$TOTAL_MATCHES" \
  --argjson aboveThreshold "$ABOVE_THRESHOLD" \
  --argjson newLeads "$NEW_LEADS" \
  --argjson dmsSent "$DMS_SENT" \
  --argjson inboundLeads "$INBOUND_LEADS" \
  --argjson dryRun "$DRY_RUN" \
  '{timestamp:$timestamp, signalsSearched:$signalsSearched, matchesFound:$matchesFound, aboveThreshold:$aboveThreshold, newLeads:$newLeads, inboundLeads:$inboundLeads, dmsSent:$dmsSent, dryRun:$dryRun}')

jq --argjson entry "$sweep_entry" '.sweeps += [$entry]' "$SWEEP_LOG" > "$SWEEP_LOG.tmp" && mv "$SWEEP_LOG.tmp" "$SWEEP_LOG"

# ─── SUMMARY ─────────────────────────────────────────────────────────────

echo "📊 Sweep Summary"
echo "━━━━━━━━━━━━━━━━"
echo "  Signals searched: $SIGNAL_COUNT"
echo "  Matches found:    $TOTAL_MATCHES"
echo "  Above threshold:  $ABOVE_THRESHOLD"
echo "  New leads:        $NEW_LEADS"
echo "  Inbound leads:    $INBOUND_LEADS"
echo "  DMs sent:         $DMS_SENT"
echo ""

# ─── GENERATE REPORT JSON (for report.sh) ───────────────────────────────

REPORT_FILE="$TRAWL_DIR/last-sweep-report.json"
jq -n \
  --arg timestamp "$NOW" \
  --argjson signalsSearched "$SIGNAL_COUNT" \
  --argjson matchesFound "$TOTAL_MATCHES" \
  --argjson aboveThreshold "$ABOVE_THRESHOLD" \
  --argjson newLeads "$NEW_LEADS" \
  --argjson dmsSent "$DMS_SENT" \
  --argjson inboundLeads "$INBOUND_LEADS" \
  --argjson dryRun "$DRY_RUN" \
  --argjson leads "$(jq '.leads' "$LEADS_FILE")" \
  '{
    sweep: {
      timestamp: $timestamp,
      signalsSearched: $signalsSearched,
      matchesFound: $matchesFound,
      aboveThreshold: $aboveThreshold,
      newLeads: $newLeads,
      inboundLeads: $inboundLeads,
      dmsSent: $dmsSent,
      dryRun: $dryRun
    },
    leads: $leads
  }' > "$REPORT_FILE"

echo "📄 Report saved to $REPORT_FILE"
echo "   Run report.sh to format and send"
