#!/bin/bash
# trawl/scripts/setup.sh — Initialize trawl config and data directories

set -euo pipefail

TRAWL_DIR="${TRAWL_DIR:-$HOME/.config/trawl}"
SKILL_DIR="$(cd "$(dirname "$0")/.." && pwd)"

echo "🔧 Trawl Setup"
echo "━━━━━━━━━━━━━━"

# Create config directory
mkdir -p "$TRAWL_DIR"
echo "✓ Created $TRAWL_DIR"

# Copy example config if none exists
if [ ! -f "$TRAWL_DIR/config.json" ]; then
  cp "$SKILL_DIR/config.example.json" "$TRAWL_DIR/config.json"
  echo "✓ Created config.json (from example — edit with your identity and signals)"
else
  echo "• config.json already exists (skipped)"
fi

# Initialize data files
for file in leads.json conversations.json sweep-log.json seen-posts.json; do
  if [ ! -f "$TRAWL_DIR/$file" ]; then
    if [ "$file" = "leads.json" ]; then
      echo '{"leads":{}}' > "$TRAWL_DIR/$file"
    elif [ "$file" = "conversations.json" ]; then
      echo '{"conversations":{}}' > "$TRAWL_DIR/$file"
    elif [ "$file" = "sweep-log.json" ]; then
      echo '{"sweeps":[]}' > "$TRAWL_DIR/$file"
    elif [ "$file" = "seen-posts.json" ]; then
      echo '{"posts":{}}' > "$TRAWL_DIR/$file"
    fi
    echo "✓ Created $file"
  else
    echo "• $file already exists (skipped)"
  fi
done

# Check for API key
if [ -f "$HOME/.clawdbot/secrets.env" ]; then
  if grep -q "MOLTBOOK_API_KEY" "$HOME/.clawdbot/secrets.env" 2>/dev/null; then
    echo "✓ MOLTBOOK_API_KEY found in secrets.env"
  else
    echo "⚠ MOLTBOOK_API_KEY not found in secrets.env"
    echo "  Add: MOLTBOOK_API_KEY=moltbook_xxx to ~/.clawdbot/secrets.env"
  fi
else
  echo "⚠ ~/.clawdbot/secrets.env not found"
  echo "  Create it and add: MOLTBOOK_API_KEY=moltbook_xxx"
fi

echo ""
echo "Next steps:"
echo "  1. Edit $TRAWL_DIR/config.json with your identity and signals"
echo "  2. Add MOLTBOOK_API_KEY to ~/.clawdbot/secrets.env"
echo "  3. Test: $(dirname "$0")/sweep.sh --dry-run"
echo ""
echo "🎯 Trawl is ready to hunt."
