#!/bin/bash
# trawl/scripts/qualify.sh — Process qualifying conversations
# Checks DM status, advances conversations, asks qualifying questions
# Handles both outbound (we initiated) and inbound (they initiated) leads
#
# Usage: qualify.sh [--dry-run] [--verbose]

set -euo pipefail

TRAWL_DIR="${TRAWL_DIR:-$HOME/.config/trawl}"
DRY_RUN=false
VERBOSE=false

while [[ $# -gt 0 ]]; do
  case $1 in
    --dry-run) DRY_RUN=true; shift ;;
    --verbose) VERBOSE=true; shift ;;
    *) shift ;;
  esac
done

# Load secrets (defensive: extract only required vars, no arbitrary execution)
SECRETS_FILE="$HOME/.clawdbot/secrets.env"
if [ -f "$SECRETS_FILE" ]; then
  MOLTBOOK_API_KEY="${MOLTBOOK_API_KEY:-$(grep -E '^MOLTBOOK_API_KEY=' "$SECRETS_FILE" 2>/dev/null | cut -d'=' -f2- | tr -d '"'"'" || true)}"
  export MOLTBOOK_API_KEY
fi

CONFIG="$TRAWL_DIR/config.json"
LEADS_FILE="$TRAWL_DIR/leads.json"

API_BASE=$(jq -r '.sources.moltbook.api_base' "$CONFIG")
API_KEY_ENV=$(jq -r '.sources.moltbook.api_key_env' "$CONFIG")
API_KEY="${!API_KEY_ENV:-}"

STALE_HOURS=$(jq -r '.qualify.stale_timeout_hours // 48' "$CONFIG")
MAX_TOTAL_Q=$(jq -r '.qualify.max_total_questions // 3' "$CONFIG")

log() { if [ "$VERBOSE" = true ]; then echo "  $1"; fi }

update_lead() {
  local key="$1" field="$2" value="$3"
  local NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
  jq --arg key "$key" --arg val "$value" --arg now "$NOW" \
    ".leads[\$key].$field = \$val | .leads[\$key].lastUpdated = \$now" \
    "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"
}

update_lead_state() {
  local key="$1" state="$2"
  local NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
  jq --arg key "$key" --arg state "$state" --arg now "$NOW" \
    '.leads[$key].state = $state | .leads[$key].lastUpdated = $now' \
    "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"
}

echo "💬 Trawl Qualify — $(date '+%Y-%m-%d %H:%M:%S')"
if [ "$DRY_RUN" = true ]; then echo "   (DRY RUN)"; fi
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

APPROVED=0
STALED=0
QUESTIONS_SENT=0
QUALIFIED_COUNT=0

# ─── HANDLE INBOUND PENDING ─────────────────────────────────────────────

echo "📥 Checking inbound leads..."

INBOUND_TMP=$(mktemp)
jq -c '[.leads | to_entries[] | select(.value.state == "INBOUND_PENDING")]' "$LEADS_FILE" > "$INBOUND_TMP"
INBOUND_COUNT=$(jq 'length' "$INBOUND_TMP")

if [ "$INBOUND_COUNT" -gt 0 ]; then
  echo "   $INBOUND_COUNT inbound leads awaiting approval"
  echo "   → Use: leads.sh decide <key> --pursue  (approves DM + starts qualifying)"
  echo "   → Or set auto_approve_inbound: true in config"
  echo ""

  # List them
  jq -c '.[]' "$INBOUND_TMP" | while IFS= read -r entry; do
    key=$(jq -r '.key' <<< "$entry")
    agent=$(jq -r '.value.agentId' <<< "$entry")
    owner=$(jq -r '.value.owner.name' <<< "$entry")
    score=$(jq -r '.value.finalScore' <<< "$entry")
    msg=$(jq -r '.value.postTitle // ""' <<< "$entry")
    echo "   📨 $agent ($owner) — score: $score"
    echo "      \"${msg:0:80}\""
  done
else
  echo "   No inbound leads pending"
fi
rm -f "$INBOUND_TMP"
echo ""

# ─── CHECK OUTBOUND DM REQUESTS ─────────────────────────────────────────

echo "📬 Checking outbound DM requests..."

DM_TMP=$(mktemp)
jq -c '[.leads | to_entries[] | select(.value.state == "DM_REQUESTED")]' "$LEADS_FILE" > "$DM_TMP"
DM_REQ_COUNT=$(jq 'length' "$DM_TMP")

if [ "$DM_REQ_COUNT" -gt 0 ]; then
  echo "   $DM_REQ_COUNT pending DM requests"

  ENTRIES_TMP=$(mktemp)
  jq -c '.[]' "$DM_TMP" > "$ENTRIES_TMP"

  while IFS= read -r entry; do
    key=$(jq -r '.key' <<< "$entry")
    conv_id=$(jq -r '.value.conversationId' <<< "$entry")
    agent_name=$(jq -r '.value.agentId' <<< "$entry")
    discovered=$(jq -r '.value.discoveredAt' <<< "$entry")

    # Check if stale
    discovered_epoch=$(date -j -f "%Y-%m-%dT%H:%M:%SZ" "$discovered" +%s 2>/dev/null || echo 0)
    now_epoch=$(date +%s)
    hours_waiting=$(( (now_epoch - discovered_epoch) / 3600 ))

    if [ "$hours_waiting" -gt "$STALE_HOURS" ]; then
      echo "   ⏰ $agent_name stale ($hours_waiting hours) → DM_STALE"
      update_lead_state "$key" "DM_STALE"
      STALED=$((STALED + 1))
      continue
    fi

    if [ "$DRY_RUN" = true ]; then
      # Dry run: simulate approval for first lead only
      if [ "$APPROVED" -eq 0 ]; then
        echo "   ✓ $agent_name DM approved → QUALIFYING (simulated)"
        update_lead_state "$key" "QUALIFYING"
        APPROVED=$((APPROVED + 1))
      else
        log "   ⏳ $agent_name still waiting"
      fi
    else
      # Live: check conversation via API
      conv_response=$(curl -s -f "$API_BASE/agents/dm/conversations/$conv_id" \
        -H "Authorization: Bearer $API_KEY" 2>/dev/null || echo '{"success":false}')

      if echo "$conv_response" | jq -e '.success and .messages' > /dev/null 2>&1; then
        echo "   ✓ $agent_name DM approved → QUALIFYING"
        update_lead_state "$key" "QUALIFYING"
        APPROVED=$((APPROVED + 1))
      else
        log "   ⏳ $agent_name still waiting ($hours_waiting hours)"
      fi
    fi
  done < "$ENTRIES_TMP"
  rm -f "$ENTRIES_TMP"
else
  echo "   No pending DM requests"
fi
rm -f "$DM_TMP"
echo ""

# ─── PROCESS QUALIFYING CONVERSATIONS ───────────────────────────────────

echo "🔍 Processing qualifying conversations..."

QUAL_TMP=$(mktemp)
jq -c '[.leads | to_entries[] | select(.value.state == "QUALIFYING")]' "$LEADS_FILE" > "$QUAL_TMP"
QUAL_COUNT_ACTIVE=$(jq 'length' "$QUAL_TMP")

if [ "$QUAL_COUNT_ACTIVE" -gt 0 ]; then
  echo "   $QUAL_COUNT_ACTIVE active qualifying conversations"

  QUAL_ENTRIES=$(mktemp)
  jq -c '.[]' "$QUAL_TMP" > "$QUAL_ENTRIES"

  while IFS= read -r entry; do
    key=$(jq -r '.key' <<< "$entry")
    agent_name=$(jq -r '.value.agentId' <<< "$entry")
    conv_id=$(jq -r '.value.conversationId' <<< "$entry")
    signal_type=$(jq -r '.value.signalType' <<< "$entry")
    existing_data=$(jq -r '.value.qualifyingData // empty' <<< "$entry")

    questions_asked=0
    if [ -n "$existing_data" ] && [ "$existing_data" != "null" ]; then
      questions_asked=$(jq '.questionsAsked // 0' <<< "$existing_data")
    fi

    echo "   📝 $agent_name (asked: $questions_asked/$MAX_TOTAL_Q) [$signal_type]"

    # Check if max questions reached
    if [ "$questions_asked" -ge "$MAX_TOTAL_Q" ]; then
      echo "      ✓ Max questions reached → QUALIFIED"
      update_lead_state "$key" "QUALIFIED"
      QUALIFIED_COUNT=$((QUALIFIED_COUNT + 1))
      continue
    fi

    # Get next question
    next_q_index=$((questions_asked))
    next_question=$(jq -r --argjson idx "$next_q_index" '.qualify.qualifying_questions[$idx] // empty' "$CONFIG")

    if [ -z "$next_question" ]; then
      echo "      ✓ No more questions → QUALIFIED"
      update_lead_state "$key" "QUALIFIED"
      QUALIFIED_COUNT=$((QUALIFIED_COUNT + 1))
      continue
    fi

    # For inbound leads, prepend a thank-you on first question
    if [ "$signal_type" = "inbound_lead" ] && [ "$questions_asked" -eq 0 ]; then
      next_question="Thanks for reaching out! I'd love to learn more. $next_question"
    fi

    echo "      → Q: $next_question"

    if [ "$DRY_RUN" = true ]; then
      echo "      (dry run: would send via DM)"
    else
      if [ -n "$conv_id" ] && [ "$conv_id" != "null" ]; then
        curl -s -f -X POST "$API_BASE/agents/dm/conversations/$conv_id/send" \
          -H "Authorization: Bearer $API_KEY" \
          -H "Content-Type: application/json" \
          -d "$(jq -n --arg msg "$next_question" '{message:$msg}')" > /dev/null 2>&1
      fi
    fi

    # Update qualifying data
    new_asked=$((questions_asked + 1))
    NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
    jq --arg key "$key" --argjson asked "$new_asked" --arg now "$NOW" \
      '.leads[$key].qualifyingData = {"questionsAsked": $asked, "responses": []} | .leads[$key].lastUpdated = $now' \
      "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"

    QUESTIONS_SENT=$((QUESTIONS_SENT + 1))

  done < "$QUAL_ENTRIES"
  rm -f "$QUAL_ENTRIES"
else
  echo "   No qualifying conversations"
fi
rm -f "$QUAL_TMP"
echo ""

# ─── CHECK FOR QUALIFIED LEADS READY TO REPORT ──────────────────────────

echo "📋 Checking for unreported qualified leads..."

UNREPORTED=$(jq '[.leads | to_entries[] | select(.value.state == "QUALIFIED" and .value.humanDecision == null)] | length' "$LEADS_FILE")

if [ "$UNREPORTED" -gt 0 ]; then
  echo "   🎯 $UNREPORTED qualified leads ready for your review!"
  echo "   → Run report.sh to see them"
  echo "   → Use leads.sh decide <key> --pursue or --pass"
else
  echo "   No unreported leads"
fi

echo ""

# ─── SUMMARY ─────────────────────────────────────────────────────────────

echo "📊 Qualify Summary"
echo "━━━━━━━━━━━━━━━━━━"
echo "  DMs approved:      $APPROVED"
echo "  DMs staled:        $STALED"
echo "  Questions sent:    $QUESTIONS_SENT"
echo "  Newly qualified:   $QUALIFIED_COUNT"
echo "  Awaiting review:   $UNREPORTED"
echo ""
echo "✓ Qualify cycle complete"
