#!/bin/bash
# trawl/scripts/leads.sh — Manage lead database
#
# Usage:
#   leads.sh list [--state <state>] [--category <cat>]
#   leads.sh get <lead_key>
#   leads.sh update <lead_key> --state <state>
#   leads.sh decide <lead_key> --pursue|--pass
#   leads.sh archive <lead_key>
#   leads.sh stats

set -euo pipefail

TRAWL_DIR="${TRAWL_DIR:-$HOME/.config/trawl}"
LEADS_FILE="$TRAWL_DIR/leads.json"

if [ ! -f "$LEADS_FILE" ]; then
  echo "❌ No leads file. Run setup.sh first."
  exit 1
fi

ACTION="${1:-list}"
shift || true

case "$ACTION" in
  list)
    STATE_FILTER=""
    CAT_FILTER=""
    while [[ $# -gt 0 ]]; do
      case $1 in
        --state) STATE_FILTER="$2"; shift 2 ;;
        --category) CAT_FILTER="$2"; shift 2 ;;
        *) shift ;;
      esac
    done

    FILTER='.leads | to_entries[]'
    if [ -n "$STATE_FILTER" ]; then
      FILTER="$FILTER | select(.value.state == \"$STATE_FILTER\")"
    fi
    if [ -n "$CAT_FILTER" ]; then
      FILTER="$FILTER | select(.value.category == \"$CAT_FILTER\")"
    fi

    jq -r "[$FILTER] | sort_by(-.value.finalScore) | .[] | \"\(.value.state | .[0:12]) | \(.value.finalScore | tostring | .[0:4]) | \(.value.agentId) | \(.value.owner.name) | \(.value.category)\"" "$LEADS_FILE" | \
      column -t -s '|' 2>/dev/null || cat

    COUNT=$(jq "[$FILTER] | length" "$LEADS_FILE")
    echo ""
    echo "Total: $COUNT leads"
    ;;

  get)
    KEY="$1"
    jq --arg key "$KEY" '.leads[$key] // "Lead not found"' "$LEADS_FILE"
    ;;

  update)
    KEY="$1"; shift
    while [[ $# -gt 0 ]]; do
      case $1 in
        --state)
          NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
          jq --arg key "$KEY" --arg state "$2" --arg now "$NOW" \
            '.leads[$key].state = $state | .leads[$key].lastUpdated = $now' \
            "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"
          echo "✓ $KEY → $2"
          shift 2 ;;
        *) shift ;;
      esac
    done
    ;;

  decide)
    KEY="$1"; shift
    DECISION=""
    while [[ $# -gt 0 ]]; do
      case $1 in
        --pursue) DECISION="PURSUE"; shift ;;
        --pass) DECISION="PASS"; shift ;;
        *) shift ;;
      esac
    done

    if [ -z "$DECISION" ]; then
      echo "Usage: leads.sh decide <key> --pursue|--pass"
      exit 1
    fi

    NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
    CURRENT_STATE=$(jq -r --arg key "$KEY" '.leads[$key].state // ""' "$LEADS_FILE")

    if [ "$DECISION" = "PURSUE" ]; then
      if [ "$CURRENT_STATE" = "INBOUND_PENDING" ]; then
        # Approve inbound DM and move to QUALIFYING
        NEW_STATE="QUALIFYING"
        CONV_ID=$(jq -r --arg key "$KEY" '.leads[$key].conversationId // ""' "$LEADS_FILE")
        echo "✓ $KEY → PURSUE (approving inbound DM, moving to QUALIFYING)"

        # Approve via API if we have credentials (defensive: extract only required vars)
        SECRETS_FILE="$HOME/.clawdbot/secrets.env"
        if [ -f "$SECRETS_FILE" ]; then
          MOLTBOOK_API_KEY="${MOLTBOOK_API_KEY:-$(grep -E '^MOLTBOOK_API_KEY=' "$SECRETS_FILE" 2>/dev/null | cut -d'=' -f2- | tr -d '"'"'" || true)}"
          export MOLTBOOK_API_KEY
        fi
        CONFIG="$TRAWL_DIR/config.json"
        API_BASE=$(jq -r '.sources.moltbook.api_base' "$CONFIG" 2>/dev/null || echo "")
        API_KEY_ENV=$(jq -r '.sources.moltbook.api_key_env' "$CONFIG" 2>/dev/null || echo "MOLTBOOK_API_KEY")
        API_KEY="${!API_KEY_ENV:-}"

        if [ -n "$API_KEY" ] && [ -n "$CONV_ID" ] && [ "$CONV_ID" != "null" ]; then
          curl -s -f -X POST "$API_BASE/agents/dm/requests/$CONV_ID/approve" \
            -H "Authorization: Bearer $API_KEY" > /dev/null 2>&1 && echo "   ✓ DM request approved via API" || echo "   ⚠ API approve failed (will retry next qualify cycle)"
        fi
      elif [ "$CURRENT_STATE" = "QUALIFIED" ] || [ "$CURRENT_STATE" = "PROFILE_SCORED" ]; then
        NEW_STATE="REPORTED"
        echo "✓ $KEY → PURSUE"
      else
        NEW_STATE="$CURRENT_STATE"
        echo "✓ $KEY → PURSUE (state unchanged: $CURRENT_STATE)"
      fi
    else
      NEW_STATE="ARCHIVED"
      echo "✓ $KEY → PASS (archived)"
    fi

    jq --arg key "$KEY" --arg decision "$DECISION" --arg state "$NEW_STATE" --arg now "$NOW" \
      '.leads[$key].humanDecision = $decision | .leads[$key].state = $state | .leads[$key].lastUpdated = $now' \
      "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"
    ;;

  archive)
    KEY="$1"
    NOW=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
    jq --arg key "$KEY" --arg now "$NOW" \
      '.leads[$key].state = "ARCHIVED" | .leads[$key].lastUpdated = $now' \
      "$LEADS_FILE" > "$LEADS_FILE.tmp" && mv "$LEADS_FILE.tmp" "$LEADS_FILE"
    echo "✓ $KEY archived"
    ;;

  stats)
    echo "📊 Lead Stats"
    echo "━━━━━━━━━━━━━"
    jq -r '.leads | to_entries | group_by(.value.state) | .[] | "\(.[0].value.state): \(length)"' "$LEADS_FILE"
    echo ""
    echo "By category:"
    jq -r '.leads | to_entries | group_by(.value.category) | .[] | "  \(.[0].value.category): \(length)"' "$LEADS_FILE"
    echo ""
    TOTAL=$(jq '.leads | length' "$LEADS_FILE")
    echo "Total: $TOTAL"
    ;;

  reset)
    echo "⚠ This will clear ALL leads, seen posts, conversations, and sweep logs."
    echo '{"leads":{}}' > "$LEADS_FILE"
    echo '{"posts":{}}' > "$TRAWL_DIR/seen-posts.json" 2>/dev/null || true
    echo '{"conversations":{}}' > "$TRAWL_DIR/conversations.json" 2>/dev/null || true
    echo '{"sweeps":[]}' > "$TRAWL_DIR/sweep-log.json" 2>/dev/null || true
    rm -f "$TRAWL_DIR/last-sweep-report.json"
    echo "✓ All data reset"
    ;;

  *)
    echo "Usage: leads.sh {list|get|update|decide|archive|reset|stats} [options]"
    exit 1
    ;;
esac
