#!/usr/bin/env python3
"""
Whale Tracker
Monitor large trades and orderbook imbalances on Binance
"""

import json
import argparse
import sys
from urllib import request, error
from datetime import datetime
from statistics import mean

BASE_URL = "https://data-api.binance.vision"

def fetch_json(url):
    """Fetch JSON data from URL with error handling"""
    try:
        with request.urlopen(url, timeout=10) as response:
            return json.loads(response.read().decode())
    except error.HTTPError as e:
        print(f"HTTP Error {e.code}: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except error.URLError as e:
        print(f"Connection Error: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

def get_recent_trades(symbol, limit=1000):
    """Get recent trades"""
    url = f"{BASE_URL}/api/v3/trades?symbol={symbol}&limit={limit}"
    return fetch_json(url)

def get_orderbook(symbol, limit=1000):
    """Get orderbook depth"""
    url = f"{BASE_URL}/api/v3/depth?symbol={symbol}&limit={limit}"
    return fetch_json(url)

def get_ticker(symbol):
    """Get 24h ticker for reference"""
    url = f"{BASE_URL}/api/v3/ticker/24hr?symbol={symbol}"
    return fetch_json(url)

def analyze_large_trades(trades, threshold_percentile=90):
    """Identify large trades (whale activity)
    
    Args:
        trades: List of recent trades
        threshold_percentile: Percentile for defining "large" trades
    """
    if not trades:
        return []
    
    # Calculate trade sizes
    trade_sizes = [float(t['qty']) * float(t['price']) for t in trades]
    
    # Find threshold (top percentile)
    sorted_sizes = sorted(trade_sizes)
    threshold_idx = int(len(sorted_sizes) * (threshold_percentile / 100))
    threshold = sorted_sizes[threshold_idx] if threshold_idx < len(sorted_sizes) else sorted_sizes[-1]
    
    # Filter large trades
    large_trades = []
    for i, trade in enumerate(trades):
        size = trade_sizes[i]
        if size >= threshold:
            large_trades.append({
                'time': trade['time'],
                'price': float(trade['price']),
                'qty': float(trade['qty']),
                'value': size,
                'is_buyer_maker': trade['isBuyerMaker'],
                'side': 'SELL' if trade['isBuyerMaker'] else 'BUY'
            })
    
    return large_trades, threshold

def analyze_orderbook_imbalance(orderbook, depth=20):
    """Analyze orderbook for bid/ask imbalances and walls
    
    Args:
        orderbook: Orderbook data
        depth: Number of levels to analyze
    """
    bids = orderbook['bids'][:depth]
    asks = orderbook['asks'][:depth]
    
    # Calculate total bid/ask volume
    total_bid_volume = sum(float(b[1]) for b in bids)
    total_ask_volume = sum(float(a[1]) for a in asks)
    
    # Calculate bid/ask value
    total_bid_value = sum(float(b[0]) * float(b[1]) for b in bids)
    total_ask_value = sum(float(a[0]) * float(a[1]) for a in asks)
    
    # Calculate imbalance ratio
    if total_ask_volume > 0:
        volume_ratio = total_bid_volume / total_ask_volume
    else:
        volume_ratio = float('inf')
    
    if total_ask_value > 0:
        value_ratio = total_bid_value / total_ask_value
    else:
        value_ratio = float('inf')
    
    # Find walls (unusually large orders)
    bid_sizes = [float(b[1]) for b in bids]
    ask_sizes = [float(a[1]) for a in asks]
    
    avg_bid_size = mean(bid_sizes) if bid_sizes else 0
    avg_ask_size = mean(ask_sizes) if ask_sizes else 0
    
    # Wall threshold: 3x average
    bid_wall_threshold = avg_bid_size * 3
    ask_wall_threshold = avg_ask_size * 3
    
    bid_walls = [
        {'price': float(b[0]), 'size': float(b[1])}
        for b in bids if float(b[1]) >= bid_wall_threshold
    ]
    
    ask_walls = [
        {'price': float(a[0]), 'size': float(a[1])}
        for a in asks if float(a[1]) >= ask_wall_threshold
    ]
    
    return {
        'total_bid_volume': total_bid_volume,
        'total_ask_volume': total_ask_volume,
        'total_bid_value': total_bid_value,
        'total_ask_value': total_ask_value,
        'volume_ratio': volume_ratio,
        'value_ratio': value_ratio,
        'bid_walls': bid_walls,
        'ask_walls': ask_walls,
        'imbalance': 'BULLISH' if volume_ratio > 1.5 else 'BEARISH' if volume_ratio < 0.67 else 'NEUTRAL'
    }

def format_large_trades_output(large_trades, threshold):
    """Format large trades output"""
    print(f"\n{'='*90}")
    print(f"🐋 Large Trades (Whale Activity)")
    print(f"{'='*90}")
    print(f"Threshold: ${threshold:,.2f} (top 10% of recent trades)")
    print(f"Total Large Trades: {len(large_trades)}")
    
    if not large_trades:
        print("No large trades detected in recent activity")
        return
    
    # Aggregate by side
    buy_count = sum(1 for t in large_trades if t['side'] == 'BUY')
    sell_count = sum(1 for t in large_trades if t['side'] == 'SELL')
    buy_volume = sum(t['value'] for t in large_trades if t['side'] == 'BUY')
    sell_volume = sum(t['value'] for t in large_trades if t['side'] == 'SELL')
    
    print(f"\nSummary:")
    print(f"  Buy Orders: {buy_count} (${buy_volume:,.2f})")
    print(f"  Sell Orders: {sell_count} (${sell_volume:,.2f})")
    
    if buy_volume > sell_volume * 1.2:
        print(f"  ⚠ Whale buying pressure detected!")
    elif sell_volume > buy_volume * 1.2:
        print(f"  ⚠ Whale selling pressure detected!")
    
    print(f"\nRecent Large Trades (last 20):")
    print(f"{'Time':<20} {'Side':<6} {'Price':<15} {'Quantity':<15} {'Value':<15}")
    print("-" * 90)
    
    # Show most recent large trades
    sorted_trades = sorted(large_trades, key=lambda x: x['time'], reverse=True)
    for trade in sorted_trades[:20]:
        time_str = datetime.fromtimestamp(trade['time']/1000).strftime('%Y-%m-%d %H:%M:%S')
        side_str = trade['side']
        print(f"{time_str:<20} {side_str:<6} ${trade['price']:<14,.2f} {trade['qty']:<14,.4f} ${trade['value']:<14,.2f}")
    
    print()

def format_orderbook_output(analysis, current_price):
    """Format orderbook analysis output"""
    print(f"\n{'='*90}")
    print(f"📊 Orderbook Analysis")
    print(f"{'='*90}")
    print(f"Current Price: ${current_price:,.2f}")
    
    print(f"\nOrderbook Imbalance (top 20 levels):")
    print(f"  Total Bid Volume: {analysis['total_bid_volume']:,.4f}")
    print(f"  Total Ask Volume: {analysis['total_ask_volume']:,.4f}")
    print(f"  Volume Ratio (Bid/Ask): {analysis['volume_ratio']:.2f}")
    print(f"  Value Ratio (Bid/Ask): {analysis['value_ratio']:.2f}")
    print(f"  Market Sentiment: {analysis['imbalance']}")
    
    if analysis['imbalance'] == 'BULLISH':
        print(f"  ✓ More buy pressure - potential upward movement")
    elif analysis['imbalance'] == 'BEARISH':
        print(f"  ✗ More sell pressure - potential downward movement")
    
    print(f"\n🧱 Orderbook Walls:")
    
    if analysis['bid_walls']:
        print(f"\n  Support Walls (Large Bids):")
        print(f"  {'Price':<15} {'Size':<15} {'Distance from Current':<20}")
        print(f"  {'-'*50}")
        for wall in analysis['bid_walls'][:5]:
            distance = ((current_price - wall['price']) / current_price) * 100
            print(f"  ${wall['price']:<14,.2f} {wall['size']:<14,.4f} {distance:>18.2f}%")
    else:
        print(f"  No significant bid walls detected")
    
    if analysis['ask_walls']:
        print(f"\n  Resistance Walls (Large Asks):")
        print(f"  {'Price':<15} {'Size':<15} {'Distance from Current':<20}")
        print(f"  {'-'*50}")
        for wall in analysis['ask_walls'][:5]:
            distance = ((wall['price'] - current_price) / current_price) * 100
            print(f"  ${wall['price']:<14,.2f} {wall['size']:<14,.4f} {distance:>18.2f}%")
    else:
        print(f"  No significant ask walls detected")
    
    print()

def main():
    parser = argparse.ArgumentParser(description='Whale Tracker - Monitor large trades and orderbook imbalances')
    parser.add_argument('--symbol', '-s', default='BTCUSDT', help='Trading pair symbol (default: BTCUSDT)')
    parser.add_argument('--trades', '-t', action='store_true', help='Analyze large trades')
    parser.add_argument('--orderbook', '-o', action='store_true', help='Analyze orderbook imbalances')
    parser.add_argument('--depth', '-d', type=int, default=20, help='Orderbook depth to analyze (default: 20)')
    parser.add_argument('--threshold', type=int, default=90, help='Percentile threshold for large trades (default: 90)')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    
    args = parser.parse_args()
    
    symbol = args.symbol.upper()
    
    # If no specific analysis requested, do both
    if not args.trades and not args.orderbook:
        args.trades = True
        args.orderbook = True
    
    try:
        result = {'symbol': symbol, 'timestamp': datetime.now().isoformat()}
        
        # Get current price for reference
        ticker = get_ticker(symbol)
        current_price = float(ticker['lastPrice'])
        
        if args.trades:
            trades = get_recent_trades(symbol, limit=1000)
            large_trades, threshold = analyze_large_trades(trades, args.threshold)
            
            result['large_trades'] = {
                'trades': large_trades,
                'threshold': threshold,
                'count': len(large_trades)
            }
            
            if not args.json:
                format_large_trades_output(large_trades, threshold)
        
        if args.orderbook:
            orderbook = get_orderbook(symbol, limit=args.depth)
            analysis = analyze_orderbook_imbalance(orderbook, args.depth)
            
            result['orderbook'] = analysis
            
            if not args.json:
                format_orderbook_output(analysis, current_price)
        
        if args.json:
            print(json.dumps(result, indent=2))
        else:
            print(f"{'='*90}")
            print(f"Whale Tracking Summary for {symbol}")
            print(f"{'='*90}")
            print(f"Timestamp: {result['timestamp']}")
            print(f"Current Price: ${current_price:,.2f}")
            
            if args.trades:
                buy_trades = [t for t in result['large_trades']['trades'] if t['side'] == 'BUY']
                sell_trades = [t for t in result['large_trades']['trades'] if t['side'] == 'SELL']
                print(f"Large Trades: {len(buy_trades)} buys, {len(sell_trades)} sells")
            
            if args.orderbook:
                print(f"Orderbook Sentiment: {result['orderbook']['imbalance']}")
                print(f"Bid/Ask Ratio: {result['orderbook']['volume_ratio']:.2f}")
            
            print(f"{'='*90}\n")
            
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
