#!/usr/bin/env python3
"""
Technical Analysis Calculator
Calculate RSI, MACD, Bollinger Bands, EMA, SMA, volume analysis from kline data
Can accept JSON input from binance_market.py or fetch directly
"""

import json
import argparse
import sys
from urllib import request, error
from statistics import mean, stdev

BASE_URL = "https://data-api.binance.vision"

def fetch_klines(symbol, interval, limit):
    """Fetch klines from Binance"""
    url = f"{BASE_URL}/api/v3/klines?symbol={symbol}&interval={interval}&limit={limit}"
    try:
        with request.urlopen(url, timeout=10) as response:
            data = json.loads(response.read().decode())
            return [
                {
                    'open_time': k[0],
                    'open': float(k[1]),
                    'high': float(k[2]),
                    'low': float(k[3]),
                    'close': float(k[4]),
                    'volume': float(k[5]),
                    'close_time': k[6],
                    'quote_volume': float(k[7])
                }
                for k in data
            ]
    except error.HTTPError as e:
        print(f"HTTP Error {e.code}: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

def calculate_sma(prices, period):
    """Calculate Simple Moving Average"""
    if len(prices) < period:
        return None
    return mean(prices[-period:])

def calculate_ema(prices, period):
    """Calculate Exponential Moving Average"""
    if len(prices) < period:
        return None
    
    multiplier = 2 / (period + 1)
    ema = mean(prices[:period])  # Start with SMA
    
    for price in prices[period:]:
        ema = (price - ema) * multiplier + ema
    
    return ema

def calculate_rsi(prices, period=14):
    """Calculate Relative Strength Index"""
    if len(prices) < period + 1:
        return None
    
    gains = []
    losses = []
    
    for i in range(1, len(prices)):
        change = prices[i] - prices[i-1]
        if change > 0:
            gains.append(change)
            losses.append(0)
        else:
            gains.append(0)
            losses.append(abs(change))
    
    avg_gain = mean(gains[-period:])
    avg_loss = mean(losses[-period:])
    
    if avg_loss == 0:
        return 100
    
    rs = avg_gain / avg_loss
    rsi = 100 - (100 / (1 + rs))
    
    return rsi

def calculate_macd(prices, fast=12, slow=26, signal=9):
    """Calculate MACD (Moving Average Convergence Divergence)"""
    if len(prices) < slow:
        return None, None, None
    
    ema_fast = calculate_ema(prices, fast)
    ema_slow = calculate_ema(prices, slow)
    
    if ema_fast is None or ema_slow is None:
        return None, None, None
    
    macd_line = ema_fast - ema_slow
    
    # Calculate signal line (EMA of MACD)
    # Simplified: using last value as approximation
    signal_line = macd_line  # In production, would track MACD history
    histogram = macd_line - signal_line
    
    return macd_line, signal_line, histogram

def calculate_bollinger_bands(prices, period=20, std_dev=2):
    """Calculate Bollinger Bands"""
    if len(prices) < period:
        return None, None, None
    
    sma = calculate_sma(prices, period)
    std = stdev(prices[-period:])
    
    upper_band = sma + (std * std_dev)
    lower_band = sma - (std * std_dev)
    
    return upper_band, sma, lower_band

def find_support_resistance(klines, lookback=20):
    """Find support and resistance levels using pivot points"""
    if len(klines) < lookback:
        return [], []
    
    highs = [k['high'] for k in klines[-lookback:]]
    lows = [k['low'] for k in klines[-lookback:]]
    
    # Simple approach: use recent highs/lows
    resistance_levels = []
    support_levels = []
    
    # Find local maxima/minima
    for i in range(1, len(highs) - 1):
        if highs[i] > highs[i-1] and highs[i] > highs[i+1]:
            resistance_levels.append(highs[i])
        if lows[i] < lows[i-1] and lows[i] < lows[i+1]:
            support_levels.append(lows[i])
    
    # Add overall high/low
    resistance_levels.append(max(highs))
    support_levels.append(min(lows))
    
    return sorted(set(resistance_levels), reverse=True)[:3], sorted(set(support_levels))[:3]

def analyze_volume(klines, period=20):
    """Analyze volume patterns"""
    if len(klines) < period:
        return None
    
    volumes = [k['volume'] for k in klines[-period:]]
    avg_volume = mean(volumes)
    current_volume = klines[-1]['volume']
    
    volume_ratio = current_volume / avg_volume if avg_volume > 0 else 0
    
    return {
        'current': current_volume,
        'average': avg_volume,
        'ratio': volume_ratio,
        'status': 'High' if volume_ratio > 1.5 else 'Normal' if volume_ratio > 0.5 else 'Low'
    }

def get_trend_signal(prices):
    """Simple trend detection"""
    if len(prices) < 20:
        return "Insufficient data"
    
    sma_20 = calculate_sma(prices, 20)
    sma_50 = calculate_sma(prices, 50) if len(prices) >= 50 else None
    current_price = prices[-1]
    
    if sma_50:
        if current_price > sma_20 > sma_50:
            return "Strong Uptrend"
        elif current_price < sma_20 < sma_50:
            return "Strong Downtrend"
        elif current_price > sma_20:
            return "Uptrend"
        else:
            return "Downtrend"
    else:
        if current_price > sma_20:
            return "Uptrend"
        else:
            return "Downtrend"

def analyze(klines, rsi_period=14, bb_period=20, show_all=False):
    """Perform comprehensive technical analysis"""
    closes = [k['close'] for k in klines]
    current_price = closes[-1]
    
    # Calculate indicators
    sma_20 = calculate_sma(closes, 20)
    sma_50 = calculate_sma(closes, 50)
    ema_12 = calculate_ema(closes, 12)
    ema_26 = calculate_ema(closes, 26)
    
    rsi = calculate_rsi(closes, rsi_period)
    macd_line, signal_line, histogram = calculate_macd(closes)
    bb_upper, bb_middle, bb_lower = calculate_bollinger_bands(closes, bb_period)
    
    resistance, support = find_support_resistance(klines)
    volume_analysis = analyze_volume(klines)
    trend = get_trend_signal(closes)
    
    # Format output
    print(f"{'='*60}")
    print(f"Technical Analysis")
    print(f"{'='*60}")
    print(f"\nCurrent Price: ${current_price:,.2f}")
    print(f"Trend: {trend}")
    
    print(f"\n{'Moving Averages:':<30}")
    if sma_20:
        print(f"  SMA(20): ${sma_20:,.2f}")
    if sma_50:
        print(f"  SMA(50): ${sma_50:,.2f}")
    if ema_12:
        print(f"  EMA(12): ${ema_12:,.2f}")
    if ema_26:
        print(f"  EMA(26): ${ema_26:,.2f}")
    
    print(f"\n{'Momentum Indicators:':<30}")
    if rsi is not None:
        rsi_status = "Overbought" if rsi > 70 else "Oversold" if rsi < 30 else "Neutral"
        print(f"  RSI({rsi_period}): {rsi:.2f} ({rsi_status})")
    
    if macd_line is not None:
        macd_status = "Bullish" if macd_line > signal_line else "Bearish"
        print(f"  MACD: {macd_line:.2f} ({macd_status})")
        print(f"  Signal: {signal_line:.2f}")
        print(f"  Histogram: {histogram:.2f}")
    
    print(f"\n{'Bollinger Bands:':<30}")
    if bb_upper:
        bb_position = ((current_price - bb_lower) / (bb_upper - bb_lower)) * 100
        print(f"  Upper: ${bb_upper:,.2f}")
        print(f"  Middle: ${bb_middle:,.2f}")
        print(f"  Lower: ${bb_lower:,.2f}")
        print(f"  Position: {bb_position:.1f}% (0%=Lower, 100%=Upper)")
    
    print(f"\n{'Support/Resistance Levels:':<30}")
    if resistance:
        print(f"  Resistance: {', '.join([f'${r:,.2f}' for r in resistance])}")
    if support:
        print(f"  Support: {', '.join([f'${s:,.2f}' for s in support])}")
    
    print(f"\n{'Volume Analysis:':<30}")
    if volume_analysis:
        print(f"  Current: {volume_analysis['current']:,.2f}")
        print(f"  Average(20): {volume_analysis['average']:,.2f}")
        print(f"  Ratio: {volume_analysis['ratio']:.2f}x ({volume_analysis['status']})")
    
    # Trading signals
    print(f"\n{'='*60}")
    print(f"Trading Signals")
    print(f"{'='*60}")
    
    signals = []
    
    if rsi is not None:
        if rsi < 30:
            signals.append("⚠ RSI oversold - potential buy opportunity")
        elif rsi > 70:
            signals.append("⚠ RSI overbought - consider taking profits")
    
    if bb_upper and bb_lower:
        if current_price >= bb_upper:
            signals.append("⚠ Price at upper Bollinger Band - overbought")
        elif current_price <= bb_lower:
            signals.append("⚠ Price at lower Bollinger Band - oversold")
    
    if volume_analysis and volume_analysis['ratio'] > 2:
        signals.append("📊 Unusually high volume - significant interest")
    
    if macd_line and signal_line:
        if macd_line > 0 and signal_line > 0:
            signals.append("✓ MACD bullish crossover")
        elif macd_line < 0 and signal_line < 0:
            signals.append("✗ MACD bearish crossover")
    
    if signals:
        for signal in signals:
            print(f"  {signal}")
    else:
        print("  No strong signals detected")
    
    print(f"\n{'='*60}\n")
    
    # Return data for JSON output
    return {
        'price': current_price,
        'trend': trend,
        'sma_20': sma_20,
        'sma_50': sma_50,
        'ema_12': ema_12,
        'ema_26': ema_26,
        'rsi': rsi,
        'macd': {'line': macd_line, 'signal': signal_line, 'histogram': histogram},
        'bollinger_bands': {'upper': bb_upper, 'middle': bb_middle, 'lower': bb_lower},
        'support': support,
        'resistance': resistance,
        'volume': volume_analysis,
        'signals': signals
    }

def main():
    parser = argparse.ArgumentParser(description='Technical Analysis Calculator')
    parser.add_argument('--symbol', '-s', default='BTCUSDT', help='Trading pair symbol (default: BTCUSDT)')
    parser.add_argument('--interval', '-i', default='1h', help='Timeframe (1m,5m,15m,1h,4h,1d,1w) (default: 1h)')
    parser.add_argument('--limit', '-l', type=int, default=200, help='Number of candles to analyze (default: 200)')
    parser.add_argument('--rsi-period', type=int, default=14, help='RSI period (default: 14)')
    parser.add_argument('--bb-period', type=int, default=20, help='Bollinger Bands period (default: 20)')
    parser.add_argument('--input', '-f', help='Read klines from JSON file instead of API')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    
    args = parser.parse_args()
    
    try:
        if args.input:
            with open(args.input, 'r') as f:
                klines = json.load(f)
        else:
            klines = fetch_klines(args.symbol.upper(), args.interval, args.limit)
        
        if not klines:
            print("No data received", file=sys.stderr)
            sys.exit(1)
        
        result = analyze(klines, args.rsi_period, args.bb_period)
        
        if args.json:
            print(json.dumps(result, indent=2))
            
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)
    except FileNotFoundError:
        print(f"Error: File '{args.input}' not found", file=sys.stderr)
        sys.exit(1)
    except json.JSONDecodeError:
        print(f"Error: Invalid JSON in '{args.input}'", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
