#!/usr/bin/env python3
"""
Position Sizer
Calculate position size based on account balance, risk percentage, entry price, and stop loss
"""

import json
import argparse
import sys

def calculate_position_size(account_balance, risk_percent, entry_price, stop_loss_price):
    """Calculate position size using risk management rules
    
    Args:
        account_balance: Total account balance
        risk_percent: Percentage of account to risk (e.g., 1 = 1%)
        entry_price: Planned entry price
        stop_loss_price: Stop loss price
    
    Returns:
        Dictionary with position sizing details
    """
    if entry_price <= 0 or stop_loss_price <= 0:
        raise ValueError("Entry and stop loss prices must be positive")
    
    if account_balance <= 0:
        raise ValueError("Account balance must be positive")
    
    if risk_percent <= 0 or risk_percent > 100:
        raise ValueError("Risk percent must be between 0 and 100")
    
    # Calculate risk amount in dollars
    risk_amount = account_balance * (risk_percent / 100)
    
    # Calculate price difference (risk per unit)
    price_difference = abs(entry_price - stop_loss_price)
    
    if price_difference == 0:
        raise ValueError("Entry price and stop loss cannot be the same")
    
    # Calculate position size in units
    position_size_units = risk_amount / price_difference
    
    # Calculate position value in dollars
    position_value = position_size_units * entry_price
    
    # Calculate position as percentage of account
    position_percent = (position_value / account_balance) * 100
    
    # Calculate stop loss percentage
    stop_loss_percent = (price_difference / entry_price) * 100
    
    return {
        'account_balance': account_balance,
        'risk_percent': risk_percent,
        'risk_amount': risk_amount,
        'entry_price': entry_price,
        'stop_loss_price': stop_loss_price,
        'stop_loss_percent': stop_loss_percent,
        'position_size_units': position_size_units,
        'position_value': position_value,
        'position_percent': position_percent
    }

def calculate_take_profit_levels(entry_price, stop_loss_price, risk_reward_ratios=[1, 2, 3]):
    """Calculate take profit levels based on risk-reward ratios
    
    Args:
        entry_price: Entry price
        stop_loss_price: Stop loss price
        risk_reward_ratios: List of R:R ratios to calculate
    """
    risk = abs(entry_price - stop_loss_price)
    is_long = entry_price > stop_loss_price
    
    take_profit_levels = []
    
    for ratio in risk_reward_ratios:
        if is_long:
            tp_price = entry_price + (risk * ratio)
        else:
            tp_price = entry_price - (risk * ratio)
        
        profit_percent = ((tp_price - entry_price) / entry_price) * 100
        
        take_profit_levels.append({
            'ratio': ratio,
            'price': tp_price,
            'profit_percent': profit_percent
        })
    
    return take_profit_levels, is_long

def calculate_ladder_strategy(position_size, num_levels=3):
    """Calculate a ladder/scaling strategy for entries or exits
    
    Args:
        position_size: Total position size
        num_levels: Number of levels to split into
    """
    size_per_level = position_size / num_levels
    
    levels = []
    for i in range(num_levels):
        levels.append({
            'level': i + 1,
            'size': size_per_level,
            'cumulative': size_per_level * (i + 1),
            'percent': (size_per_level / position_size) * 100
        })
    
    return levels

def format_position_output(result, take_profit_levels, is_long):
    """Format position sizing output"""
    print(f"\n{'='*70}")
    print(f"Position Sizing Analysis")
    print(f"{'='*70}")
    
    print(f"\nAccount & Risk:")
    print(f"  Account Balance: ${result['account_balance']:,.2f}")
    print(f"  Risk Percentage: {result['risk_percent']:.2f}%")
    print(f"  Risk Amount: ${result['risk_amount']:,.2f}")
    
    print(f"\nPosition Details:")
    direction = "LONG" if is_long else "SHORT"
    print(f"  Direction: {direction}")
    print(f"  Entry Price: ${result['entry_price']:,.2f}")
    print(f"  Stop Loss: ${result['stop_loss_price']:,.2f} ({result['stop_loss_percent']:.2f}%)")
    
    print(f"\nPosition Size:")
    print(f"  Units to Buy: {result['position_size_units']:.6f}")
    print(f"  Position Value: ${result['position_value']:,.2f}")
    print(f"  Position % of Account: {result['position_percent']:.2f}%")
    
    print(f"\nTake Profit Levels:")
    print(f"  {'R:R':<6} {'Price':<15} {'Profit %':<12} {'Profit $':<15}")
    print(f"  {'-'*50}")
    
    for tp in take_profit_levels:
        profit_amount = (tp['price'] - result['entry_price']) * result['position_size_units']
        print(f"  {tp['ratio']:>3}:1  ${tp['price']:<14,.2f} {tp['profit_percent']:>10.2f}%  ${profit_amount:>13,.2f}")
    
    print(f"\n{'='*70}")
    print(f"Risk Management Summary")
    print(f"{'='*70}")
    print(f"Maximum Loss if Stop Hit: ${result['risk_amount']:,.2f} ({result['risk_percent']:.2f}%)")
    print(f"Recommended: Risk 1-2% per trade for conservative approach")
    
    if result['position_percent'] > 50:
        print(f"\n⚠ WARNING: Position is {result['position_percent']:.1f}% of account - consider reducing")
    elif result['position_percent'] > 25:
        print(f"\n⚠ CAUTION: Position is {result['position_percent']:.1f}% of account - monitor closely")
    else:
        print(f"\n✓ Position size is {result['position_percent']:.1f}% of account - reasonable allocation")
    
    print(f"\n{'='*70}\n")

def format_ladder_output(levels, price, label="Entry Ladder"):
    """Format ladder strategy output"""
    print(f"\n{label}:")
    print(f"  {'Level':<8} {'Size (Units)':<15} {'Cumulative':<15} {'Percent':<10}")
    print(f"  {'-'*50}")
    
    for level in levels:
        print(f"  {level['level']:<8} {level['size']:<15.6f} {level['cumulative']:<15.6f} {level['percent']:<10.1f}%")

def main():
    parser = argparse.ArgumentParser(description='Position Size Calculator')
    parser.add_argument('--balance', '-b', type=float, required=True, help='Account balance')
    parser.add_argument('--risk', '-r', type=float, default=1.0, help='Risk percentage (default: 1.0)')
    parser.add_argument('--entry', '-e', type=float, required=True, help='Entry price')
    parser.add_argument('--stop-loss', '-s', type=float, required=True, help='Stop loss price')
    parser.add_argument('--take-profit', '-t', nargs='+', type=float, default=[1, 2, 3],
                       help='Risk:reward ratios for take profit (default: 1 2 3)')
    parser.add_argument('--ladder', '-l', type=int, help='Number of levels for ladder strategy')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    
    args = parser.parse_args()
    
    try:
        # Calculate position size
        result = calculate_position_size(
            args.balance,
            args.risk,
            args.entry,
            args.stop_loss
        )
        
        # Calculate take profit levels
        take_profit_levels, is_long = calculate_take_profit_levels(
            args.entry,
            args.stop_loss,
            args.take_profit
        )
        
        if args.json:
            output = {
                'position': result,
                'take_profit_levels': take_profit_levels,
                'direction': 'LONG' if is_long else 'SHORT'
            }
            
            if args.ladder:
                output['ladder'] = calculate_ladder_strategy(result['position_size_units'], args.ladder)
            
            print(json.dumps(output, indent=2))
        else:
            format_position_output(result, take_profit_levels, is_long)
            
            if args.ladder:
                levels = calculate_ladder_strategy(result['position_size_units'], args.ladder)
                format_ladder_output(levels, args.entry)
                print(f"\nNote: Scale into position across {args.ladder} levels to reduce timing risk\n")
            
    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
