#!/usr/bin/env python3
"""
Market Scanner
Scan Binance for top gainers/losers, volume spikes, and unusual activity
"""

import json
import argparse
import sys
from urllib import request, error
from datetime import datetime

BASE_URL = "https://data-api.binance.vision"

def fetch_json(url):
    """Fetch JSON data from URL with error handling"""
    try:
        with request.urlopen(url, timeout=10) as response:
            return json.loads(response.read().decode())
    except error.HTTPError as e:
        print(f"HTTP Error {e.code}: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except error.URLError as e:
        print(f"Connection Error: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

def get_all_tickers():
    """Get 24h ticker data for all symbols"""
    url = f"{BASE_URL}/api/v3/ticker/24hr"
    return fetch_json(url)

def get_exchange_info():
    """Get exchange information including trading pairs"""
    url = f"{BASE_URL}/api/v3/exchangeInfo"
    return fetch_json(url)

def filter_usdt_pairs(tickers):
    """Filter for USDT trading pairs only"""
    return [t for t in tickers if t['symbol'].endswith('USDT') and t['symbol'] != 'USDT']

def filter_active_pairs(tickers, min_volume=100000):
    """Filter for pairs with significant volume"""
    return [t for t in tickers if float(t['quoteVolume']) >= min_volume]

def find_top_gainers(tickers, limit=10):
    """Find top gaining pairs by 24h percentage change"""
    sorted_tickers = sorted(tickers, key=lambda x: float(x['priceChangePercent']), reverse=True)
    return sorted_tickers[:limit]

def find_top_losers(tickers, limit=10):
    """Find top losing pairs by 24h percentage change"""
    sorted_tickers = sorted(tickers, key=lambda x: float(x['priceChangePercent']))
    return sorted_tickers[:limit]

def find_volume_spikes(tickers, limit=10):
    """Find pairs with highest 24h volume"""
    sorted_tickers = sorted(tickers, key=lambda x: float(x['quoteVolume']), reverse=True)
    return sorted_tickers[:limit]

def find_volatile_pairs(tickers, limit=10):
    """Find most volatile pairs (highest high-low spread)"""
    volatility_tickers = []
    
    for t in tickers:
        high = float(t['highPrice'])
        low = float(t['lowPrice'])
        if low > 0:
            volatility = ((high - low) / low) * 100
            t['volatility'] = volatility
            volatility_tickers.append(t)
    
    sorted_tickers = sorted(volatility_tickers, key=lambda x: x['volatility'], reverse=True)
    return sorted_tickers[:limit]

def find_breakout_candidates(tickers, limit=10):
    """Find potential breakout candidates (near 24h high with high volume)"""
    breakout_candidates = []
    
    for t in tickers:
        current = float(t['lastPrice'])
        high = float(t['highPrice'])
        volume = float(t['quoteVolume'])
        
        if high > 0:
            distance_from_high = ((high - current) / high) * 100
            
            # Near 24h high (within 2%) with significant volume
            if distance_from_high < 2 and volume > 500000:
                t['distance_from_high'] = distance_from_high
                breakout_candidates.append(t)
    
    sorted_candidates = sorted(breakout_candidates, key=lambda x: x['distance_from_high'])
    return sorted_candidates[:limit]

def format_ticker_table(tickers, title, show_volume=True):
    """Format ticker data as a table"""
    print(f"\n{'='*90}")
    print(f"{title}")
    print(f"{'='*90}")
    
    if show_volume:
        print(f"{'#':<4} {'Symbol':<12} {'Price':<15} {'24h Change':<15} {'Volume (USDT)':<20}")
    else:
        print(f"{'#':<4} {'Symbol':<12} {'Price':<15} {'24h Change':<15} {'High':<15} {'Low':<15}")
    
    print("-" * 90)
    
    for i, t in enumerate(tickers, 1):
        symbol = t['symbol']
        price = float(t['lastPrice'])
        change_pct = float(t['priceChangePercent'])
        
        # Color coding for terminal (green/red)
        change_str = f"{change_pct:+.2f}%"
        
        if show_volume:
            volume = float(t['quoteVolume'])
            print(f"{i:<4} {symbol:<12} ${price:<14,.4f} {change_str:<15} ${volume:<19,.0f}")
        else:
            high = float(t['highPrice'])
            low = float(t['lowPrice'])
            print(f"{i:<4} {symbol:<12} ${price:<14,.4f} {change_str:<15} ${high:<14,.4f} ${low:<14,.4f}")
    
    print()

def format_volatile_table(tickers, title):
    """Format volatile pairs table"""
    print(f"\n{'='*90}")
    print(f"{title}")
    print(f"{'='*90}")
    print(f"{'#':<4} {'Symbol':<12} {'Price':<15} {'24h Change':<15} {'Volatility':<15}")
    print("-" * 90)
    
    for i, t in enumerate(tickers, 1):
        symbol = t['symbol']
        price = float(t['lastPrice'])
        change_pct = float(t['priceChangePercent'])
        volatility = t.get('volatility', 0)
        
        print(f"{i:<4} {symbol:<12} ${price:<14,.4f} {change_pct:>+13.2f}%  {volatility:>13.2f}%")
    
    print()

def format_breakout_table(tickers, title):
    """Format breakout candidates table"""
    print(f"\n{'='*90}")
    print(f"{title}")
    print(f"{'='*90}")
    print(f"{'#':<4} {'Symbol':<12} {'Price':<15} {'24h High':<15} {'Distance':<15} {'Volume':<20}")
    print("-" * 90)
    
    for i, t in enumerate(tickers, 1):
        symbol = t['symbol']
        price = float(t['lastPrice'])
        high = float(t['highPrice'])
        distance = t.get('distance_from_high', 0)
        volume = float(t['quoteVolume'])
        
        print(f"{i:<4} {symbol:<12} ${price:<14,.4f} ${high:<14,.4f} {distance:>13.2f}%  ${volume:<19,.0f}")
    
    print()

def scan_market(min_volume=100000, limit=10):
    """Perform comprehensive market scan"""
    print(f"Scanning Binance market... (minimum volume: ${min_volume:,.0f})")
    
    # Fetch all tickers
    all_tickers = get_all_tickers()
    
    # Filter for USDT pairs with significant volume
    usdt_tickers = filter_usdt_pairs(all_tickers)
    active_tickers = filter_active_pairs(usdt_tickers, min_volume)
    
    print(f"Found {len(active_tickers)} active USDT pairs\n")
    
    # Perform different scans
    top_gainers = find_top_gainers(active_tickers, limit)
    top_losers = find_top_losers(active_tickers, limit)
    top_volume = find_volume_spikes(active_tickers, limit)
    volatile_pairs = find_volatile_pairs(active_tickers, limit)
    breakout_candidates = find_breakout_candidates(active_tickers, limit)
    
    return {
        'top_gainers': top_gainers,
        'top_losers': top_losers,
        'top_volume': top_volume,
        'volatile_pairs': volatile_pairs,
        'breakout_candidates': breakout_candidates,
        'total_pairs': len(active_tickers),
        'timestamp': datetime.now().isoformat()
    }

def main():
    parser = argparse.ArgumentParser(description='Binance Market Scanner')
    parser.add_argument('--min-volume', '-v', type=float, default=100000,
                       help='Minimum 24h volume in USDT (default: 100000)')
    parser.add_argument('--limit', '-l', type=int, default=10,
                       help='Number of results per category (default: 10)')
    parser.add_argument('--gainers', '-g', action='store_true', help='Show only top gainers')
    parser.add_argument('--losers', '-L', action='store_true', help='Show only top losers')
    parser.add_argument('--volume', '-V', action='store_true', help='Show only top volume')
    parser.add_argument('--volatile', '-o', action='store_true', help='Show only volatile pairs')
    parser.add_argument('--breakout', '-b', action='store_true', help='Show only breakout candidates')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    
    args = parser.parse_args()
    
    try:
        results = scan_market(args.min_volume, args.limit)
        
        if args.json:
            print(json.dumps(results, indent=2))
        else:
            # Show all categories if no specific flag
            show_all = not any([args.gainers, args.losers, args.volume, args.volatile, args.breakout])
            
            if show_all or args.gainers:
                format_ticker_table(results['top_gainers'], f"🚀 Top {args.limit} Gainers (24h)")
            
            if show_all or args.losers:
                format_ticker_table(results['top_losers'], f"📉 Top {args.limit} Losers (24h)")
            
            if show_all or args.volume:
                format_ticker_table(results['top_volume'], f"💰 Highest Volume (24h)")
            
            if show_all or args.volatile:
                format_volatile_table(results['volatile_pairs'], f"📊 Most Volatile Pairs (24h)")
            
            if show_all or args.breakout:
                format_breakout_table(results['breakout_candidates'], f"🎯 Potential Breakouts (Near 24h High)")
            
            print(f"{'='*90}")
            print(f"Market Scan Summary")
            print(f"{'='*90}")
            print(f"Total Active Pairs: {results['total_pairs']}")
            print(f"Timestamp: {results['timestamp']}")
            print(f"{'='*90}\n")
            
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
