#!/usr/bin/env python3
"""
DCA (Dollar Cost Averaging) Calculator
Plan DCA schedule, amount distribution, and cost averaging projections
"""

import json
import argparse
import sys
from datetime import datetime, timedelta

def calculate_dca_schedule(total_amount, frequency, duration_days, start_date=None):
    """Calculate DCA purchase schedule
    
    Args:
        total_amount: Total amount to invest
        frequency: 'daily', 'weekly', 'biweekly', 'monthly'
        duration_days: Total duration in days
        start_date: Start date (datetime object or None for today)
    """
    if start_date is None:
        start_date = datetime.now()
    
    # Determine interval days
    intervals = {
        'daily': 1,
        'weekly': 7,
        'biweekly': 14,
        'monthly': 30
    }
    
    interval_days = intervals.get(frequency.lower())
    if not interval_days:
        raise ValueError(f"Invalid frequency: {frequency}. Use: daily, weekly, biweekly, monthly")
    
    # Calculate number of purchases
    num_purchases = duration_days // interval_days
    if num_purchases == 0:
        num_purchases = 1
    
    amount_per_purchase = total_amount / num_purchases
    
    # Generate schedule
    schedule = []
    current_date = start_date
    
    for i in range(num_purchases):
        schedule.append({
            'purchase_number': i + 1,
            'date': current_date.strftime('%Y-%m-%d'),
            'amount': amount_per_purchase,
            'cumulative': amount_per_purchase * (i + 1)
        })
        current_date += timedelta(days=interval_days)
    
    return schedule, num_purchases, amount_per_purchase

def simulate_dca_returns(schedule, price_history):
    """Simulate DCA strategy with historical prices
    
    Args:
        schedule: List of purchase dates and amounts
        price_history: Dict of {date: price}
    """
    total_invested = 0
    total_units = 0
    purchases = []
    
    for entry in schedule:
        date = entry['date']
        amount = entry['amount']
        
        # Find price for this date (or closest available)
        price = price_history.get(date)
        if price is None:
            # Try to find closest date
            available_dates = sorted(price_history.keys())
            for d in available_dates:
                if d >= date:
                    price = price_history[d]
                    break
        
        if price is None:
            continue
        
        units = amount / price
        total_invested += amount
        total_units += units
        
        purchases.append({
            'date': date,
            'price': price,
            'amount': amount,
            'units': units,
            'total_units': total_units,
            'avg_price': total_invested / total_units
        })
    
    return purchases, total_invested, total_units

def calculate_lump_sum_comparison(total_amount, start_price, end_price):
    """Compare DCA to lump sum investment"""
    lump_sum_units = total_amount / start_price
    lump_sum_value = lump_sum_units * end_price
    lump_sum_return = ((lump_sum_value - total_amount) / total_amount) * 100
    
    return {
        'units': lump_sum_units,
        'final_value': lump_sum_value,
        'return_pct': lump_sum_return
    }

def format_schedule_output(schedule, num_purchases, amount_per_purchase, total_amount):
    """Format DCA schedule for display"""
    print(f"\n{'='*70}")
    print(f"DCA Investment Schedule")
    print(f"{'='*70}")
    print(f"Total Investment: ${total_amount:,.2f}")
    print(f"Number of Purchases: {num_purchases}")
    print(f"Amount per Purchase: ${amount_per_purchase:,.2f}")
    print(f"First Purchase: {schedule[0]['date']}")
    print(f"Last Purchase: {schedule[-1]['date']}")
    
    print(f"\n{'#':<5} {'Date':<12} {'Amount':<15} {'Cumulative':<15}")
    print("-" * 70)
    
    for entry in schedule[:10]:  # Show first 10
        print(f"{entry['purchase_number']:<5} {entry['date']:<12} ${entry['amount']:<14,.2f} ${entry['cumulative']:<14,.2f}")
    
    if len(schedule) > 10:
        print(f"... ({len(schedule) - 10} more purchases)")
        last = schedule[-1]
        print(f"{last['purchase_number']:<5} {last['date']:<12} ${last['amount']:<14,.2f} ${last['cumulative']:<14,.2f}")
    
    print()

def format_simulation_output(purchases, total_invested, total_units, current_price=None):
    """Format DCA simulation results"""
    if not purchases:
        print("No simulation data available")
        return
    
    avg_price = total_invested / total_units if total_units > 0 else 0
    
    print(f"\n{'='*70}")
    print(f"DCA Simulation Results")
    print(f"{'='*70}")
    print(f"Total Invested: ${total_invested:,.2f}")
    print(f"Total Units Acquired: {total_units:.6f}")
    print(f"Average Purchase Price: ${avg_price:,.2f}")
    
    if current_price:
        current_value = total_units * current_price
        profit_loss = current_value - total_invested
        roi = (profit_loss / total_invested) * 100 if total_invested > 0 else 0
        
        print(f"Current Price: ${current_price:,.2f}")
        print(f"Current Value: ${current_value:,.2f}")
        print(f"Profit/Loss: ${profit_loss:,.2f} ({roi:+.2f}%)")
    
    print(f"\n{'Purchase History:'}")
    print(f"{'Date':<12} {'Price':<12} {'Amount':<12} {'Units':<12} {'Avg Price':<12}")
    print("-" * 70)
    
    for p in purchases[:10]:
        print(f"{p['date']:<12} ${p['price']:<11,.2f} ${p['amount']:<11,.2f} {p['units']:<11.6f} ${p['avg_price']:<11,.2f}")
    
    if len(purchases) > 10:
        print(f"... ({len(purchases) - 10} more purchases)")
        last = purchases[-1]
        print(f"{last['date']:<12} ${last['price']:<11,.2f} ${last['amount']:<11,.2f} {last['units']:<11.6f} ${last['avg_price']:<11,.2f}")
    
    print()

def simple_projection(amount_per_purchase, num_purchases, current_price, price_scenarios):
    """Project DCA outcomes under different price scenarios"""
    print(f"\n{'='*70}")
    print(f"DCA Price Scenario Analysis")
    print(f"{'='*70}")
    print(f"Amount per Purchase: ${amount_per_purchase:,.2f}")
    print(f"Number of Purchases: {num_purchases}")
    print(f"Current Price: ${current_price:,.2f}")
    
    print(f"\n{'Scenario':<20} {'Avg Price':<15} {'Units':<15} {'Final Value':<15} {'ROI':<10}")
    print("-" * 70)
    
    total_invested = amount_per_purchase * num_purchases
    
    for scenario_name, avg_price in price_scenarios.items():
        total_units = total_invested / avg_price
        final_value = total_units * current_price
        roi = ((final_value - total_invested) / total_invested) * 100
        
        print(f"{scenario_name:<20} ${avg_price:<14,.2f} {total_units:<14.6f} ${final_value:<14,.2f} {roi:>9.2f}%")
    
    print()

def main():
    parser = argparse.ArgumentParser(description='DCA (Dollar Cost Averaging) Calculator')
    parser.add_argument('--total', '-t', type=float, required=True, help='Total amount to invest')
    parser.add_argument('--frequency', '-f', default='weekly', 
                       choices=['daily', 'weekly', 'biweekly', 'monthly'],
                       help='Purchase frequency (default: weekly)')
    parser.add_argument('--duration', '-d', type=int, required=True, 
                       help='Investment duration in days')
    parser.add_argument('--start-date', '-s', help='Start date (YYYY-MM-DD, default: today)')
    parser.add_argument('--current-price', '-p', type=float, 
                       help='Current asset price for projections')
    parser.add_argument('--scenarios', '-c', action='store_true',
                       help='Show price scenario analysis')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    
    args = parser.parse_args()
    
    try:
        # Parse start date
        start_date = None
        if args.start_date:
            start_date = datetime.strptime(args.start_date, '%Y-%m-%d')
        
        # Calculate schedule
        schedule, num_purchases, amount_per_purchase = calculate_dca_schedule(
            args.total, args.frequency, args.duration, start_date
        )
        
        if args.json:
            output = {
                'total_amount': args.total,
                'frequency': args.frequency,
                'duration_days': args.duration,
                'num_purchases': num_purchases,
                'amount_per_purchase': amount_per_purchase,
                'schedule': schedule
            }
            print(json.dumps(output, indent=2))
        else:
            format_schedule_output(schedule, num_purchases, amount_per_purchase, args.total)
            
            # Show scenarios if current price provided
            if args.current_price and args.scenarios:
                scenarios = {
                    'Flat market': args.current_price,
                    'Bull market (+30%)': args.current_price * 0.85,  # Lower avg price in bull
                    'Bear market (-30%)': args.current_price * 1.15,  # Higher avg price in bear
                    'Volatile (±20%)': args.current_price,
                }
                simple_projection(amount_per_purchase, num_purchases, args.current_price, scenarios)
            
            # Summary
            print(f"{'='*70}")
            print(f"Summary")
            print(f"{'='*70}")
            print(f"This DCA strategy spreads ${args.total:,.2f} over {num_purchases} purchases")
            print(f"Each {args.frequency} purchase: ${amount_per_purchase:,.2f}")
            print(f"Duration: {args.duration} days (~{args.duration/30:.1f} months)")
            
            if args.current_price:
                # Show what immediate purchase would get
                immediate_units = args.total / args.current_price
                print(f"\nImmediate purchase at ${args.current_price:,.2f}: {immediate_units:.6f} units")
                print(f"DCA benefit: Reduces timing risk and emotional decision-making")
            
            print(f"\n{'='*70}\n")
            
    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
