#!/usr/bin/env python3
"""
Binance Market Data Fetcher
Fetch price, orderbook, 24h stats, funding rates, and klines from Binance public API
"""

import json
import argparse
import sys
from urllib import request, error
from urllib.parse import urlencode

BASE_URL = "https://data-api.binance.vision"
FAPI_URL = "https://fapi.binance.com"

def fetch_json(url):
    """Fetch JSON data from URL with error handling"""
    try:
        with request.urlopen(url, timeout=10) as response:
            return json.loads(response.read().decode())
    except error.HTTPError as e:
        print(f"HTTP Error {e.code}: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except error.URLError as e:
        print(f"Connection Error: {e.reason}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

def get_price(symbol):
    """Get current price for a symbol"""
    url = f"{BASE_URL}/api/v3/ticker/price?symbol={symbol}"
    data = fetch_json(url)
    return data

def get_ticker_24h(symbol):
    """Get 24h ticker statistics"""
    url = f"{BASE_URL}/api/v3/ticker/24hr?symbol={symbol}"
    data = fetch_json(url)
    return data

def get_orderbook(symbol, limit=20):
    """Get orderbook depth"""
    url = f"{BASE_URL}/api/v3/depth?symbol={symbol}&limit={limit}"
    data = fetch_json(url)
    return data

def get_recent_trades(symbol, limit=100):
    """Get recent trades"""
    url = f"{BASE_URL}/api/v3/trades?symbol={symbol}&limit={limit}"
    data = fetch_json(url)
    return data

def get_klines(symbol, interval, limit=100):
    """Get candlestick data (klines)
    
    Intervals: 1m, 3m, 5m, 15m, 30m, 1h, 2h, 4h, 6h, 8h, 12h, 1d, 3d, 1w, 1M
    """
    url = f"{BASE_URL}/api/v3/klines?symbol={symbol}&interval={interval}&limit={limit}"
    data = fetch_json(url)
    
    # Format klines for readability
    formatted = []
    for k in data:
        formatted.append({
            'open_time': k[0],
            'open': float(k[1]),
            'high': float(k[2]),
            'low': float(k[3]),
            'close': float(k[4]),
            'volume': float(k[5]),
            'close_time': k[6],
            'quote_volume': float(k[7]),
            'trades': k[8],
            'taker_buy_base': float(k[9]),
            'taker_buy_quote': float(k[10])
        })
    return formatted

def get_funding_rate(symbol):
    """Get current funding rate (futures only)"""
    try:
        url = f"{FAPI_URL}/fapi/v1/premiumIndex?symbol={symbol}"
        data = fetch_json(url)
        return data
    except:
        return {"error": "Funding rate only available for futures symbols"}

def format_price_output(data):
    """Format price data for display"""
    print(f"Symbol: {data['symbol']}")
    print(f"Price: ${float(data['price']):,.2f}")

def format_ticker_output(data):
    """Format 24h ticker data for display"""
    price_change = float(data['priceChange'])
    price_change_pct = float(data['priceChangePercent'])
    
    print(f"Symbol: {data['symbol']}")
    print(f"Current Price: ${float(data['lastPrice']):,.2f}")
    print(f"24h Change: ${price_change:,.2f} ({price_change_pct:+.2f}%)")
    print(f"24h High: ${float(data['highPrice']):,.2f}")
    print(f"24h Low: ${float(data['lowPrice']):,.2f}")
    print(f"24h Volume: {float(data['volume']):,.2f} {data['symbol'][:-4]}")
    print(f"24h Quote Volume: ${float(data['quoteVolume']):,.2f}")
    print(f"Trades: {data['count']:,}")

def format_orderbook_output(data, depth=10):
    """Format orderbook data for display"""
    print(f"\nOrderbook (Top {depth} levels):")
    print(f"\n{'Price':<15} {'Bids':<15} {'Asks':<15}")
    print("-" * 45)
    
    bids = data['bids'][:depth]
    asks = data['asks'][:depth]
    max_len = max(len(bids), len(asks))
    
    for i in range(max_len):
        bid_price = f"${float(bids[i][0]):,.2f}" if i < len(bids) else ""
        bid_qty = f"{float(bids[i][1]):,.4f}" if i < len(bids) else ""
        ask_price = f"${float(asks[i][0]):,.2f}" if i < len(asks) else ""
        ask_qty = f"{float(asks[i][1]):,.4f}" if i < len(asks) else ""
        
        print(f"{bid_price:<15} {bid_qty:<15} {ask_qty:<15}")

def format_trades_output(data, limit=10):
    """Format recent trades for display"""
    print(f"\nRecent Trades (Last {limit}):")
    print(f"{'Price':<15} {'Quantity':<15} {'Time':<20} {'Buyer Maker'}")
    print("-" * 70)
    
    for trade in data[-limit:]:
        price = f"${float(trade['price']):,.2f}"
        qty = f"{float(trade['qty']):,.4f}"
        from datetime import datetime
        time_str = datetime.fromtimestamp(trade['time']/1000).strftime('%Y-%m-%d %H:%M:%S')
        buyer_maker = "Sell" if trade['isBuyerMaker'] else "Buy"
        
        print(f"{price:<15} {qty:<15} {time_str:<20} {buyer_maker}")

def format_klines_output(data, limit=10):
    """Format klines for display"""
    print(f"\nCandlestick Data (Last {limit}):")
    print(f"{'Open Time':<20} {'Open':<12} {'High':<12} {'Low':<12} {'Close':<12} {'Volume':<15}")
    print("-" * 95)
    
    from datetime import datetime
    for k in data[-limit:]:
        time_str = datetime.fromtimestamp(k['open_time']/1000).strftime('%Y-%m-%d %H:%M')
        print(f"{time_str:<20} ${k['open']:<11,.2f} ${k['high']:<11,.2f} ${k['low']:<11,.2f} ${k['close']:<11,.2f} {k['volume']:<14,.2f}")

def format_funding_output(data):
    """Format funding rate for display"""
    if 'error' in data:
        print(f"Error: {data['error']}")
        return
    
    print(f"Symbol: {data['symbol']}")
    print(f"Mark Price: ${float(data['markPrice']):,.2f}")
    print(f"Index Price: ${float(data['indexPrice']):,.2f}")
    print(f"Funding Rate: {float(data['lastFundingRate']) * 100:.4f}%")
    print(f"Next Funding Time: {data['nextFundingTime']}")

def main():
    parser = argparse.ArgumentParser(description='Fetch Binance market data')
    parser.add_argument('--symbol', '-s', default='BTCUSDT', help='Trading pair symbol (default: BTCUSDT)')
    parser.add_argument('--price', '-p', action='store_true', help='Get current price')
    parser.add_argument('--ticker', '-t', action='store_true', help='Get 24h ticker stats')
    parser.add_argument('--orderbook', '-o', action='store_true', help='Get orderbook depth')
    parser.add_argument('--trades', '-r', action='store_true', help='Get recent trades')
    parser.add_argument('--klines', '-k', metavar='INTERVAL', help='Get klines (1m,5m,15m,1h,4h,1d,1w)')
    parser.add_argument('--funding', '-f', action='store_true', help='Get funding rate (futures)')
    parser.add_argument('--limit', '-l', type=int, default=100, help='Limit for klines/trades (default: 100)')
    parser.add_argument('--depth', '-d', type=int, default=10, help='Orderbook depth to display (default: 10)')
    parser.add_argument('--json', '-j', action='store_true', help='Output raw JSON')
    parser.add_argument('--all', '-a', action='store_true', help='Fetch all data types')
    
    args = parser.parse_args()
    
    # If no specific flag, show price and ticker
    if not any([args.price, args.ticker, args.orderbook, args.trades, args.klines, args.funding, args.all]):
        args.price = True
        args.ticker = True
    
    symbol = args.symbol.upper()
    
    try:
        if args.all or args.price:
            data = get_price(symbol)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_price_output(data)
                print()
        
        if args.all or args.ticker:
            data = get_ticker_24h(symbol)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_ticker_output(data)
                print()
        
        if args.all or args.orderbook:
            data = get_orderbook(symbol, limit=20)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_orderbook_output(data, args.depth)
                print()
        
        if args.all or args.trades:
            data = get_recent_trades(symbol, limit=args.limit)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_trades_output(data, min(10, len(data)))
                print()
        
        if args.klines:
            data = get_klines(symbol, args.klines, limit=args.limit)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_klines_output(data, min(10, len(data)))
                print()
        
        if args.all or args.funding:
            data = get_funding_rate(symbol)
            if args.json:
                print(json.dumps(data, indent=2))
            else:
                format_funding_output(data)
                print()
                
    except KeyboardInterrupt:
        print("\nInterrupted", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
