# Binance API Quick Reference

## Base URLs

- **Spot REST API**: `https://api.binance.com`
- **Futures REST API**: `https://fapi.binance.com`
- **WebSocket Streams**: `wss://stream.binance.com:9443`

## Rate Limits

- **Request Weight**: 1200 per minute per IP
- **Orders**: 10 orders per second per account
- **RAW_REQUESTS**: 5000 per 5 minutes per IP

Each endpoint has a weight. Stay under limits to avoid bans.

## Public Endpoints (No Authentication)

### Market Data

#### Get Current Price
```
GET /api/v3/ticker/price?symbol=BTCUSDT
```
Weight: 1 (single symbol), 2 (all symbols)

#### Get 24h Ticker Statistics
```
GET /api/v3/ticker/24hr?symbol=BTCUSDT
```
Weight: 1 (single symbol), 40 (all symbols)

Returns: price change, high, low, volume, number of trades

#### Get Orderbook Depth
```
GET /api/v3/depth?symbol=BTCUSDT&limit=100
```
Weight: Adjusted based on limit (5-50)
Limits: 5, 10, 20, 50, 100, 500, 1000, 5000

#### Get Recent Trades
```
GET /api/v3/trades?symbol=BTCUSDT&limit=500
```
Weight: 1
Max limit: 1000

#### Get Kline/Candlestick Data
```
GET /api/v3/klines?symbol=BTCUSDT&interval=1h&limit=100
```
Weight: 1

**Intervals**: 1m, 3m, 5m, 15m, 30m, 1h, 2h, 4h, 6h, 8h, 12h, 1d, 3d, 1w, 1M

**Response Format**:
```json
[
  [
    1499040000000,      // Open time
    "0.01634000",       // Open
    "0.80000000",       // High
    "0.01575800",       // Low
    "0.01577100",       // Close
    "148976.11427815",  // Volume
    1499644799999,      // Close time
    "2434.19055334",    // Quote asset volume
    308,                // Number of trades
    "1756.87402397",    // Taker buy base asset volume
    "28.46694368",      // Taker buy quote asset volume
    "17928899.62484339" // Ignore
  ]
]
```

#### Get Exchange Info
```
GET /api/v3/exchangeInfo
```
Weight: 10

Returns all trading pairs, trading rules, filters, etc.

#### Get Average Price
```
GET /api/v3/avgPrice?symbol=BTCUSDT
```
Weight: 1

### Futures-Specific Endpoints

#### Get Funding Rate
```
GET /fapi/v1/premiumIndex?symbol=BTCUSDT
```
Weight: 1

#### Get Open Interest
```
GET /fapi/v1/openInterest?symbol=BTCUSDT
```
Weight: 1

## Signed Endpoints (Authentication Required)

### Authentication

Signed endpoints require:
1. **API Key** in header: `X-MBX-APIKEY`
2. **Signature** in query string: `signature=<HMAC SHA256>`
3. **Timestamp** in query string: `timestamp=<milliseconds>`

### Creating a Signature

```python
import hmac
import hashlib
import time

secret_key = "YOUR_SECRET_KEY"
query_string = "symbol=BTCUSDT&side=BUY&type=LIMIT&quantity=1&price=50000&timestamp=1234567890"

signature = hmac.new(
    secret_key.encode(),
    query_string.encode(),
    hashlib.sha256
).hexdigest()
```

### Account Endpoints

#### Get Account Information
```
GET /api/v3/account?timestamp=<timestamp>&signature=<signature>
```
Weight: 10

Returns balances, permissions, trading status

#### Get Order
```
GET /api/v3/order?symbol=BTCUSDT&orderId=123&timestamp=<timestamp>&signature=<signature>
```
Weight: 2

#### Query All Orders
```
GET /api/v3/allOrders?symbol=BTCUSDT&timestamp=<timestamp>&signature=<signature>
```
Weight: 10

### Trading Endpoints

#### Place New Order
```
POST /api/v3/order
```
Weight: 1

**Required Parameters**:
- `symbol` (e.g., BTCUSDT)
- `side` (BUY or SELL)
- `type` (LIMIT, MARKET, STOP_LOSS_LIMIT, etc.)
- `quantity`
- `timestamp`
- `signature`

**Additional for LIMIT orders**:
- `timeInForce` (GTC, IOC, FOK)
- `price`

**Example**:
```
symbol=BTCUSDT&side=BUY&type=LIMIT&timeInForce=GTC&quantity=0.01&price=50000&timestamp=1234567890&signature=xxx
```

#### Cancel Order
```
DELETE /api/v3/order
```
Weight: 1

**Required Parameters**:
- `symbol`
- `orderId` or `origClientOrderId`
- `timestamp`
- `signature`

#### Cancel All Open Orders
```
DELETE /api/v3/openOrders?symbol=BTCUSDT&timestamp=<timestamp>&signature=<signature>
```
Weight: 1

## Order Types

- **LIMIT**: Buy/sell at specific price or better
- **MARKET**: Execute immediately at current market price
- **STOP_LOSS**: Market order triggered when stop price reached
- **STOP_LOSS_LIMIT**: Limit order triggered when stop price reached
- **TAKE_PROFIT**: Market order triggered when take profit reached
- **TAKE_PROFIT_LIMIT**: Limit order triggered when take profit reached
- **LIMIT_MAKER**: Limit order that will be rejected if would execute immediately (post-only)

## Time in Force

- **GTC** (Good Till Canceled): Order stays active until filled or canceled
- **IOC** (Immediate or Cancel): Fill what you can immediately, cancel rest
- **FOK** (Fill or Kill): Either fill entire order immediately or cancel

## Error Codes

- **-1000**: Unknown error
- **-1001**: Disconnected
- **-1003**: Too many requests (rate limit exceeded)
- **-1021**: Timestamp outside recv window
- **-1100**: Illegal characters in parameter
- **-2010**: Insufficient balance
- **-2011**: Order would trigger immediately (LIMIT_MAKER)

## Best Practices

1. **Use testnet first**: `https://testnet.binance.vision` for spot
2. **Implement rate limiting**: Track your request weight
3. **Handle errors gracefully**: Retry with exponential backoff
4. **Sync time**: Ensure system clock is accurate (within 1 second)
5. **Use WebSockets for real-time data**: More efficient than polling REST
6. **Keep API keys secure**: Never commit to git, use environment variables
7. **Use IP whitelisting**: Restrict API key to specific IPs
8. **Set trading restrictions**: Limit withdrawal permissions if not needed

## WebSocket Streams

### Individual Symbol Ticker
```
wss://stream.binance.com:9443/ws/btcusdt@ticker
```

### All Market Tickers
```
wss://stream.binance.com:9443/ws/!ticker@arr
```

### Kline/Candlestick Streams
```
wss://stream.binance.com:9443/ws/btcusdt@kline_1h
```

### Orderbook Depth
```
wss://stream.binance.com:9443/ws/btcusdt@depth
```

### Aggregate Trade Streams
```
wss://stream.binance.com:9443/ws/btcusdt@aggTrade
```

## Python Example (Using requests)

```python
import requests
import hmac
import hashlib
import time

API_KEY = "your_api_key"
SECRET_KEY = "your_secret_key"
BASE_URL = "https://api.binance.com"

# Public endpoint (no auth)
def get_price(symbol):
    url = f"{BASE_URL}/api/v3/ticker/price"
    params = {"symbol": symbol}
    response = requests.get(url, params=params)
    return response.json()

# Signed endpoint (with auth)
def get_account():
    endpoint = "/api/v3/account"
    timestamp = int(time.time() * 1000)
    
    query_string = f"timestamp={timestamp}"
    signature = hmac.new(
        SECRET_KEY.encode(),
        query_string.encode(),
        hashlib.sha256
    ).hexdigest()
    
    headers = {"X-MBX-APIKEY": API_KEY}
    url = f"{BASE_URL}{endpoint}?{query_string}&signature={signature}"
    
    response = requests.get(url, headers=headers)
    return response.json()

# Place order
def place_order(symbol, side, order_type, quantity, price=None):
    endpoint = "/api/v3/order"
    timestamp = int(time.time() * 1000)
    
    params = {
        "symbol": symbol,
        "side": side,
        "type": order_type,
        "quantity": quantity,
        "timestamp": timestamp
    }
    
    if order_type == "LIMIT":
        params["timeInForce"] = "GTC"
        params["price"] = price
    
    query_string = "&".join([f"{k}={v}" for k, v in params.items()])
    signature = hmac.new(
        SECRET_KEY.encode(),
        query_string.encode(),
        hashlib.sha256
    ).hexdigest()
    
    headers = {"X-MBX-APIKEY": API_KEY}
    url = f"{BASE_URL}{endpoint}?{query_string}&signature={signature}"
    
    response = requests.post(url, headers=headers)
    return response.json()
```

## Resources

- **Official Docs**: https://binance-docs.github.io/apidocs/spot/en/
- **API Status**: https://www.binance.com/en/support/announcement
- **Testnet**: https://testnet.binance.vision/
- **Python SDK**: `python-binance` (pip install python-binance)
