---
name: trading-research
description: Binance cryptocurrency trading research, technical analysis, and position management. Triggers on requests for crypto prices, market data, trading analysis, DCA planning, position sizing, whale activity, or any trading research questions about Bitcoin, altcoins, or crypto markets.
---

# Trading Research Skill

Comprehensive cryptocurrency trading research and analysis focused on Binance markets. Designed for conservative-moderate risk traders using DCA (Dollar Cost Averaging) strategies with technical analysis support.

## When to Use This Skill

Activate when user requests:
- Current crypto prices or market data
- Technical analysis (RSI, MACD, Bollinger Bands, etc.)
- DCA strategy planning or schedule calculation
- Position sizing with risk management
- Market scanning for opportunities
- Whale tracking or large order monitoring
- Trading strategy advice or risk assessment

## Core Philosophy

- **Conservative first**: Preserve capital, minimize risk
- **DCA-focused**: Time in market > timing the market
- **Risk management**: Never risk more than 1-2% per trade
- **Data-driven**: Use technical indicators for confirmation, not prediction
- **Transparent**: Show calculations, explain reasoning

## Available Tools

### 1. Market Data (`binance_market.py`)

Fetch real-time Binance market data.

**Use when**: User asks for price, volume, orderbook, recent trades, or funding rates.

**Common commands**:
```bash
# Current price and 24h stats (default)
python3 scripts/binance_market.py --symbol BTCUSDT

# Orderbook depth
python3 scripts/binance_market.py --symbol BTCUSDT --orderbook --depth 20

# Candlestick data
python3 scripts/binance_market.py --symbol BTCUSDT --klines 1h --limit 100

# Recent trades
python3 scripts/binance_market.py --symbol BTCUSDT --trades --limit 100

# Funding rate (futures)
python3 scripts/binance_market.py --symbol BTCUSDT --funding

# All data at once
python3 scripts/binance_market.py --symbol BTCUSDT --all

# JSON output (for piping)
python3 scripts/binance_market.py --symbol BTCUSDT --json > btc_data.json
```

**Intervals**: 1m, 5m, 15m, 30m, 1h, 4h, 1d, 1w

### 2. Technical Analysis (`technical_analysis.py`)

Calculate and interpret technical indicators.

**Use when**: User asks for TA, indicators, buy/sell signals, or market analysis.

**Common commands**:
```bash
# Full analysis (default: 1h timeframe, 200 candles)
python3 scripts/technical_analysis.py --symbol BTCUSDT

# Different timeframe
python3 scripts/technical_analysis.py --symbol BTCUSDT --interval 4h

# Custom RSI period
python3 scripts/technical_analysis.py --symbol BTCUSDT --rsi-period 21

# From saved klines JSON
python3 scripts/technical_analysis.py --input btc_klines.json

# JSON output
python3 scripts/technical_analysis.py --symbol BTCUSDT --json
```

**What it analyzes**:
- Trend direction (SMA 20/50, EMA 12/26)
- RSI (14) - overbought/oversold
- MACD - momentum and crossovers
- Bollinger Bands - volatility and position
- Support/resistance levels
- Volume analysis
- Trading signals and recommendations

### 3. DCA Calculator (`dca_calculator.py`)

Plan Dollar Cost Averaging strategies.

**Use when**: User wants to set up DCA, calculate investment schedules, or compare strategies.

**Common commands**:
```bash
# Basic DCA plan
python3 scripts/dca_calculator.py --total 5000 --frequency weekly --duration 180

# With current price for projections
python3 scripts/dca_calculator.py --total 10000 --frequency monthly --duration 365 --current-price 100000

# Show scenario analysis
python3 scripts/dca_calculator.py --total 5000 --frequency weekly --duration 180 --current-price 100000 --scenarios

# Custom start date
python3 scripts/dca_calculator.py --total 5000 --frequency weekly --duration 180 --start-date 2026-03-01

# JSON output
python3 scripts/dca_calculator.py --total 5000 --frequency weekly --duration 180 --json
```

**Frequencies**: daily, weekly, biweekly, monthly

**Output includes**:
- Purchase schedule with dates and amounts
- Number of purchases and amount per purchase
- Scenario analysis (flat, bull, bear markets)
- Comparison to lump sum approach

### 4. Position Sizer (`position_sizer.py`)

Calculate safe position sizes using risk management rules.

**Use when**: User wants to enter a trade and needs to know position size, stop loss, or take profit levels.

**Common commands**:
```bash
# Basic position sizing (2% risk recommended)
python3 scripts/position_sizer.py --balance 10000 --risk 2 --entry 100000 --stop-loss 95000

# Conservative 1% risk
python3 scripts/position_sizer.py --balance 10000 --risk 1 --entry 100000 --stop-loss 97000

# Custom take-profit ratios
python3 scripts/position_sizer.py --balance 10000 --risk 2 --entry 100000 --stop-loss 95000 --take-profit 2 3 5

# Ladder strategy (scaling in)
python3 scripts/position_sizer.py --balance 10000 --risk 2 --entry 100000 --stop-loss 95000 --ladder 3

# JSON output
python3 scripts/position_sizer.py --balance 10000 --risk 2 --entry 100000 --stop-loss 95000 --json
```

**Output includes**:
- Position size in units and dollar value
- Risk amount in dollars
- Stop loss percentage
- Take profit levels at multiple R:R ratios
- Position as percentage of account
- Warnings if position too large

**Rules**:
- Conservative: Risk 1% per trade
- Moderate: Risk 2% per trade
- Never exceed 3% risk per trade
- Position should be <50% of account

### 5. Market Scanner (`market_scanner.py`)

Scan all Binance USDT pairs for opportunities.

**Use when**: User wants to find top movers, volume spikes, or new opportunities.

**Common commands**:
```bash
# Full market scan (default)
python3 scripts/market_scanner.py

# Top gainers only
python3 scripts/market_scanner.py --gainers --limit 20

# High volume pairs
python3 scripts/market_scanner.py --volume

# Most volatile pairs
python3 scripts/market_scanner.py --volatile

# Breakout candidates (near 24h high with volume)
python3 scripts/market_scanner.py --breakout

# Filter by minimum volume
python3 scripts/market_scanner.py --min-volume 500000

# JSON output
python3 scripts/market_scanner.py --json
```

**Categories scanned**:
- Top gainers (24h price change)
- Top losers (24h price change)
- Highest volume pairs
- Most volatile pairs (high-low spread)
- Potential breakouts (near 24h high + volume)

### 6. Whale Tracker (`whale_tracker.py`)

Monitor large trades and orderbook imbalances.

**Use when**: User asks about whale activity, large orders, or orderbook pressure.

**Common commands**:
```bash
# Full whale analysis (default)
python3 scripts/whale_tracker.py --symbol BTCUSDT

# Large trades only
python3 scripts/whale_tracker.py --symbol BTCUSDT --trades

# Orderbook imbalances only
python3 scripts/whale_tracker.py --symbol BTCUSDT --orderbook

# Custom orderbook depth
python3 scripts/whale_tracker.py --symbol BTCUSDT --orderbook --depth 50

# Adjust threshold (default 90th percentile)
python3 scripts/whale_tracker.py --symbol BTCUSDT --threshold 95

# JSON output
python3 scripts/whale_tracker.py --symbol BTCUSDT --json
```

**Output includes**:
- Large trades (top 10% by value)
- Buy vs sell pressure from large trades
- Orderbook bid/ask imbalance
- Orderbook walls (large orders)
- Market sentiment (bullish/bearish/neutral)

## Quick Start Workflows

### "What's BTC doing?"
```bash
# Get overview
python3 scripts/binance_market.py --symbol BTCUSDT --ticker

# Technical analysis
python3 scripts/technical_analysis.py --symbol BTCUSDT --interval 1h
```

### "Should I buy now?"
```bash
# Check technicals first
python3 scripts/technical_analysis.py --symbol BTCUSDT

# Check whale activity
python3 scripts/whale_tracker.py --symbol BTCUSDT

# If signals look good, calculate position size
python3 scripts/position_sizer.py --balance 10000 --risk 2 --entry <CURRENT_PRICE> --stop-loss <SUPPORT_LEVEL>
```

### "Set up a DCA plan"
```bash
# Plan the strategy
python3 scripts/dca_calculator.py --total 5000 --frequency weekly --duration 180 --current-price <CURRENT_PRICE> --scenarios

# Show them the schedule and explain
```

### "Find me opportunities"
```bash
# Scan market
python3 scripts/market_scanner.py

# For interesting pairs, do deeper analysis
python3 scripts/technical_analysis.py --symbol <PAIR>
python3 scripts/whale_tracker.py --symbol <PAIR>
```

### "What's the market sentiment?"
```bash
# Check whale activity
python3 scripts/whale_tracker.py --symbol BTCUSDT

# Check volume and volatility
python3 scripts/market_scanner.py --volume --volatile
```

## Reference Materials

Located in `references/` directory:

### `binance-api.md`
- API endpoints and parameters
- Rate limits
- Authentication for signed requests
- Order types and time-in-force
- Error codes
- Python examples

**Use when**: Need API details, building custom queries, or troubleshooting

### `indicators.md`
- Technical indicator formulas
- Interpretation guidelines
- Common settings per timeframe
- Combining indicators
- Reliability assessment
- Common mistakes

**Use when**: Explaining indicators, interpreting signals, or educating user

### `strategies.md`
- DCA variations (fixed, value-based, RSI-based, ladder)
- Risk management (1-2% rule, stop loss strategies)
- Trend following strategies
- Entry/exit strategies
- Position sizing examples
- Performance tracking

**Use when**: Planning trades, explaining strategies, or risk management questions

## Trading Guidance

### For Conservative Traders

**DCA Approach**:
- Start with weekly or monthly purchases
- Fixed amount: $50-200 per purchase
- Duration: 6-12 months minimum
- Don't try to time the market
- Accumulate and hold long-term

**Risk Management**:
- No leverage
- 50%+ of account in cash/stablecoins
- Risk 1% per trade maximum
- Only trade with 3+ confirmations
- Stop losses always active

### For Moderate Risk Traders

**Enhanced DCA**:
- Adjust amounts based on RSI (buy more when oversold)
- Use technical analysis for better entries
- 60-70% DCA, 30-40% active trading
- Risk 2% per trade on active positions

**Position Trading**:
- Wait for confluence of indicators
- Use position_sizer.py for every trade
- Risk:Reward ratio minimum 2:1
- Trail stops as profit grows

### Red Flags (Don't Trade)

- RSI >70 and rising (overbought)
- Low volume breakout (likely false)
- Against major trend (don't short bull market)
- Multiple indicators conflicting
- No clear support level for stop loss
- Risk:Reward ratio <1.5:1
- During extreme fear or greed

## Response Format

When user asks for analysis:

1. **Current State**: Price, trend, key levels
2. **Technical View**: Indicator readings and what they mean
3. **Sentiment**: Whale activity, volume, market pressure
4. **Recommendation**: Buy/wait/sell with reasoning
5. **Risk Management**: Position size, stop loss, take profit if applicable
6. **Caveats**: What could go wrong, alternative scenarios

Always include:
- Specific numbers (don't just say "oversold", say "RSI at 28")
- Risk warnings for trades
- Clear next steps
- Timeframe context (day trade vs swing trade vs long-term)

## Important Notes

### API Access
- All scripts use Binance public API (no authentication needed for data)
- Respect rate limits (built into scripts)
- If API blocked by geo-restrictions, scripts will error gracefully

### Limitations
- **No trading execution**: These tools are for research only
- **No real-time WebSocket**: Data is snapshot-based (REST API)
- **No futures-specific features**: Primarily spot market focused (except funding rates)
- **No backtesting engine**: Manual strategy evaluation

### Authentication Required For
- Placing orders
- Checking account balance
- Viewing open orders
- Accessing trade history

**Note**: Guide users to Binance API documentation (see `references/binance-api.md`) for authenticated trading setup.

## Error Handling

If script fails:
1. Check internet connection
2. Verify symbol format (uppercase, e.g., BTCUSDT not btc-usdt)
3. Check if Binance API accessible in user's location
4. Verify script path and Python availability
5. Check for typos in parameters

Common errors:
- **HTTP 451**: API blocked in location (suggest VPN)
- **Invalid symbol**: Check symbol exists on Binance
- **Rate limit**: Wait 60 seconds and retry
- **Connection timeout**: Network issue or API down

## Best Practices

1. **Always show your work**: Display the command you ran
2. **Interpret results**: Don't just dump data, explain what it means
3. **Context matters**: Different advice for day trade vs DCA accumulation
4. **Risk first**: Mention risk management before entry signals
5. **Be honest**: If indicators conflict, say so
6. **Update knowledge**: If market conditions changed, acknowledge it
7. **No predictions**: Frame as "if X then Y", not "X will happen"
8. **Show alternatives**: Bull and bear case scenarios

## Skill Maintenance

### Testing
Run each script monthly to ensure API compatibility:
```bash
python3 scripts/binance_market.py --symbol BTCUSDT --help
python3 scripts/technical_analysis.py --help
python3 scripts/dca_calculator.py --help
python3 scripts/position_sizer.py --help
python3 scripts/market_scanner.py --help
python3 scripts/whale_tracker.py --help
```

### Updates Needed If
- Binance changes API endpoints
- New technical indicators requested
- Additional risk management tools needed
- User feedback suggests improvements

---

**Remember**: This skill helps users make informed decisions. It does not make decisions for them. Always emphasize personal responsibility and risk disclosure.
