#!/usr/bin/env bash
# Setup script for Total Recall
# Creates directory structure and optionally installs watcher service (Linux)

set -euo pipefail

SKILL_DIR="$(cd "$(dirname "$0")/.." && pwd)"
source "$SKILL_DIR/scripts/_compat.sh"

WORKSPACE="${OPENCLAW_WORKSPACE:-$(cd "$SKILL_DIR/../.." && pwd)}"
MEMORY_DIR="${MEMORY_DIR:-$WORKSPACE/memory}"

echo "🧠 Total Recall — Setup"
echo "========================"
echo "Workspace: $WORKSPACE"
echo "Memory dir: $MEMORY_DIR"
echo "Skill dir: $SKILL_DIR"
echo "Platform: $(uname -s)"
echo ""

# --- Check dependencies ---
echo "Checking dependencies..."
MISSING=0
for bin in jq curl; do
  if ! command -v "$bin" &>/dev/null; then
    echo "❌ Missing: $bin"
    MISSING=1
  fi
done
if [ "$MISSING" -eq 1 ]; then
  echo ""
  echo "Install missing dependencies and re-run setup."
  exit 1
fi
echo "✅ Core dependencies found (jq, curl)"

if has_inotify; then
  echo "✅ inotifywait found (reactive watcher available)"
else
  echo "ℹ️  inotifywait not found — reactive watcher won't be available"
  if $_IS_MACOS; then
    echo "   This is expected on macOS. Cron-only mode works fine."
  else
    echo "   Install: sudo apt install inotify-tools (Debian/Ubuntu)"
    echo "   Or: sudo dnf install inotify-tools (Fedora/RHEL)"
  fi
fi

# --- Check API key ---
if [ -f "$WORKSPACE/.env" ]; then
  set -a
  # Only load OPENROUTER_API_KEY (minimal credential exposure)
  eval "$(grep -E '^OPENROUTER_API_KEY=' "$WORKSPACE/.env" 2>/dev/null)" || true
  set +a
fi
if [ -z "${OPENROUTER_API_KEY:-}" ]; then
  echo ""
  echo "⚠️  OPENROUTER_API_KEY not found in environment or $WORKSPACE/.env"
  echo "   The observer needs this to call the LLM. Add it to your .env file:"
  echo "   echo 'OPENROUTER_API_KEY=sk-or-v1-xxxxx' >> $WORKSPACE/.env"
  echo ""
fi

# --- Create directories ---
echo ""
echo "Creating directory structure..."
mkdir -p "$MEMORY_DIR/observation-backups"
mkdir -p "$MEMORY_DIR/archive/observations"
mkdir -p "$MEMORY_DIR/dream-logs"
mkdir -p "$MEMORY_DIR/.dream-backups"
mkdir -p "$WORKSPACE/logs"
mkdir -p "$WORKSPACE/research/dream-cycle-metrics/daily"
echo "✅ Directories created"

# --- Create initial observations file ---
if [ ! -f "$MEMORY_DIR/observations.md" ]; then
  cat > "$MEMORY_DIR/observations.md" << 'EOF'
# Observations Log

Auto-generated by Observer agent. Loaded at session startup for cross-session memory.

---

EOF
  echo "✅ Created observations.md"
else
  echo "ℹ️  observations.md already exists, keeping it"
fi

# --- Make scripts executable ---
chmod +x "$SKILL_DIR/scripts/"*.sh
echo "✅ Scripts made executable"

# --- Install systemd watcher service (Linux only) ---
if has_inotify && has_systemd_user; then
  echo ""
  echo "Installing reactive watcher service..."
  SYSTEMD_DIR="$HOME/.config/systemd/user"
  mkdir -p "$SYSTEMD_DIR"

  cat > "$SYSTEMD_DIR/total-recall-watcher.service" << EOF
[Unit]
Description=Total Recall — Reactive Observer Watcher
After=default.target

[Service]
Type=simple
ExecStart=$SKILL_DIR/scripts/observer-watcher.sh
Restart=on-failure
RestartSec=30
Environment=OPENCLAW_WORKSPACE=$WORKSPACE

[Install]
WantedBy=default.target
EOF

  systemctl --user daemon-reload
  systemctl --user enable total-recall-watcher.service
  systemctl --user start total-recall-watcher.service 2>/dev/null || true
  echo "✅ Watcher service installed and started"
else
  echo ""
  echo "ℹ️  Skipping reactive watcher service (requires Linux + systemd + inotify-tools)"
  echo "   The cron-based observer (every 15 min) provides full coverage without it."
fi

# --- Print next steps ---
echo ""
echo "================================================"
echo "🎉 Setup complete!"
echo "================================================"
echo ""
echo "Next steps — you need to configure these manually:"
echo ""
echo "1. ADD CRON JOBS (via OpenClaw cron or system crontab):"
echo ""
echo "   # Observer — every 15 minutes"
echo "   */15 * * * * OPENCLAW_WORKSPACE=$WORKSPACE bash $SKILL_DIR/scripts/observer-agent.sh >> $WORKSPACE/logs/observer.log 2>&1"
echo ""
echo "   # Reflector — hourly check"
echo "   0 * * * * OPENCLAW_WORKSPACE=$WORKSPACE bash $SKILL_DIR/scripts/reflector-agent.sh >> $WORKSPACE/logs/reflector.log 2>&1"
echo ""
echo "   # Dream Cycle — nightly (3am recommended; adjust to your timezone)"
echo "   # Run a sub-agent with prompts/dream-cycle-prompt.md — see SKILL.md for full setup"
echo "   # 0 3 * * * OPENCLAW_WORKSPACE=$WORKSPACE bash $SKILL_DIR/scripts/dream-cycle.sh preflight"
echo ""
echo "2. ADD SESSION RECOVERY to your agent's startup procedure:"
echo "   bash $SKILL_DIR/scripts/session-recovery.sh"
echo ""
echo "3. ADD TO YOUR AGENT'S CONTEXT (system prompt or MEMORY.md):"
echo "   'At session startup, read memory/observations.md for cross-session context.'"
echo ""
echo "4. (OPTIONAL) CONFIGURE memoryFlush pre-compaction hook:"
echo "   Run the observer in --flush mode before OpenClaw compacts context."
echo ""
echo "Logs:"
echo "  Observer:  tail -f $WORKSPACE/logs/observer.log"
echo "  Reflector: tail -f $WORKSPACE/logs/reflector.log"
if has_inotify && has_systemd_user; then
  echo "  Watcher:   systemctl --user status total-recall-watcher"
fi
