"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SCAN_RULES = void 0;
exports.SCAN_RULES = [
    // --- SEC rules: code-level security ---
    {
        id: 'SEC-001',
        name: 'eval() usage',
        severity: 'high',
        pattern: /\beval\s*\(/,
        description: 'Use of eval() allows arbitrary code execution and is a common injection vector.',
        remediation: 'Replace eval() with JSON.parse(), Function(), or a safe expression parser.',
    },
    {
        id: 'SEC-002',
        name: 'Function constructor',
        severity: 'high',
        pattern: /new\s+Function\s*\(/,
        description: 'The Function constructor creates functions from strings, enabling code injection.',
        remediation: 'Use static function definitions or a sandboxed evaluator.',
    },
    {
        id: 'SEC-003',
        name: 'child_process usage',
        severity: 'critical',
        pattern: /(?:require\s*\(\s*['"]child_process['"]\s*\)|from\s+['"]child_process['"]|child_process\s*\.\s*(?:exec|spawn|execSync|spawnSync|execFile|fork)\s*\()/,
        description: 'Direct use of child_process can execute arbitrary system commands.',
        remediation: 'Use Tork Guardian governTool() to validate commands before execution.',
    },
    {
        id: 'SEC-004',
        name: 'Write to sensitive paths',
        severity: 'high',
        pattern: /(?:writeFile|writeFileSync|appendFile|appendFileSync)\s*\(\s*['"`](?:\/etc\/|~?\/?\.env|~?\/?\.ssh|~?\/?\.aws|~?\/?\.npmrc|\/root\/)/,
        description: 'Writing to sensitive system paths can compromise credentials or system configuration.',
        remediation: 'Restrict file writes to an allowed directory. Use Tork Guardian blockedPaths config.',
    },
    {
        id: 'SEC-005',
        name: 'Hardcoded API keys/secrets',
        severity: 'critical',
        pattern: /(?:api[_-]?key|api[_-]?secret|auth[_-]?token|access[_-]?token|secret[_-]?key|private[_-]?key)\s*[:=]\s*['"`][a-zA-Z0-9_\-]{16,}['"`]/i,
        description: 'Hardcoded secrets can be extracted from source code and used to compromise services.',
        remediation: 'Use environment variables or a secrets manager. Never commit secrets to source.',
    },
    {
        id: 'SEC-006',
        name: 'Prompt injection patterns',
        severity: 'critical',
        pattern: /(?:ignore\s+(?:previous|above|all)\s+instructions|you\s+are\s+now\s+(?:a|an)\s+|system\s*:\s*override|forget\s+(?:your|all)\s+(?:rules|instructions)|IMPORTANT:\s*ignore)/i,
        description: 'Prompt injection attempts can override LLM system instructions and bypass safety controls.',
        remediation: 'Sanitize user inputs before including in prompts. Use Tork Guardian governLLM().',
    },
    {
        id: 'SEC-007',
        name: 'Data exfiltration via HTTP',
        severity: 'high',
        pattern: /(?:fetch|axios\.(?:get|post|put|patch)|http\.request|https\.request)\s*\(\s*['"`]https?:\/\/(?!localhost|127\.0\.0\.1)/,
        description: 'Outbound HTTP requests to external URLs may exfiltrate sensitive data.',
        remediation: 'Use Tork Guardian network governance with an explicit domain allowlist.',
    },
    // --- NET rules: network-level security ---
    {
        id: 'NET-001',
        name: 'Server creation',
        severity: 'medium',
        pattern: /(?:net\.createServer|http\.createServer|https\.createServer)\s*\(/,
        description: 'Creating network servers may expose the host to inbound connections.',
        remediation: 'Use Tork Guardian validatePortBind() to govern which ports may be opened.',
    },
    {
        id: 'NET-002',
        name: 'Hardcoded IP addresses',
        severity: 'high',
        pattern: /['"`]\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}['"`]/,
        description: 'Hardcoded IPs bypass DNS-based security controls and domain allowlists.',
        remediation: 'Use domain names instead of IPs. Tork Guardian blocks raw IP connections.',
    },
    {
        id: 'NET-003',
        name: 'UDP socket creation',
        severity: 'high',
        pattern: /dgram\.createSocket\s*\(/,
        description: 'UDP sockets can be used for DNS tunneling, data exfiltration, or C2 communication.',
        remediation: 'Avoid UDP unless required. Use Tork Guardian network governance for monitoring.',
    },
    {
        id: 'NET-004',
        name: 'DNS TXT record queries',
        severity: 'critical',
        pattern: /dns\.(?:resolveTxt|resolve)\s*\(|\.setServers\s*\(|type:\s*['"]TXT['"]/,
        description: 'DNS TXT queries are a common data exfiltration and C2 channel technique.',
        remediation: 'DNS TXT lookups should be blocked unless explicitly required and allowlisted.',
    },
    {
        id: 'NET-005',
        name: 'Process spawn with network call',
        severity: 'critical',
        pattern: /(?:exec|spawn|execSync|spawnSync)\s*\(\s*['"`](?:curl|wget|nc|ncat|socat|telnet|nslookup|dig)\b/,
        description: 'Spawning network tools from child processes can bypass application-level controls.',
        remediation: 'Block shell-based network tools. Use Tork Guardian governTool() for command validation.',
    },
    {
        id: 'NET-006',
        name: 'WebSocket to non-standard domains',
        severity: 'medium',
        pattern: /new\s+WebSocket\s*\(\s*['"`]wss?:\/\/(?!localhost|127\.0\.0\.1)/,
        description: 'WebSocket connections to external domains may establish persistent C2 channels.',
        remediation: 'Restrict WebSocket connections to known, allowlisted domains.',
    },
    {
        id: 'NET-007',
        name: 'Sequential port scanning',
        severity: 'critical',
        pattern: /(?:net\.connect|net\.createConnection|new\s+net\.Socket)\s*\(\s*(?:\{[^}]*port\s*:|[0-9])/,
        description: 'Sequential port connection attempts indicate port scanning or service discovery.',
        remediation: 'Port scanning should be blocked. Use Tork Guardian validateEgress() for all connections.',
    },
];
//# sourceMappingURL=rules.js.map