---
name: tokenbroker
description: AI Agent Skill for GitHub project analysis and nad.fun token launch. Analyzes repos, generates token identity/promo, and launches on nad.fun.
version: 1.01
metadata:
  tags: monad, nadfun, token, launch, github, memecoin, autonomous
---

# SKILL.md - TokenBroker Skillset

## Security & Data Privacy

### Local Storage Only
- All credentials (GitHub token, private keys, API keys) are stored **locally** in a `.env` file
- No credentials are transmitted to external servers beyond their intended endpoints (GitHub API, nad.fun API, Monad RPC)
- The skill operates entirely within your local environment

### .env File Generation
- The Install Wizard generates a `.env` file on your local machine
- This file is **never committed** to version control (gitignored)
- You can review and edit it at any time

### Credential Scope
- `GITHUB_TOKEN`: Used only for GitHub API calls to read public repository data
- `PRIVATE_KEY`: Used only for EVM transaction signing (never exposed in plain text)
- `BUILDER_ID`: Local identifier for A2A protocol
- `NAD_FUN_API_KEY`: Used only for nad.fun token creation API

### Testnet Mode
- Default operation is on **testnet** for safety
- Mainnet requires explicit configuration
- Always review transactions before signing

---

**The AI agent skill for memecoin launches on nad.fun.** Analyze GitHub projects, generate token metadata, and launch directly on nad.fun bonding curves.

## What is TokenBroker?

TokenBroker is a **complete memecoin launch solution** for AI agents:

1. **Analyzes** GitHub projects to identify meme-worthy projects
2. **Generates** token names, tickers, descriptions, and marketing content
3. **Launches** tokens on nad.fun (image, metadata, salt, deploy)
4. **Promotes** launches with X/Telegram/Discord content

## When to Use This Skill

### TokenBroker Handles
- GitHub repository analysis and scoring
- Token identity generation (name, ticker, description)
- Meme-style image generation
- Nad.fun API integration (upload, salt mining)
- Marketing content creation (X threads, Telegram, Discord)
- Full launch orchestration

### Not Included
- Wallet private key management (handled by host)
- On-chain transactions beyond nad.fun bonding curves

## Architecture (tokenbroker/src/generators/)

```
generators/
├── identity.ts     # Token name, ticker, description generation
├── reasoning.ts    # Investment thesis, narrative creation
├── promo.ts        # X threads, Telegram, Discord content
├── nadfun.ts       # Nad.fun API: upload image/metadata, mine salt
└── index.ts        # Pipeline orchestrator (generateAll)
```

## Quick Start for Agents

```typescript
import { generateAll, prepareLaunch } from './generators/index.js';

// 1. Analyze repo and generate all launch assets
const assets = await generateAll({
  repoAnalysis: await analyzeGitHubRepo('https://github.com/user/project')
});

console.log('Token name:', assets.identity.name);
console.log('Ticker:', assets.identity.ticker);
console.log('X Thread:', assets.promo.xThread.tweets);

// 2. Prepare launch on nad.fun (API calls only)
const prepared = await prepareLaunch(assets.identity, 'mainnet');
// -> Returns: { imageUri, metadataUri, salt, saltAddress }

// 3. Deploy on-chain (requires ethers + private key)
// Use deploy.ts module with wallet for on-chain execution
```

## Generator Functions

### generateIdentity(input)
Analyzes repo and generates token identity:
```typescript
{
  name: "SWAPPRO",
  ticker: "SWAP", 
  tagline: "The next generation DeFi protocol",
  description: "Full token description...",
  nameReasoning: "How the name was derived"
}
```

### generateReasoning(input)
Creates investment thesis and narrative:
```typescript
{
  investmentThesis: "Why this token should exist...",
  problemStatement: "The problem being solved",
  solution: "The proposed solution",
  marketOpportunity: "Market size and opportunity",
  competitiveAdvantage: "Why this wins",
  tokenUtilityRationale: "Token value proposition",
  vision: "Long-term vision"
}
```

### generatePromo(input)
Generates marketing content:
```typescript
{
  xThread: { title, tweets: [...], hashtags, mentions },
  telegramPost: { title, content, hasButton, buttonText, buttonUrl },
  discordAnnouncement: { title, content, hasEmbed, embedColor, embedFields },
  tagline: "Marketing tagline",
  elevatorPitch: "One-liner pitch"
}
```

### prepareLaunch(identity, network)
Prepares token for nad.fun launch (API calls):
```typescript
{
  imageUri: "ipfs://...",
  metadataUri: "ipfs://...", 
  salt: "0x...",
  saltAddress: "0x..."
}
```

## Nad.fun Integration

TokenBroker integrates directly with nad.fun API:

| Step | API Endpoint | Function |
|------|-------------|----------|
| 1 | POST /agent/token/image | `uploadImage()` |
| 2 | POST /agent/token/metadata | `uploadMetadata()` |
| 3 | POST /agent/salt | `mineSalt()` |
| 4 | BondingCurveRouter.create() | On-chain deployment |

### Network Configuration
| Network | API | RPC |
|---------|-----|-----|
| Testnet | https://dev-api.nad.fun | https://testnet-rpc.monad.xyz |
| Mainnet | https://api.nadapp.net | https://rpc.monad.xyz |

## Install

```bash
npm install
```

## Configuration

```bash
# Network (testnet | mainnet)
NETWORK=mainnet

# GitHub (optional - for repo analysis)
GITHUB_TOKEN=ghp_...
```

## For On-Chain Deployment

TokenBroker prepares all launch data. For actual on-chain deployment:

```bash
npm install ethers
```

Then use with a wallet:
```typescript
import { prepareLaunch } from './generators/nadfun.js';
import { ethers } from 'ethers';

const prepared = await prepareLaunch(identity, 'mainnet');

// Deploy with wallet
const wallet = new ethers.Wallet(privateKey, provider);
const router = new ethers.Contract(BONDING_CURVE_ROUTER, abi, wallet);
await router.create(tokenParams, fee, toll, tradingAmt, { value: deployFee });
```

---

*Built for the agentic future.* 🦞
