#!/usr/bin/env python3
"""
Token Optimizer - Analyzer Module
Analyzes OpenClaw configuration and estimates token usage & savings.
"""

import json
import os
import sys
from pathlib import Path
from datetime import datetime
from typing import Dict, List, Optional, Tuple

try:
    from src.colors import Colors, colorize
except ImportError:
    from colors import Colors, colorize

class OpenClawAnalyzer:
    """Analyzes OpenClaw configuration for token optimization opportunities."""

    # Cost per 1K tokens (approximate)
    COSTS = {
        'sonnet': 0.003,
        'haiku': 0.00025,
        'opus': 0.015,
        'ollama': 0.0
    }

    # Average token estimates
    ESTIMATES = {
        'large_context': 50000,      # 50KB unoptimized context
        'lean_context': 8000,        # 8KB optimized context
        'heartbeat_tokens': 500,     # Tokens per heartbeat
        'heartbeats_per_day': 24,    # Hourly heartbeats
        'avg_messages_per_day': 100, # Average API calls
    }

    def __init__(self):
        self.config_path = self._find_config()
        self.config = self._load_config()
        self.workspace_files = self._scan_workspace()
        self.issues: List[Dict] = []
        self.optimizations: List[Dict] = []

    def _find_config(self) -> Optional[Path]:
        """Find OpenClaw configuration file."""
        possible_paths = [
            Path.home() / '.openclaw' / 'openclaw.json',
            Path.home() / '.openclaw' / 'openclaw-config.json',
            Path.home() / '.openclaw' / 'config.json',
            Path.cwd() / '.openclaw.json',
            Path.cwd() / 'openclaw.json',
        ]

        for path in possible_paths:
            if path.exists():
                return path
        return None

    def _load_config(self) -> Dict:
        """Load OpenClaw configuration."""
        if self.config_path and self.config_path.exists():
            try:
                with open(self.config_path, 'r') as f:
                    return json.load(f)
            except json.JSONDecodeError:
                return {}
        return {}

    def _scan_workspace(self) -> Dict[str, int]:
        """Scan workspace files and their sizes."""
        workspace_files = {}
        workspace_paths = [
            Path.cwd(),
            Path.home() / '.openclaw' / 'workspace',
        ]

        target_files = ['SOUL.md', 'USER.md', 'IDENTITY.md', 'MEMORY.md',
                       'TOOLS.md', 'REFERENCE.md', 'CONTEXT.md']

        for base_path in workspace_paths:
            if base_path.exists():
                for file_name in target_files:
                    file_path = base_path / file_name
                    if file_path.exists():
                        workspace_files[file_name] = file_path.stat().st_size

        return workspace_files

    def analyze_model_routing(self) -> Dict:
        """Analyze model routing configuration."""
        result = {
            'status': 'not_configured',
            'default_model': 'unknown',
            'has_haiku': False,
            'has_sonnet': False,
            'has_aliases': False,
            'monthly_savings': 0
        }

        agents_config = self.config.get('agents', {})
        defaults = agents_config.get('defaults', {})
        model_config = defaults.get('model', {})
        models = defaults.get('models', {})

        # Check primary model
        primary = model_config.get('primary', '')
        if 'haiku' in primary.lower():
            result['default_model'] = 'haiku'
            result['status'] = 'optimized'
        elif 'sonnet' in primary.lower():
            result['default_model'] = 'sonnet'
            result['status'] = 'needs_optimization'
        elif 'opus' in primary.lower():
            result['default_model'] = 'opus'
            result['status'] = 'needs_optimization'

        # Check available models
        for model_name in models:
            if 'haiku' in model_name.lower():
                result['has_haiku'] = True
            if 'sonnet' in model_name.lower():
                result['has_sonnet'] = True
            if models[model_name].get('alias'):
                result['has_aliases'] = True

        # Calculate potential savings
        if result['status'] == 'needs_optimization':
            daily_calls = self.ESTIMATES['avg_messages_per_day']
            avg_tokens = 2000  # per call

            if result['default_model'] == 'sonnet':
                current_cost = (daily_calls * avg_tokens / 1000) * self.COSTS['sonnet']
            else:  # opus
                current_cost = (daily_calls * avg_tokens / 1000) * self.COSTS['opus']

            optimized_cost = (daily_calls * avg_tokens / 1000) * self.COSTS['haiku']
            result['monthly_savings'] = (current_cost - optimized_cost) * 30

        return result

    def analyze_heartbeat(self) -> Dict:
        """Analyze heartbeat configuration."""
        result = {
            'status': 'not_configured',
            'provider': 'api',
            'interval': 3600,
            'monthly_cost': 0,
            'monthly_savings': 0
        }

        heartbeat_config = self.config.get('heartbeat', {})

        if heartbeat_config:
            result['interval'] = heartbeat_config.get('every', '1h')
            model = heartbeat_config.get('model', '')

            if 'ollama' in model.lower() or 'local' in model.lower():
                result['provider'] = 'ollama'
                result['status'] = 'optimized'
                result['monthly_cost'] = 0
            else:
                result['provider'] = 'api'
                result['status'] = 'needs_optimization'

                # Calculate cost
                heartbeats_per_day = self.ESTIMATES['heartbeats_per_day']
                tokens_per_heartbeat = self.ESTIMATES['heartbeat_tokens']
                cost_per_1k = self.COSTS['haiku']  # assume haiku at minimum

                daily_cost = (heartbeats_per_day * tokens_per_heartbeat / 1000) * cost_per_1k
                result['monthly_cost'] = daily_cost * 30
                result['monthly_savings'] = result['monthly_cost']
        else:
            result['status'] = 'not_configured'

        return result

    def analyze_session_management(self) -> Dict:
        """Analyze session initialization and context management."""
        result = {
            'status': 'unknown',
            'estimated_context_size': 0,
            'optimized_context_size': 8000,
            'monthly_savings': 0
        }

        # Calculate current context size from workspace files
        total_size = sum(self.workspace_files.values())
        result['estimated_context_size'] = total_size if total_size > 0 else self.ESTIMATES['large_context']

        # Check if files are reasonably sized
        if total_size > 20000:  # > 20KB
            result['status'] = 'needs_optimization'
        elif total_size > 0:
            result['status'] = 'optimized'
        else:
            result['status'] = 'no_workspace_files'

        # Calculate savings
        if result['status'] == 'needs_optimization':
            daily_calls = self.ESTIMATES['avg_messages_per_day']
            excess_tokens = (result['estimated_context_size'] - result['optimized_context_size'])
            cost_per_1k = self.COSTS['haiku']

            daily_waste = (excess_tokens / 1000) * cost_per_1k * daily_calls
            result['monthly_savings'] = daily_waste * 30

        return result

    def analyze_caching(self) -> Dict:
        """Analyze prompt caching configuration."""
        result = {
            'status': 'not_configured',
            'enabled': False,
            'ttl': '5m',
            'monthly_savings': 0
        }

        agents_config = self.config.get('agents', {})
        defaults = agents_config.get('defaults', {})
        cache_config = defaults.get('cache', {})

        if cache_config.get('enabled'):
            result['enabled'] = True
            result['status'] = 'optimized'
            result['ttl'] = cache_config.get('ttl', '5m')
        else:
            result['status'] = 'needs_optimization'

            # Calculate potential savings (90% on cached content)
            daily_calls = self.ESTIMATES['avg_messages_per_day']
            prompt_size = 5000  # 5KB typical
            cost_per_1k = self.COSTS['sonnet']  # caching matters most for sonnet

            uncached_cost = (prompt_size / 1000) * cost_per_1k * daily_calls
            cached_cost = uncached_cost * 0.1  # 90% discount
            result['monthly_savings'] = (uncached_cost - cached_cost) * 30

        return result

    def analyze_rate_limits(self) -> Dict:
        """Check if rate limits are configured."""
        result = {
            'status': 'unknown',
            'has_api_limit': False,
            'has_budget': False,
            'daily_budget': None,
            'monthly_budget': None
        }

        # Check for rate limit configuration
        rate_limits = self.config.get('rate_limits', {})
        budgets = self.config.get('budgets', {})

        if rate_limits:
            result['has_api_limit'] = True
        if budgets:
            result['has_budget'] = True
            result['daily_budget'] = budgets.get('daily')
            result['monthly_budget'] = budgets.get('monthly')

        if result['has_api_limit'] or result['has_budget']:
            result['status'] = 'configured'
        else:
            result['status'] = 'not_configured'

        return result

    def run_full_analysis(self) -> Dict:
        """Run complete analysis and return results."""
        print(colorize("\n=== OpenClaw Token Optimizer - Analysis ===\n", Colors.BOLD + Colors.CYAN))

        results = {
            'timestamp': datetime.now().isoformat(),
            'config_found': self.config_path is not None,
            'config_path': str(self.config_path) if self.config_path else None,
            'workspace_files': self.workspace_files,
            'model_routing': self.analyze_model_routing(),
            'heartbeat': self.analyze_heartbeat(),
            'session_management': self.analyze_session_management(),
            'caching': self.analyze_caching(),
            'rate_limits': self.analyze_rate_limits(),
        }

        # Calculate total potential savings
        total_savings = (
            results['model_routing']['monthly_savings'] +
            results['heartbeat']['monthly_savings'] +
            results['session_management']['monthly_savings'] +
            results['caching']['monthly_savings']
        )
        results['total_monthly_savings'] = total_savings

        self._print_results(results)
        return results

    def _print_results(self, results: Dict):
        """Print formatted analysis results."""

        # Config status
        if results['config_found']:
            print(colorize(f"[FOUND] Config: {results['config_path']}", Colors.GREEN))
        else:
            print(colorize("[NOT FOUND] No OpenClaw config file detected", Colors.YELLOW))
            print("  Run 'optimize' to create optimized configuration\n")

        # Workspace files
        print(colorize("\n--- Workspace Files ---", Colors.BOLD))
        if results['workspace_files']:
            for name, size in results['workspace_files'].items():
                size_kb = size / 1024
                color = Colors.GREEN if size_kb < 5 else Colors.YELLOW if size_kb < 15 else Colors.RED
                print(f"  {name}: {colorize(f'{size_kb:.1f}KB', color)}")
        else:
            print(colorize("  No workspace files found", Colors.YELLOW))

        # Model Routing
        print(colorize("\n--- Model Routing ---", Colors.BOLD))
        mr = results['model_routing']
        status_color = Colors.GREEN if mr['status'] == 'optimized' else Colors.RED
        print(f"  Status: {colorize(mr['status'].upper(), status_color)}")
        print(f"  Default Model: {mr['default_model']}")
        if mr['monthly_savings'] > 0:
            print(colorize(f"  Potential Savings: ${mr['monthly_savings']:.2f}/month", Colors.GREEN))

        # Heartbeat
        print(colorize("\n--- Heartbeat Configuration ---", Colors.BOLD))
        hb = results['heartbeat']
        status_color = Colors.GREEN if hb['status'] == 'optimized' else Colors.YELLOW if hb['status'] == 'not_configured' else Colors.RED
        print(f"  Status: {colorize(hb['status'].upper(), status_color)}")
        print(f"  Provider: {hb['provider']}")
        if hb['monthly_savings'] > 0:
            print(colorize(f"  Potential Savings: ${hb['monthly_savings']:.2f}/month", Colors.GREEN))

        # Session Management
        print(colorize("\n--- Session Management ---", Colors.BOLD))
        sm = results['session_management']
        status_color = Colors.GREEN if sm['status'] == 'optimized' else Colors.YELLOW if sm['status'] == 'no_workspace_files' else Colors.RED
        print(f"  Status: {colorize(sm['status'].upper(), status_color)}")
        print(f"  Estimated Context: {sm['estimated_context_size'] / 1024:.1f}KB")
        if sm['monthly_savings'] > 0:
            print(colorize(f"  Potential Savings: ${sm['monthly_savings']:.2f}/month", Colors.GREEN))

        # Caching
        print(colorize("\n--- Prompt Caching ---", Colors.BOLD))
        cache = results['caching']
        status_color = Colors.GREEN if cache['status'] == 'optimized' else Colors.RED
        print(f"  Status: {colorize(cache['status'].upper(), status_color)}")
        print(f"  Enabled: {cache['enabled']}")
        if cache['monthly_savings'] > 0:
            print(colorize(f"  Potential Savings: ${cache['monthly_savings']:.2f}/month", Colors.GREEN))

        # Rate Limits
        print(colorize("\n--- Rate Limits & Budgets ---", Colors.BOLD))
        rl = results['rate_limits']
        status_color = Colors.GREEN if rl['status'] == 'configured' else Colors.YELLOW
        print(f"  Status: {colorize(rl['status'].upper(), status_color)}")
        print(f"  API Limits: {'Yes' if rl['has_api_limit'] else 'No'}")
        print(f"  Budgets: {'Yes' if rl['has_budget'] else 'No'}")

        # Total Savings
        print(colorize("\n========================================", Colors.BOLD + Colors.CYAN))
        print(colorize(f"TOTAL POTENTIAL SAVINGS: ${results['total_monthly_savings']:.2f}/month", Colors.BOLD + Colors.GREEN))
        print(colorize(f"                         ${results['total_monthly_savings'] * 12:.2f}/year", Colors.GREEN))
        print(colorize("========================================\n", Colors.BOLD + Colors.CYAN))

        # Recommendations
        if results['total_monthly_savings'] > 0:
            print(colorize("RECOMMENDATIONS:", Colors.BOLD + Colors.YELLOW))
            if mr['status'] == 'needs_optimization':
                print("  1. Switch default model to Haiku")
            if hb['status'] != 'optimized':
                print("  2. Route heartbeats to Ollama (free)")
            if sm['status'] == 'needs_optimization':
                print("  3. Implement session initialization rules")
            if cache['status'] != 'optimized':
                print("  4. Enable prompt caching")
            if rl['status'] != 'configured':
                print("  5. Add rate limits and budgets")
            print(colorize("\nRun 'token-optimizer optimize' to apply all optimizations", Colors.CYAN))


def main():
    """Main entry point."""
    analyzer = OpenClawAnalyzer()
    results = analyzer.run_full_analysis()

    # Save results to file
    output_path = Path.cwd() / '.token-optimizer-analysis.json'
    with open(output_path, 'w') as f:
        json.dump(results, f, indent=2, default=str)
    print(f"\nDetailed results saved to: {output_path}")
    print(colorize("\nRun 'python src/verify.py' to see your accumulated savings report.", Colors.CYAN))

    return 0 if results['total_monthly_savings'] == 0 else 1


if __name__ == '__main__':
    sys.exit(main())
