#!/usr/bin/env npx tsx
/**
 * Swap ETH for PumpClaw tokens via Uniswap V4
 */

import {
  createPublicClient,
  createWalletClient,
  http,
  parseEther,
  formatUnits,
  encodeFunctionData,
} from "viem";
import { base } from "viem/chains";
import { privateKeyToAccount } from "viem/accounts";
import { CONTRACTS, CHAIN } from "../../../shared/contracts.js";
import { ERC20_ABI } from "../../../shared/abis.js";

const UNIVERSAL_ROUTER = "0x6fF5693b99212Da76ad316178A184AB56D299b43"; // Uniswap Universal Router on Base
const SWAP_ROUTER = "0x2626664c2603336E57B271c5C0b26F421741e481"; // SwapRouter02 on Base

const publicClient = createPublicClient({
  chain: base,
  transport: http(CHAIN.RPC),
});

function getWalletClient() {
  const key = process.env.BASE_PRIVATE_KEY;
  if (!key) {
    console.error("Error: BASE_PRIVATE_KEY required");
    process.exit(1);
  }
  const privateKey = key.startsWith("0x") ? key : `0x${key}`;
  const account = privateKeyToAccount(privateKey as `0x${string}`);
  return createWalletClient({
    account,
    chain: base,
    transport: http(CHAIN.RPC),
  });
}

// V4 swap via SwapRouter02
async function swapExactETHForTokens(tokenOut: string, ethAmount: string) {
  const walletClient = getWalletClient();
  const account = walletClient.account!;
  
  const value = parseEther(ethAmount);
  
  console.log(`Swapping ${ethAmount} ETH for ${tokenOut}`);
  console.log(`From: ${account.address}`);
  
  // Get token balance before
  const balanceBefore = await publicClient.readContract({
    address: tokenOut as `0x${string}`,
    abi: ERC20_ABI,
    functionName: "balanceOf",
    args: [account.address],
  });

  // exactInputSingle for V4
  const hash = await walletClient.sendTransaction({
    to: SWAP_ROUTER,
    value,
    data: encodeFunctionData({
      abi: [{
        type: "function",
        name: "exactInputSingle",
        inputs: [{
          type: "tuple",
          name: "params",
          components: [
            { name: "tokenIn", type: "address" },
            { name: "tokenOut", type: "address" },
            { name: "fee", type: "uint24" },
            { name: "recipient", type: "address" },
            { name: "amountIn", type: "uint256" },
            { name: "amountOutMinimum", type: "uint256" },
            { name: "sqrtPriceLimitX96", type: "uint160" },
          ],
        }],
        outputs: [{ name: "amountOut", type: "uint256" }],
        stateMutability: "payable",
      }],
      functionName: "exactInputSingle",
      args: [{
        tokenIn: CONTRACTS.WETH as `0x${string}`,
        tokenOut: tokenOut as `0x${string}`,
        fee: 10000, // 1% fee (matches PumpClaw pool)
        recipient: account.address,
        amountIn: value,
        amountOutMinimum: 0n, // No slippage protection for test
        sqrtPriceLimitX96: 0n,
      }],
    }),
  });

  console.log(`Tx: ${hash}`);
  const receipt = await publicClient.waitForTransactionReceipt({ hash });

  if (receipt.status === "success") {
    const balanceAfter = await publicClient.readContract({
      address: tokenOut as `0x${string}`,
      abi: ERC20_ABI,
      functionName: "balanceOf",
      args: [account.address],
    });
    
    const received = balanceAfter - balanceBefore;
    console.log(`✅ Received: ${formatUnits(received, 18)} tokens`);
  } else {
    console.log("❌ Swap failed");
  }
}

const token = process.argv[2];
const ethAmount = process.argv[3] || "0.0001";

if (!token) {
  console.log("Usage: swap.ts <token_address> [eth_amount]");
  process.exit(1);
}

swapExactETHForTokens(token, ethAmount);
