#!/usr/bin/env python3
"""
Todoist API client for OpenClaw.
Uses TODOIST_API_KEY environment variable for authentication.
API: https://developer.todoist.com/api/v1/
"""

import os
import sys
import json
import argparse
from urllib.request import Request, urlopen
from urllib.error import HTTPError, URLError
from urllib.parse import urlencode

API_BASE = "https://api.todoist.com/api/v1"

def get_auth_headers():
    """Get authorization headers with API token."""
    token = os.environ.get("TODOIST_API_KEY")
    if not token:
        print(json.dumps({"error": "TODOIST_API_KEY environment variable not set"}), file=sys.stderr)
        sys.exit(1)
    return {
        "Authorization": f"Bearer {token}",
        "Content-Type": "application/json"
    }

def make_request(endpoint, method="GET", data=None, params=None):
    """Make an API request to Todoist."""
    url = f"{API_BASE}{endpoint}"
    if params:
        url += "?" + urlencode(params)
    
    headers = get_auth_headers()
    
    try:
        req = Request(url, method=method, headers=headers)
        if data:
            req.data = json.dumps(data).encode("utf-8")
        
        with urlopen(req, timeout=30) as response:
            body = response.read().decode("utf-8")
            # Handle empty responses (e.g., 204 No Content from close/reopen)
            if not body:
                return {"success": True}
            return json.loads(body)
    except HTTPError as e:
        error_body = e.read().decode("utf-8")
        try:
            error_json = json.loads(error_body)
            print(json.dumps({"error": error_json}), file=sys.stderr)
        except:
            print(json.dumps({"error": f"HTTP {e.code}: {error_body}"}), file=sys.stderr)
        sys.exit(1)
    except URLError as e:
        print(json.dumps({"error": f"Request failed: {e.reason}"}), file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(json.dumps({"error": str(e)}), file=sys.stderr)
        sys.exit(1)

# --- Task Operations ---

def list_tasks(filter_str=None, project_id=None, label=None, priority=None, limit=50):
    """List tasks with optional filters."""
    from datetime import datetime
    import os
    
    params = {"limit": limit}
    if filter_str:
        params["filter"] = filter_str
    if project_id:
        params["project_id"] = project_id
    if label:
        params["label"] = label
    
    response = make_request("/tasks", params=params)
    tasks = response.get("results", []) if isinstance(response, dict) else response
    
    # Always filter out completed tasks
    tasks = [t for t in tasks if not t.get("checked", False)]
    
    # Client-side filter: when using "today" filter, API returns recurring tasks too
    # Filter to only tasks actually due today (respecting user's timezone)
    if filter_str and filter_str.lower() == "today":
        # Use US/Central timezone (from USER.md) or fallback to local
        tz = os.environ.get("TZ", "America/Chicago")
        try:
            from zoneinfo import ZoneInfo
            now = datetime.now(ZoneInfo(tz))
        except:
            now = datetime.now()
        today_str = now.strftime("%Y-%m-%d")
        tasks = [t for t in tasks if t.get("due", {}).get("date") == today_str]
    
    # Client-side priority filter since API doesn't support it directly
    if priority:
        tasks = [t for t in tasks if t.get("priority") == int(priority)]
    
    return tasks

def get_task(task_id):
    """Get a specific task by ID."""
    return make_request(f"/tasks/{task_id}")

def resolve_project(project_ref):
    """Resolve a project name or ID to a project ID."""
    if not project_ref:
        return None
    # If it looks like an ID (alphanumeric string), return as-is
    if len(project_ref) > 10 and all(c.isalnum() or c in '-_' for c in project_ref):
        return project_ref
    # Otherwise, look up by name (case-insensitive)
    projects = list_projects()
    if isinstance(projects, dict):
        projects = projects.get('results', [])
    project_ref_lower = project_ref.lower()
    for p in projects:
        if p.get('name', '').lower() == project_ref_lower:
            return p.get('id')
    # Not found, return original (will fail later with clearer error)
    return project_ref

def create_task(content, project_id=None, due_string=None, priority=None, labels=None, description=None, duration_amount=None, duration_unit=None):
    """Create a new task."""
    data = {"content": content}
    if project_id:
        data["project_id"] = project_id
    if due_string:
        data["due_string"] = due_string
    if priority:
        data["priority"] = int(priority)
    if labels:
        data["labels"] = labels if isinstance(labels, list) else [labels]
    if description:
        data["description"] = description
    if duration_amount is not None and duration_unit:
        # REST API v2 uses duration_amount and duration_unit as separate fields
        data["duration_amount"] = int(duration_amount)
        data["duration_unit"] = duration_unit
    
    return make_request("/tasks", method="POST", data=data)

def complete_task(task_id):
    """Mark a task as completed."""
    return make_request(f"/tasks/{task_id}/close", method="POST")

def uncomplete_task(task_id):
    """Reopen a completed task."""
    return make_request(f"/tasks/{task_id}/reopen", method="POST")

def update_task(task_id, content=None, due_string=None, priority=None, labels=None, description=None, duration_amount=None, duration_unit=None):
    """Update an existing task."""
    data = {}
    if content:
        data["content"] = content
    if due_string:
        data["due_string"] = due_string
    if priority:
        data["priority"] = int(priority)
    if labels:
        data["labels"] = labels if isinstance(labels, list) else [labels]
    if description is not None:
        data["description"] = description
    if duration_amount is not None and duration_unit:
        # REST API v2 uses duration_amount and duration_unit as separate fields
        data["duration_amount"] = int(duration_amount)
        data["duration_unit"] = duration_unit
    
    return make_request(f"/tasks/{task_id}", method="POST", data=data)

def delete_task(task_id):
    """Delete a task permanently."""
    make_request(f"/tasks/{task_id}", method="DELETE")
    return {"success": True, "message": f"Task {task_id} deleted"}

def create_task_with_duration_sync(content, due_string=None, duration_amount=None, duration_unit=None, project_id=None, priority=None):
    """Create a task using Sync API to support duration (workaround for REST API limitation)."""
    import uuid
    
    command = {
        "type": "item_add",
        "temp_id": str(uuid.uuid4()),
        "uuid": str(uuid.uuid4()),
        "args": {
            "content": content
        }
    }
    
    if due_string:
        command["args"]["due"] = {"string": due_string}
    if project_id:
        command["args"]["project_id"] = project_id
    if priority:
        command["args"]["priority"] = priority
    if duration_amount and duration_unit:
        command["args"]["duration"] = {"amount": int(duration_amount), "unit": duration_unit}
    
    # Use Sync API endpoint
    sync_url = "https://api.todoist.com/sync/v9/sync"
    headers = get_auth_headers()
    headers["Content-Type"] = "application/x-www-form-urlencoded"
    
    data = {
        "commands": json.dumps([command]),
        "sync_token": "*"
    }
    
    try:
        from urllib.parse import urlencode
        req = Request(sync_url, method="POST", headers=headers, data=urlencode(data).encode("utf-8"))
        with urlopen(req, timeout=30) as response:
            body = response.read().decode("utf-8")
            result = json.loads(body)
            
            # Check for errors
            cmd_uuid = command["uuid"]
            if "sync_status" in result and cmd_uuid in result["sync_status"]:
                status = result["sync_status"][cmd_uuid]
                if status == "ok":
                    # Return the created task info
                    return {
                        "success": True,
                        "content": content,
                        "due": due_string,
                        "duration": {"amount": duration_amount, "unit": duration_unit} if duration_amount else None
                    }
                else:
                    return {"error": status}
            return result
    except HTTPError as e:
        error_body = e.read().decode("utf-8")
        return {"error": f"HTTP {e.code}: {error_body}"}
    except Exception as e:
        return {"error": str(e)}

# --- Project Operations ---

def list_projects():
    """List all projects."""
    return make_request("/projects")

def get_project(project_id):
    """Get a specific project."""
    return make_request(f"/projects/{project_id}")

# --- Main CLI ---

def main():
    parser = argparse.ArgumentParser(description="Todoist CLI for OpenClaw")
    subparsers = parser.add_subparsers(dest="command", help="Commands")
    
    # List tasks
    list_parser = subparsers.add_parser("list", help="List tasks")
    list_parser.add_argument("--filter", "-f", help="Todoist filter string (e.g., 'today', 'overdue', 'p1')")
    list_parser.add_argument("--project", "-p", help="Project name or ID (e.g., 'Personal', 'Work')")
    list_parser.add_argument("--label", "-l", help="Label name")
    list_parser.add_argument("--priority", type=int, choices=[1, 2, 3, 4], help="Priority (1=urgent, 4=low)")
    list_parser.add_argument("--limit", type=int, default=50, help="Max results")
    
    # Get task
    get_parser = subparsers.add_parser("get", help="Get a specific task")
    get_parser.add_argument("task_id", help="Task ID")
    
    # Add task
    add_parser = subparsers.add_parser("add", help="Add a new task")
    add_parser.add_argument("content", help="Task content")
    add_parser.add_argument("--project", "-p", help="Project name or ID (e.g., 'Personal', 'Work')")
    add_parser.add_argument("--due", "-d", help="Due date (natural language: 'tomorrow', 'next Monday')")
    add_parser.add_argument("--priority", type=int, choices=[1, 2, 3, 4], default=4, help="Priority (1=urgent, 4=low)")
    add_parser.add_argument("--label", "-l", action="append", help="Label (can use multiple)")
    add_parser.add_argument("--description", help="Task description")
    add_parser.add_argument("--duration-amount", type=int, help="Duration amount (e.g., 30, 60)")
    add_parser.add_argument("--duration-unit", choices=["minute", "min", "day"], help="Duration unit (minute/min or day)")
    
    # Complete task
    complete_parser = subparsers.add_parser("complete", help="Complete a task")
    complete_parser.add_argument("task_id", help="Task ID")
    
    # Uncomplete task
    uncomplete_parser = subparsers.add_parser("uncomplete", help="Reopen a completed task")
    uncomplete_parser.add_argument("task_id", help="Task ID")
    
    # Update task
    update_parser = subparsers.add_parser("update", help="Update a task")
    update_parser.add_argument("task_id", help="Task ID")
    update_parser.add_argument("--content", "-c", help="New task content")
    update_parser.add_argument("--due", "-d", help="New due date")
    update_parser.add_argument("--priority", type=int, choices=[1, 2, 3, 4], help="Priority")
    update_parser.add_argument("--label", "-l", action="append", help="Labels (replaces existing)")
    update_parser.add_argument("--description", help="Task description")
    update_parser.add_argument("--duration-amount", type=int, help="Duration amount (e.g., 30, 60)")
    update_parser.add_argument("--duration-unit", choices=["minute", "min", "day"], help="Duration unit (minute/min or day)")
    
    # Delete task
    delete_parser = subparsers.add_parser("delete", help="Delete a task permanently")
    delete_parser.add_argument("task_id", help="Task ID")
    
    # List projects
    subparsers.add_parser("projects", help="List all projects")
    
    # Get project
    get_project_parser = subparsers.add_parser("project", help="Get a project")
    get_project_parser.add_argument("project_id", help="Project ID")
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    # Execute command
    result = None
    
    if args.command == "list":
        result = list_tasks(
            filter_str=args.filter,
            project_id=resolve_project(args.project),
            label=args.label,
            priority=args.priority,
            limit=args.limit
        )
    elif args.command == "get":
        result = get_task(args.task_id)
    elif args.command == "add":
        # Use Sync API if duration is specified (REST API has issues with duration)
        if args.duration_amount and args.duration_unit:
            result = create_task_with_duration_sync(
                content=args.content,
                due_string=args.due,
                duration_amount=args.duration_amount,
                duration_unit=args.duration_unit,
                project_id=resolve_project(args.project),
                priority=args.priority
            )
        else:
            result = create_task(
                content=args.content,
                project_id=resolve_project(args.project),
                due_string=args.due,
                priority=args.priority,
                labels=args.label,
                description=args.description
            )
    elif args.command == "complete":
        result = complete_task(args.task_id)
    elif args.command == "uncomplete":
        result = uncomplete_task(args.task_id)
    elif args.command == "update":
        result = update_task(
            task_id=args.task_id,
            content=args.content,
            due_string=args.due,
            priority=args.priority,
            labels=args.label,
            description=args.description,
            duration_amount=args.duration_amount,
            duration_unit=args.duration_unit
        )
    elif args.command == "delete":
        result = delete_task(args.task_id)
    elif args.command == "projects":
        result = list_projects()
    elif args.command == "project":
        result = get_project(args.project_id)
    
    if result is not None:
        print(json.dumps(result, indent=2))

if __name__ == "__main__":
    main()
