#!/usr/bin/env python3
"""
TikTok Android Bot - Main Script

Two modes of operation:

1. SEARCH MODE - Search specific topics and comment on related videos
   python3 tiktok_bot.py search --topics fitness,gaming --videos 5

2. EXPLORE MODE - Scroll through For You feed and comment on random videos
   python3 tiktok_bot.py explore --videos 10

Usage Examples:
  # Search for fitness videos and comment on 5 of them
  python3 tiktok_bot.py search --topics fitness --videos 5
  
  # Search multiple topics, 3 videos each
  python3 tiktok_bot.py search --topics "fitness,cooking,travel" --videos 3
  
  # Explore For You feed, comment on 10 random videos
  python3 tiktok_bot.py explore --videos 10
  
  # Use specific device (optional)
  python3 tiktok_bot.py search --topics gaming --device 001431538002547
"""

import sys
import os
import subprocess
import time
import random
import argparse
from datetime import datetime

sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from src.bot.android.tiktok_android_bot import TikTokAndroidBot
from src.bot.android.tiktok_navigation import TikTokNavigation

# Check if config exists
if not os.path.exists("config.py"):
    print("\n" + "="*60)
    print("⚠️  Configuration not found!")
    print("="*60)
    print("\nThis is your first time running the bot.")
    print("Let's set it up!\n")
    
    response = input("Run interactive setup now? [Y/n]: ").strip().lower()
    if response in ['', 'y', 'yes']:
        print("\nStarting setup wizard...\n")
        os.system("python3 setup.py")
        if not os.path.exists("config.py"):
            print("\n❌ Setup was not completed. Exiting.")
            sys.exit(1)
        print("\n✅ Setup complete! Starting bot...\n")
    else:
        print("\n❌ Cannot run without configuration.")
        print("   Run: python3 setup.py")
        sys.exit(1)

# Import config
try:
    from config import TOPICS, COMMENT_STYLE
    
    # Import based on comment style
    if COMMENT_STYLE == "static":
        from config import COMMENTS_BY_TOPIC, GENERIC_COMMENTS
    else:  # AI mode
        from config import AI_PROVIDER, AI_MODEL, AI_COMMENT_PROMPT, GENERIC_COMMENTS
        from src.ai_comments import generate_ai_comment
        
except ImportError as e:
    print(f"❌ Error loading config: {e}")
    print("   Run: python3 setup.py")
    sys.exit(1)


def get_device_id(specified_device=None):
    """Get device ID from argument, environment, or auto-detect."""
    if specified_device:
        return specified_device
    
    device_id = os.environ.get("ANDROID_DEVICE_ID")
    if device_id:
        return device_id
    
    # Auto-detect first connected device
    result = subprocess.run(["adb", "devices"], capture_output=True, text=True)
    devices = [line.split()[0] for line in result.stdout.split("\n")[1:] 
               if line.strip() and "device" in line]
    
    if not devices:
        print("❌ No Android device found!")
        print("   1. Connect device via USB")
        print("   2. Enable USB debugging")
        print("   3. Authorize computer on device")
        sys.exit(1)
    
    return devices[0]


def get_comment_static(topic, used_comments):
    """Get a static comment from templates."""
    if topic not in COMMENTS_BY_TOPIC:
        # Fall back to generic comments
        available = [c for c in GENERIC_COMMENTS if c not in used_comments]
    else:
        available = [c for c in COMMENTS_BY_TOPIC[topic] if c not in used_comments]
    
    if not available:
        # All used, reset
        used_comments.clear()
        available = COMMENTS_BY_TOPIC.get(topic, GENERIC_COMMENTS)
    
    comment = random.choice(available)
    used_comments.add(comment)
    return comment


def get_comment_ai(screenshot_path, topic):
    """Generate comment using AI vision."""
    try:
        comment = generate_ai_comment(
            screenshot_path=screenshot_path,
            topic=topic,
            provider=AI_PROVIDER,
            model=AI_MODEL,
            prompt_template=AI_COMMENT_PROMPT
        )
        
        if comment:
            return comment
        else:
            # Fall back to generic
            return random.choice(GENERIC_COMMENTS)
    except Exception as e:
        print(f"    ⚠️  AI failed, using fallback: {e}")
        return random.choice(GENERIC_COMMENTS)


def get_comment(topic, screenshot_path=None, used_comments=None):
    """Get comment based on configured style."""
    if COMMENT_STYLE == "ai":
        if not screenshot_path:
            print("    ⚠️  No screenshot for AI, using fallback")
            return random.choice(GENERIC_COMMENTS)
        return get_comment_ai(screenshot_path, topic)
    else:  # static
        if used_comments is None:
            used_comments = set()
        return get_comment_static(topic, used_comments)


def search_mode(bot, nav, topics, videos_per_topic, device_id):
    """Search specific topics and comment on videos."""
    print("\n" + "="*60)
    print("🔍 SEARCH MODE")
    print("="*60)
    print(f"Topics: {', '.join(topics)}")
    print(f"Videos per topic: {videos_per_topic}")
    print(f"Device: {device_id}")
    print("="*60)
    
    all_results = []
    
    for topic_idx, topic in enumerate(topics, 1):
        print(f"\n📍 TOPIC {topic_idx}/{len(topics)}: {topic.upper()}")
        print("-" * 60)
        
        used_comments = set()
        commented_videos = set()
        scroll_count = 0
        
        # Navigate and search
        bot.launch_tiktok()
        bot.wait_for_feed()
        nav.go_to_home()
        time.sleep(1)
        nav.tap_search_icon()
        nav.search_query(topic)
        
        # Engage with videos
        for video_num in range(1, videos_per_topic + 1):
            print(f"\n  📹 Video {video_num}/{videos_per_topic}")
            
            try:
                position = ((video_num - 1) % 4) + 1
                
                # Scroll if needed
                if video_num > 4 and (video_num - 1) % 4 == 0:
                    scroll_count += 1
                    print(f"    📜 Scrolling for more videos...")
                    bot.scroll_down()
                    time.sleep(2)
                
                video_id = f"s{scroll_count}_p{position}"
                
                if video_id in commented_videos:
                    print(f"    ⏭️  Already commented, skipping")
                    continue
                
                # Open video
                nav.tap_video_from_grid(position)
                time.sleep(2)
                
                # Take screenshot for AI analysis
                screenshot_path = f"data/search_{topic}_v{video_num}.png"
                bot.take_screenshot(screenshot_path)
                
                # Generate comment (AI or static)
                if COMMENT_STYLE == "ai":
                    print(f"    🤖 Analyzing video with AI...")
                    comment = get_comment(topic, screenshot_path=screenshot_path)
                else:
                    comment = get_comment(topic, used_comments=used_comments)
                
                print(f"    💬 {comment}")
                
                success = bot.post_comment(comment)
                commented_videos.add(video_id)
                
                if success:
                    print(f"    ✅ Posted!")
                    all_results.append({
                        "topic": topic,
                        "video_num": video_num,
                        "comment": comment,
                        "success": True
                    })
                else:
                    print(f"    ❌ Failed")
                
                # Go back
                if video_num < videos_per_topic:
                    bot.go_back()
                    time.sleep(2)
                    
            except Exception as e:
                print(f"    ❌ Error: {e}")
        
        # Pause between topics
        if topic_idx < len(topics):
            print(f"\n⏸️  Pausing 10s before next topic...")
            time.sleep(10)
    
    return all_results


def explore_mode(bot, nav, num_videos, device_id):
    """Explore For You feed and comment on random videos."""
    print("\n" + "="*60)
    print("🌟 EXPLORE MODE (For You Feed)")
    print("="*60)
    print(f"Videos to engage: {num_videos}")
    print(f"Device: {device_id}")
    print("="*60)
    
    results = []
    used_comments = set()
    
    # Launch TikTok
    bot.launch_tiktok()
    bot.wait_for_feed()
    time.sleep(2)
    
    for video_num in range(1, num_videos + 1):
        print(f"\n📹 Video {video_num}/{num_videos}")
        
        try:
            # Take screenshot
            screenshot_path = f"data/explore_v{video_num}.png"
            bot.take_screenshot(screenshot_path)
            
            # Generate comment (AI or generic)
            if COMMENT_STYLE == "ai":
                print(f"  🤖 Analyzing video with AI...")
                comment = get_comment("general", screenshot_path=screenshot_path)
            else:
                comment = get_comment(None, used_comments=used_comments)
            
            print(f"  💬 {comment}")
            
            # Post comment
            success = bot.post_comment(comment)
            
            if success:
                print(f"  ✅ Posted!")
                results.append({
                    "video_num": video_num,
                    "comment": comment,
                    "success": True
                })
            else:
                print(f"  ❌ Failed")
            
            # Scroll to next video
            if video_num < num_videos:
                print(f"  ⏭️  Scrolling to next video...")
                bot.scroll_down()
                time.sleep(3)
                
        except Exception as e:
            print(f"  ❌ Error: {e}")
    
    return results


def print_summary(results, mode):
    """Print session summary."""
    print("\n" + "="*60)
    print("📊 SESSION SUMMARY")
    print("="*60)
    
    success_count = len([r for r in results if r.get("success")])
    total = len(results)
    
    print(f"\nMode: {mode.upper()}")
    print(f"✅ Success: {success_count}/{total} videos")
    
    if mode == "search" and results:
        # Group by topic
        by_topic = {}
        for r in results:
            topic = r.get("topic", "unknown")
            if topic not in by_topic:
                by_topic[topic] = []
            by_topic[topic].append(r)
        
        print(f"\nBy topic:")
        for topic, items in by_topic.items():
            success = len([i for i in items if i.get("success")])
            print(f"  {topic}: {success}/{len(items)}")
    
    print("\nSample comments:")
    for r in results[:5]:
        if r.get("success"):
            print(f"  • {r['comment'][:60]}...")
    
    print("="*60)


def main():
    parser = argparse.ArgumentParser(
        description="TikTok Android Bot - Automated commenting",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  Search mode (specific topics):
    %(prog)s search --topics fitness --videos 5
    %(prog)s search --topics "fitness,gaming,cooking" --videos 3
  
  Explore mode (For You feed):
    %(prog)s explore --videos 10
    %(prog)s explore --videos 5 --device 001431538002547
        """
    )
    
    subparsers = parser.add_subparsers(dest="mode", help="Operation mode")
    
    # Search mode
    search_parser = subparsers.add_parser("search", help="Search topics and comment")
    search_parser.add_argument("--topics", required=True, 
                              help="Comma-separated topics to search (e.g., fitness,gaming)")
    search_parser.add_argument("--videos", type=int, default=5,
                              help="Videos per topic (default: 5)")
    search_parser.add_argument("--device", help="Device ID (optional, auto-detects if not specified)")
    
    # Explore mode
    explore_parser = subparsers.add_parser("explore", help="Explore For You feed")
    explore_parser.add_argument("--videos", type=int, default=10,
                               help="Number of videos to comment on (default: 10)")
    explore_parser.add_argument("--device", help="Device ID (optional, auto-detects if not specified)")
    
    args = parser.parse_args()
    
    if not args.mode:
        parser.print_help()
        sys.exit(1)
    
    # Get device
    device_id = get_device_id(args.device)
    print(f"\n📱 Using device: {device_id}")
    
    # Initialize bot
    bot = TikTokAndroidBot(device_id=device_id)
    nav = TikTokNavigation(bot)
    
    # Run mode
    if args.mode == "search":
        topics = [t.strip() for t in args.topics.split(",")]
        results = search_mode(bot, nav, topics, args.videos, device_id)
    else:  # explore
        results = explore_mode(bot, nav, args.videos, device_id)
    
    # Print summary
    print_summary(results, args.mode)


if __name__ == "__main__":
    main()
