#!/usr/bin/env python3
"""TikTok Android Bot - Mobile automation for TikTok comment posting."""

import subprocess
import time
import random
import re
from typing import Optional, Dict, List
from loguru import logger as log


class TikTokAndroidBot:
    """Android-based TikTok comment bot using ADB."""

    def __init__(self, device_id: str = "emulator-5554"):
        """Initialize Android bot.

        Args:
            device_id: ADB device ID (e.g., "emulator-5554" or physical device serial)
        """
        self.device_id = device_id
        self.tiktok_package = "com.zhiliaoapp.musically"  # TikTok package name

        log.info(f"[Android Bot] Initializing for device: {device_id}")

        # Verify device is connected
        if not self._is_device_connected():
            raise RuntimeError(f"Device {device_id} not connected via ADB")

        log.info(f"[Android Bot] Device {device_id} connected and ready")

    def _is_device_connected(self) -> bool:
        """Check if device is connected via ADB."""
        try:
            result = subprocess.run(
                ["adb", "-s", self.device_id, "get-state"],
                capture_output=True,
                text=True,
                timeout=5
            )
            return result.stdout.strip() == "device"
        except Exception as e:
            log.error(f"[Android Bot] Device check failed: {e}")
            return False

    def _adb_shell(self, command: str, timeout: int = 10) -> str:
        """Execute ADB shell command.

        Args:
            command: Shell command to execute
            timeout: Command timeout in seconds

        Returns:
            Command output
        """
        try:
            result = subprocess.run(
                ["adb", "-s", self.device_id, "shell", command],
                capture_output=True,
                text=True,
                timeout=timeout
            )
            return result.stdout.strip()
        except subprocess.TimeoutExpired:
            log.warning(f"[Android Bot] Command timed out: {command}")
            return ""
        except Exception as e:
            log.error(f"[Android Bot] ADB command failed: {e}")
            return ""

    def _tap(self, x: int, y: int, add_randomness: bool = True):
        """Tap at coordinates with optional randomness.

        Args:
            x: X coordinate
            y: Y coordinate
            add_randomness: Add slight random offset to appear more human
        """
        if add_randomness:
            x += random.randint(-5, 5)
            y += random.randint(-5, 5)

        self._adb_shell(f"input tap {x} {y}")
        log.debug(f"[Android Bot] Tapped at ({x}, {y})")
        self._human_delay()

    def _swipe(self, x1: int, y1: int, x2: int, y2: int, duration_ms: int = 300):
        """Swipe from one point to another.

        Args:
            x1, y1: Start coordinates
            x2, y2: End coordinates
            duration_ms: Swipe duration in milliseconds
        """
        self._adb_shell(f"input swipe {x1} {y1} {x2} {y2} {duration_ms}")
        log.debug(f"[Android Bot] Swiped from ({x1},{y1}) to ({x2},{y2})")
        self._human_delay()

    def _type_text(self, text: str):
        """Type text using ADB input.

        Args:
            text: Text to type (will escape special characters)
        """
        # Remove emojis and special characters that cause issues
        # Keep only alphanumeric, spaces, and basic punctuation
        import re
        # Remove emojis
        text_clean = re.sub(r'[^\w\s\.,!?\'-]', '', text, flags=re.UNICODE)

        # Escape special characters for shell
        escaped = text_clean.replace(" ", "%s").replace("'", "").replace('"', '')

        if not escaped:
            log.warning(f"[Android Bot] Text became empty after cleaning: {text}")
            return

        self._adb_shell(f"input text {escaped}")
        log.debug(f"[Android Bot] Typed: {text_clean[:30]}...")
        self._human_delay(0.5, 1.5)

    def _press_key(self, keycode: str):
        """Press Android keycode.

        Args:
            keycode: Android keycode (e.g., "KEYCODE_ENTER", "KEYCODE_BACK")
        """
        self._adb_shell(f"input keyevent {keycode}")
        log.debug(f"[Android Bot] Pressed key: {keycode}")
        self._human_delay()

    def _human_delay(self, min_sec: float = 0.3, max_sec: float = 1.0):
        """Random human-like delay.

        Args:
            min_sec: Minimum delay in seconds
            max_sec: Maximum delay in seconds
        """
        delay = random.uniform(min_sec, max_sec)
        time.sleep(delay)

    def _get_screen_size(self) -> tuple[int, int]:
        """Get device screen size.

        Returns:
            (width, height) tuple
        """
        output = self._adb_shell("wm size")
        match = re.search(r'(\d+)x(\d+)', output)
        if match:
            return int(match.group(1)), int(match.group(2))
        return (1080, 1920)  # Default size

    def install_tiktok(self, apk_path: Optional[str] = None) -> bool:
        """Install TikTok APK on device.

        Args:
            apk_path: Path to TikTok APK file (optional)

        Returns:
            True if installed successfully
        """
        try:
            if apk_path:
                log.info(f"[Android Bot] Installing TikTok from {apk_path}...")
                result = subprocess.run(
                    ["adb", "-s", self.device_id, "install", "-r", apk_path],
                    capture_output=True,
                    text=True,
                    timeout=120
                )
                if "Success" in result.stdout:
                    log.info("[Android Bot] ✓ TikTok installed successfully")
                    return True
                else:
                    log.error(f"[Android Bot] Installation failed: {result.stdout}")
                    return False
            else:
                log.info("[Android Bot] TikTok APK path not provided, assuming app is already installed")
                return self.is_tiktok_installed()
        except Exception as e:
            log.error(f"[Android Bot] Installation error: {e}")
            return False

    def is_tiktok_installed(self) -> bool:
        """Check if TikTok is installed."""
        output = self._adb_shell(f"pm list packages | grep {self.tiktok_package}")
        return self.tiktok_package in output

    def launch_tiktok(self) -> bool:
        """Launch TikTok app.

        Returns:
            True if launched successfully
        """
        try:
            log.info("[Android Bot] Launching TikTok...")
            # Launch main activity
            self._adb_shell(f"am start -n {self.tiktok_package}/com.ss.android.ugc.aweme.splash.SplashActivity")
            time.sleep(5)  # Wait for app to load

            # Verify app is in foreground
            output = self._adb_shell("dumpsys window windows | grep mCurrentFocus")
            if self.tiktok_package in output:
                log.info("[Android Bot] ✓ TikTok launched successfully")
                return True
            else:
                log.warning("[Android Bot] TikTok may not be in foreground")
                return True  # Return True anyway, might just be slow
        except Exception as e:
            log.error(f"[Android Bot] Failed to launch TikTok: {e}")
            return False

    def wait_for_feed(self) -> bool:
        """Wait for TikTok feed to load (For You page).

        Returns:
            True if feed loaded
        """
        try:
            log.info("[Android Bot] Waiting for For You feed to load...")
            # TikTok opens directly to For You feed with video already playing
            time.sleep(3)
            log.info("[Android Bot] ✓ For You feed ready (video playing)")
            return True
        except Exception as e:
            log.error(f"[Android Bot] Feed loading failed: {e}")
            return False

    def skip_to_next_video(self, count: int = 1) -> bool:
        """Skip to next video(s) by swiping up.

        Args:
            count: Number of videos to skip

        Returns:
            True if skipped successfully
        """
        try:
            log.info(f"[Android Bot] Skipping {count} video(s)...")

            for i in range(count):
                # Swipe up to go to next video
                self.scroll_down()  # scroll_down swipes up (next video)
                time.sleep(2)  # Wait for video to load

            log.info(f"[Android Bot] ✓ Skipped to video #{count}")
            return True
        except Exception as e:
            log.error(f"[Android Bot] Failed to skip video: {e}")
            return False

    def post_comment(self, comment_text: str) -> bool:
        """Post a comment on current video.

        Args:
            comment_text: Comment to post

        Returns:
            True if comment posted successfully
        """
        try:
            log.info(f"[Android Bot] Posting comment: {comment_text[:50]}...")

            # Get screen size
            width, height = self._get_screen_size()

            # Step 1: Tap on comment icon (RIGHT SIDE of screen, middle area)
            # TikTok mobile has icons stacked vertically on the right:
            # Like, Comment, Share, Profile - Comment is usually 2nd from bottom
            comment_icon_x = int(width * 0.95)  # Far right edge
            comment_icon_y = int(height * 0.70)  # Lower area (70% from top)

            log.debug("[Android Bot] Tapping comment icon on right side...")
            self._tap(comment_icon_x, comment_icon_y)
            time.sleep(2)  # Wait for comment drawer to slide up

            # Step 2: Tap on "Add comment..." input field
            # Comment drawer appears from bottom with input field at the very bottom
            comment_input_x = int(width * 0.4)  # Center-left where text appears
            comment_input_y = int(height * 0.93)  # Very bottom

            log.debug("[Android Bot] Tapping comment input field...")
            self._tap(comment_input_x, comment_input_y)
            time.sleep(1)  # Wait for keyboard to appear

            # Dismiss any popups (like stylus tutorial)
            self._dismiss_popups()
            time.sleep(0.5)

            # Step 3: Type comment text
            log.debug("[Android Bot] Typing comment...")
            self._type_text(comment_text)
            time.sleep(1)

            # Step 4: Dismiss keyboard (so Post button is at consistent position)
            log.debug("[Android Bot] Dismissing keyboard...")
            self._press_key("KEYCODE_BACK")
            time.sleep(1)  # Wait for keyboard to hide

            # Step 5: Submit comment (Post/Send button with keyboard DISMISSED)
            # With keyboard dismissed, button is at consistent distance from bottom
            # Tested and verified position on device 1080x2392
            send_button_x = int(width * 0.92)  # 92% from left edge (right side)
            # Calculate Y as fixed distance from BOTTOM (not percentage from top)
            pixels_from_bottom = 130  # Post button is 130px from bottom (verified)
            send_button_y = height - pixels_from_bottom

            log.debug(f"[Android Bot] Tapping Post button at ({send_button_x}, {send_button_y}) - {pixels_from_bottom}px from bottom...")
            self._tap(send_button_x, send_button_y)
            time.sleep(2)  # Wait for comment to post

            log.info("[Android Bot] ✓ Comment posted successfully!")
            return True

        except Exception as e:
            log.error(f"[Android Bot] Comment posting failed: {e}")
            return False

    def go_back(self):
        """Press back button."""
        self._press_key("KEYCODE_BACK")
        time.sleep(1)

    def _dismiss_popups(self):
        """Dismiss any popups that might be blocking interaction."""
        width, height = self._get_screen_size()

        # Tap "Cancel" button for stylus popup (bottom center-left)
        cancel_x = int(width * 0.30)  # Center-left where "Cancel" button is
        cancel_y = int(height * 0.93)  # Bottom

        log.debug(f"[Android Bot] Tapping Cancel button at ({cancel_x}, {cancel_y})")
        self._tap(cancel_x, cancel_y, add_randomness=False)
        time.sleep(1)

        log.debug("[Android Bot] Dismissed popups")

    def scroll_down(self):
        """Scroll down (swipe up)."""
        width, height = self._get_screen_size()
        start_y = int(height * 0.7)
        end_y = int(height * 0.3)
        x = int(width * 0.5)

        self._swipe(x, start_y, x, end_y, duration_ms=random.randint(200, 400))

    def take_screenshot(self, save_path: str) -> bool:
        """Take screenshot and save to file.

        Args:
            save_path: Local path to save screenshot

        Returns:
            True if screenshot saved successfully
        """
        try:
            # Take screenshot on device
            device_path = "/sdcard/screenshot.png"
            self._adb_shell(f"screencap -p {device_path}")

            # Pull screenshot to local machine
            subprocess.run(
                ["adb", "-s", self.device_id, "pull", device_path, save_path],
                capture_output=True,
                timeout=10
            )

            # Clean up device
            self._adb_shell(f"rm {device_path}")

            log.info(f"[Android Bot] Screenshot saved: {save_path}")
            return True
        except Exception as e:
            log.error(f"[Android Bot] Screenshot failed: {e}")
            return False

    def close_app(self):
        """Force close TikTok app."""
        log.info("[Android Bot] Closing TikTok...")
        self._adb_shell(f"am force-stop {self.tiktok_package}")
        time.sleep(1)
