"""
AI comment generation using Claude Vision or OpenAI
"""

import os
import base64
from anthropic import Anthropic
from openai import OpenAI


def generate_comment_anthropic(screenshot_path: str, topic: str, model: str, prompt_template: str) -> str:
    """Generate comment using Claude Vision."""
    api_key = os.environ.get("ANTHROPIC_API_KEY")
    if not api_key:
        raise ValueError("ANTHROPIC_API_KEY not set in environment")
    
    client = Anthropic(api_key=api_key)
    
    # Read and encode image
    with open(screenshot_path, "rb") as f:
        image_data = base64.standard_b64encode(f.read()).decode("utf-8")
    
    prompt = prompt_template.replace("{topic}", topic)
    
    message = client.messages.create(
        model=model,
        max_tokens=200,
        messages=[
            {
                "role": "user",
                "content": [
                    {
                        "type": "image",
                        "source": {
                            "type": "base64",
                            "media_type": "image/png",
                            "data": image_data
                        }
                    },
                    {
                        "type": "text",
                        "text": prompt
                    }
                ]
            }
        ]
    )
    
    return message.content[0].text.strip()


def generate_comment_openai(screenshot_path: str, topic: str, model: str, prompt_template: str) -> str:
    """Generate comment using OpenAI Vision."""
    api_key = os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("OPENAI_API_KEY not set in environment")
    
    client = OpenAI(api_key=api_key)
    
    # Read and encode image
    with open(screenshot_path, "rb") as f:
        image_data = base64.standard_b64encode(f.read()).decode("utf-8")
    
    prompt = prompt_template.replace("{topic}", topic)
    
    response = client.chat.completions.create(
        model=model,
        max_tokens=200,
        messages=[
            {
                "role": "user",
                "content": [
                    {
                        "type": "text",
                        "text": prompt
                    },
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/png;base64,{image_data}"
                        }
                    }
                ]
            }
        ]
    )
    
    return response.choices[0].message.content.strip()


def generate_comment_openrouter(screenshot_path: str, topic: str, model: str, prompt_template: str) -> str:
    """Generate comment using OpenRouter."""
    api_key = os.environ.get("OPENROUTER_API_KEY")
    if not api_key:
        raise ValueError("OPENROUTER_API_KEY not set in environment")
    
    client = OpenAI(
        api_key=api_key,
        base_url="https://openrouter.ai/api/v1"
    )
    
    # Read and encode image
    with open(screenshot_path, "rb") as f:
        image_data = base64.standard_b64encode(f.read()).decode("utf-8")
    
    prompt = prompt_template.replace("{topic}", topic)
    
    response = client.chat.completions.create(
        model=model,
        max_tokens=200,
        messages=[
            {
                "role": "user",
                "content": [
                    {
                        "type": "text",
                        "text": prompt
                    },
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/png;base64,{image_data}"
                        }
                    }
                ]
            }
        ]
    )
    
    return response.choices[0].message.content.strip()


def generate_ai_comment(screenshot_path: str, topic: str, provider: str, model: str, prompt_template: str) -> str:
    """Generate comment using configured AI provider."""
    try:
        if provider == "anthropic":
            return generate_comment_anthropic(screenshot_path, topic, model, prompt_template)
        elif provider == "openai":
            return generate_comment_openai(screenshot_path, topic, model, prompt_template)
        elif provider == "openrouter":
            return generate_comment_openrouter(screenshot_path, topic, model, prompt_template)
        else:
            raise ValueError(f"Unknown provider: {provider}")
    except Exception as e:
        print(f"    ⚠️  AI generation failed: {e}")
        return None
