#!/usr/bin/env python3
"""
Interactive setup wizard for TikTok Android Bot

Runs on first use to configure topics, comment style, and AI settings.
"""

import os
import sys
import json

CONFIG_FILE = "config.py"
SETTINGS_FILE = ".bot_settings.json"


def print_header():
    print("\n" + "="*60)
    print("🤖 TikTok Android Bot - Setup Wizard")
    print("="*60)


def print_section(title):
    print(f"\n{'─'*60}")
    print(f"  {title}")
    print(f"{'─'*60}")


def ask_choice(question, options, default=None):
    """Ask user to choose from options."""
    print(f"\n{question}")
    for i, opt in enumerate(options, 1):
        print(f"  {i}. {opt}")
    
    while True:
        if default:
            choice = input(f"\nChoice [1-{len(options)}] (default: {default}): ").strip()
            if not choice:
                return default
        else:
            choice = input(f"\nChoice [1-{len(options)}]: ").strip()
        
        if choice.isdigit() and 1 <= int(choice) <= len(options):
            return int(choice)
        print("❌ Invalid choice. Try again.")


def ask_text(question, default=None):
    """Ask user for text input."""
    if default:
        result = input(f"\n{question} (default: {default}): ").strip()
        return result if result else default
    else:
        while True:
            result = input(f"\n{question}: ").strip()
            if result:
                return result
            print("❌ This field is required.")


def ask_yes_no(question, default=True):
    """Ask yes/no question."""
    default_str = "Y/n" if default else "y/N"
    while True:
        answer = input(f"\n{question} [{default_str}]: ").strip().lower()
        if not answer:
            return default
        if answer in ['y', 'yes']:
            return True
        if answer in ['n', 'no']:
            return False
        print("❌ Please answer yes or no.")


def ask_topics():
    """Ask user for topics to engage with."""
    print_section("📍 Step 1: Topics")
    print("\nWhat topics do you want to engage with?")
    print("Examples: fitness, cooking, travel, technology, gaming")
    print("You can add as many as you want (comma-separated)")
    
    topics_input = ask_text("Enter your topics (comma-separated)")
    topics = [t.strip() for t in topics_input.split(",") if t.strip()]
    
    print(f"\n✓ Topics configured: {', '.join(topics)}")
    return topics


def ask_comment_style():
    """Ask user how they want to generate comments."""
    print_section("💬 Step 2: Comment Style")
    
    style = ask_choice(
        "How do you want to generate comments?",
        [
            "Static templates - Predefined comment templates (fast, no API needed)",
            "AI-generated - Use AI to analyze videos and create comments (smarter, requires API key)"
        ],
        default=1
    )
    
    return "static" if style == 1 else "ai"


def ask_ai_config():
    """Ask for AI configuration."""
    print_section("🤖 Step 3: AI Configuration")
    
    print("\nWhich AI provider do you want to use?")
    provider = ask_choice(
        "Choose provider:",
        [
            "Anthropic (Claude) - Recommended for vision + text",
            "OpenAI (GPT-4) - Good alternative",
            "OpenRouter - Access multiple models"
        ],
        default=1
    )
    
    provider_map = {1: "anthropic", 2: "openai", 3: "openrouter"}
    provider_name = provider_map[provider]
    
    # Ask for API key
    print(f"\n📝 Enter your {provider_name.upper()} API key")
    print("(This will be stored in .env file, not committed to git)")
    
    api_key = ask_text(f"API key")
    
    # Model selection
    models = {
        "anthropic": ["claude-3-5-sonnet-20241022", "claude-3-opus-20240229"],
        "openai": ["gpt-4o", "gpt-4-turbo"],
        "openrouter": ["anthropic/claude-3.5-sonnet", "openai/gpt-4o"]
    }
    
    print(f"\n🎯 Recommended models for {provider_name}:")
    for m in models[provider_name]:
        print(f"  • {m}")
    
    model = ask_text(f"Model name", default=models[provider_name][0])
    
    return {
        "provider": provider_name,
        "api_key": api_key,
        "model": model
    }


def ask_static_comments(topics):
    """Ask user for static comment templates."""
    print_section("💬 Step 3: Comment Templates")
    
    print("\nFor each topic, provide 6-8 comment variations.")
    print("These should be natural, engaging questions or statements.")
    print("\n💡 Tips:")
    print("  • Ask questions to encourage replies")
    print("  • Be specific to the topic")
    print("  • Keep 10-25 words")
    print("  • No emojis (sounds more genuine)")
    
    comments_by_topic = {}
    
    for topic in topics:
        print(f"\n📍 Topic: {topic}")
        print("Enter comments one by one (empty line when done, minimum 6):")
        
        comments = []
        while True:
            idx = len(comments) + 1
            comment = input(f"  {idx}. ").strip()
            
            if not comment:
                if len(comments) >= 6:
                    break
                else:
                    print(f"     (Need at least {6 - len(comments)} more)")
                    continue
            
            comments.append(comment)
            
            if len(comments) >= 8:
                if not ask_yes_no("Add more comments?", default=False):
                    break
        
        comments_by_topic[topic] = comments
        print(f"✓ Added {len(comments)} comments for '{topic}'")
    
    return comments_by_topic


def save_config(topics, comment_style, static_comments=None, ai_config=None):
    """Save configuration to config.py."""
    print_section("💾 Saving Configuration")
    
    config_content = f'''"""
TikTok Android Bot Configuration
Generated by setup wizard
"""

# Topics to engage with
TOPICS = {topics}

# Comment generation style: "static" or "ai"
COMMENT_STYLE = "{comment_style}"
'''
    
    if comment_style == "static":
        config_content += f'''
# Static comment templates by topic
COMMENTS_BY_TOPIC = {json.dumps(static_comments, indent=4)}

# Generic comments for explore mode
GENERIC_COMMENTS = [
    "This is great content! Keep it up!",
    "Really enjoyed this! More please!",
    "Awesome work! Very informative!",
    "Love this! Where can I learn more?",
    "Impressive! How long did this take?",
    "Quality content! Following for more!",
    "Well done! What inspired this?",
    "This is exactly what I needed! Thanks!",
]
'''
    else:  # AI
        config_content += f'''
# AI Configuration
AI_PROVIDER = "{ai_config['provider']}"
AI_MODEL = "{ai_config['model']}"

# Comment generation prompt
AI_COMMENT_PROMPT = """
Analyze this video screenshot and generate a natural, engaging comment.
The video is related to: {{topic}}

Guidelines:
- 10-25 words
- Ask a question or make a specific observation
- Sound like a genuine enthusiast
- No emojis
- Be encouraging and positive
"""

# Generic comments as fallback (if AI fails)
GENERIC_COMMENTS = [
    "This is great content! Keep it up!",
    "Really enjoyed this! More please!",
    "Awesome work! Very informative!",
]
'''
    
    with open(CONFIG_FILE, 'w') as f:
        f.write(config_content)
    
    print(f"✓ Saved to {CONFIG_FILE}")
    
    # Save AI API key to .env if AI mode
    if comment_style == "ai" and ai_config:
        env_content = ""
        if os.path.exists(".env"):
            with open(".env", 'r') as f:
                env_content = f.read()
        
        key_name = {
            "anthropic": "ANTHROPIC_API_KEY",
            "openai": "OPENAI_API_KEY",
            "openrouter": "OPENROUTER_API_KEY"
        }[ai_config['provider']]
        
        # Remove existing key if present
        lines = [l for l in env_content.split('\n') if not l.startswith(f"{key_name}=")]
        lines.append(f"{key_name}={ai_config['api_key']}")
        
        with open(".env", 'w') as f:
            f.write('\n'.join(lines))
        
        print(f"✓ Saved API key to .env")


def save_settings(topics, comment_style, ai_config=None):
    """Save settings to JSON for later reference."""
    settings = {
        "topics": topics,
        "comment_style": comment_style,
        "setup_complete": True
    }
    
    if ai_config:
        settings["ai_provider"] = ai_config["provider"]
        settings["ai_model"] = ai_config["model"]
    
    with open(SETTINGS_FILE, 'w') as f:
        json.dump(settings, f, indent=2)


def main():
    print_header()
    
    # Check if already configured
    if os.path.exists(CONFIG_FILE):
        print("\n⚠️  Configuration already exists!")
        if not ask_yes_no("Run setup again (will overwrite)?", default=False):
            print("\n👋 Setup cancelled. Use existing config.")
            return
    
    # Run setup steps
    topics = ask_topics()
    comment_style = ask_comment_style()
    
    if comment_style == "ai":
        ai_config = ask_ai_config()
        save_config(topics, comment_style, ai_config=ai_config)
        save_settings(topics, comment_style, ai_config)
    else:
        static_comments = ask_static_comments(topics)
        save_config(topics, comment_style, static_comments=static_comments)
        save_settings(topics, comment_style)
    
    # Summary
    print_section("✅ Setup Complete!")
    print(f"\nTopics: {', '.join(topics)}")
    print(f"Comment style: {comment_style.upper()}")
    
    print("\n🚀 Next steps:")
    print("1. Review your config.py file")
    print("2. Run the bot:")
    if comment_style == "static":
        print(f"   python3 tiktok_bot.py search --topics {topics[0]} --videos 5")
    else:
        print(f"   python3 tiktok_bot.py search --topics {topics[0]} --videos 5")
    print("\n3. Schedule daily runs with OpenClaw cron")
    
    print("\n" + "="*60)


if __name__ == "__main__":
    main()
