#!/usr/bin/env python3
"""Complete TikTok campaign across all car/racing topics."""

import sys
sys.path.insert(0, '/Users/mladjanantic/.openclaw/workspace/androidSkill')

from src.bot.android.tiktok_android_bot import TikTokAndroidBot
from src.bot.android.tiktok_navigation import TikTokNavigation
import time
import random
import os
import subprocess
from datetime import datetime

# All car/racing topics
TOPICS = ["dragy", "laptimer", "circuit", "acceleration", "trackday"]

# Expanded comment library with more variety
COMMENTS_BY_TOPIC = {
    "dragy": [
        "That 60ft time is insane! What mods are you running?",
        "Dragy never lies! What was the trap speed on this run?",
        "Those are some serious numbers! Street tires or slicks?",
        "Love seeing actual data! What's your best quarter mile?",
        "That launch is perfect! What power are you making?",
        "GPS timing is the real deal! Stock turbo?",
        "Those splits are clean! Track prepped or street?",
        "Impressive consistency! How many runs did you do?",
    ],
    "laptimer": [
        "That sector time is incredible! What track is this?",
        "Sub-2-minute lap! What's your setup?",
        "Those times are getting better! Track day soon?",
        "Love seeing the progression! What tires?",
        "That's a fast lap! More seat time paying off?",
        "Impressive! What suspension mods helped most?",
        "Clean lap! Aiming for any specific time?",
        "Great consistency! How many laps in the session?",
    ],
    "circuit": [
        "That racing line through turn 3 is perfect!",
        "Circuit looks fast! Which track is this?",
        "That apex speed is impressive! What car?",
        "Love the commitment through that corner!",
        "Smooth and fast! What's your lap record here?",
        "Those elevation changes look challenging!",
        "Beautiful track! Planning to go back soon?",
        "Clean line! More track days this season?",
    ],
    "acceleration": [
        "That launch was perfect! AWD or RWD?",
        "0-60 in how many seconds? Looks fast!",
        "Clean pull! What power are you making?",
        "That acceleration is brutal! Turbo or supercharged?",
        "Love the sound! Stock exhaust?",
        "Wheelspin control is on point! Launch control?",
        "That pull is nasty! What's the setup?",
        "Impressive! Street or race fuel?",
    ],
    "trackday": [
        "Track day vibes! Which circuit?",
        "That looks like a perfect day! More sessions coming?",
        "Love the track content! What times did you hit?",
        "Great footage! What group were you running with?",
        "Perfect weather for it! How many sessions?",
        "Track days are the best! Same car every time?",
        "Looks like a blast! Any instructors out there?",
        "Awesome! Planning your next track day already?",
    ]
}

def get_unique_comment(topic, used_comments):
    """Get a comment that hasn't been used yet."""
    available = [c for c in COMMENTS_BY_TOPIC[topic] if c not in used_comments]
    if not available:
        # All used, reset
        available = COMMENTS_BY_TOPIC[topic]
        used_comments.clear()
    
    comment = random.choice(available)
    used_comments.add(comment)
    return comment


def run_topic_session(bot, nav, topic, num_videos=5):
    """Run engagement session for one topic."""
    print("\n" + "="*60)
    print(f"📍 TOPIC: {topic.upper()}")
    print("="*60)
    
    session_start = time.time()
    results = []
    used_comments = set()
    commented_videos = set()  # Track videos we've already commented on
    scroll_count = 0  # Track how many times we've scrolled
    
    try:
        # Navigate and search
        print("\n1️⃣  Launching...")
        bot.launch_tiktok()
        bot.wait_for_feed()
        
        print("2️⃣  Going to Home...")
        nav.go_to_home()
        time.sleep(1)
        
        print("3️⃣  Opening search...")
        nav.tap_search_icon()
        
        print(f"4️⃣  Searching '{topic}'...")
        nav.search_query(topic)
        
        bot.take_screenshot(f"data/campaign_{topic}_results.png")
        print(f"✓ Search results loaded")
        
        # Engage with videos
        attempts = 0
        max_attempts = num_videos * 3  # Safety limit
        
        while len(results) < num_videos and attempts < max_attempts:
            attempts += 1
            video_num = len(results) + 1
            
            print(f"\n📹 VIDEO {video_num}/{num_videos} (attempt {attempts})")
            print("-" * 40)
            
            try:
                # Calculate grid position (1-4)
                position = ((attempts - 1) % 4) + 1
                
                # Scroll if we've checked all 4 positions in current view
                if attempts > 4 and (attempts - 1) % 4 == 0:
                    scroll_count += 1
                    print(f"  📜 Scrolling for fresh videos (scroll #{scroll_count})...")
                    bot.scroll_down()
                    time.sleep(2)  # Wait for new videos to load
                
                # Create unique video identifier: scroll_count + position
                video_id = f"s{scroll_count}_p{position}"
                
                if video_id in commented_videos:
                    print(f"  ⏭️  Skipping - already commented on this position")
                    continue
                
                print(f"  Tapping video #{position} from grid (ID: {video_id})...")
                
                # Take screenshot of grid before opening
                grid_screenshot = f"data/grid_{topic}_{video_id}_before.png"
                bot.take_screenshot(grid_screenshot)
                
                nav.tap_video_from_grid(position)
                
                screenshot_path = f"data/campaign_{topic}_v{video_num}_{video_id}.png"
                bot.take_screenshot(screenshot_path)
                
                # Generate unique comment
                comment = get_unique_comment(topic, used_comments)
                print(f"  💬 Comment: {comment}")
                
                # Post comment
                video_start = time.time()
                success = bot.post_comment(comment)
                video_time = time.time() - video_start
                
                if success:
                    print(f"  ✅ Posted! ({video_time:.1f}s)")
                    
                    # Mark this video as commented
                    commented_videos.add(video_id)
                    
                    results.append({
                        "video_num": video_num,
                        "video_id": video_id,
                        "position": position,
                        "scroll": scroll_count,
                        "comment": comment,
                        "success": True,
                        "time": video_time,
                        "timestamp": datetime.now().strftime("%H:%M:%S")
                    })
                else:
                    print(f"  ❌ Failed - will skip this position")
                    commented_videos.add(video_id)  # Mark as tried to avoid retry
                    results.append({
                        "video_num": video_num,
                        "video_id": video_id,
                        "success": False
                    })
                
                # Go back to search results for next video
                if len(results) < num_videos:
                    print(f"  ↩️  Going back to search results...")
                    bot.go_back()
                    time.sleep(2)
                    
            except Exception as e:
                print(f"  ❌ Error: {e}")
                
                # Mark as tried to avoid retrying same video
                commented_videos.add(video_id)
                
                results.append({
                    "video_num": video_num,
                    "video_id": video_id,
                    "success": False,
                    "error": str(e)
                })
                
                # Try to go back
                try:
                    bot.go_back()
                    time.sleep(2)
                except:
                    print("  ⚠️  Could not go back, will continue...")
        
        if len(results) < num_videos:
            print(f"\n⚠️  Only completed {len([r for r in results if r.get('success')])}/{num_videos} videos")
        
    except Exception as e:
        print(f"\n❌ Topic session failed: {e}")
        import traceback
        traceback.print_exc()
    
    session_time = time.time() - session_start
    
    return {
        "topic": topic,
        "results": results,
        "total_time": session_time,
        "success_count": len([r for r in results if r.get("success")]),
        "total_videos": num_videos
    }


def main():
    print("\n" + "="*60)
    print("🚀 FULL TIKTOK CAMPAIGN")
    print("="*60)
    print(f"Topics: {', '.join(TOPICS)}")
    print(f"Videos per topic: 5")
    print(f"Total videos: {len(TOPICS) * 5}")
    print("="*60)
    
    # Get device ID from environment or use first connected device
    device_id = os.environ.get("ANDROID_DEVICE_ID")
    if not device_id:
        # Auto-detect first connected device
        result = subprocess.run(["adb", "devices"], capture_output=True, text=True)
        devices = [line.split()[0] for line in result.stdout.split("\n")[1:] if line.strip() and "device" in line]
        if not devices:
            print("❌ No Android device found. Connect device and enable USB debugging.")
            return
        device_id = devices[0]
        print(f"📱 Using device: {device_id}")
    
    bot = TikTokAndroidBot(device_id=device_id)
    nav = TikTokNavigation(bot)
    
    campaign_start = time.time()
    all_sessions = []
    
    # Run session for each topic
    for i, topic in enumerate(TOPICS, 1):
        print(f"\n\n{'='*60}")
        print(f"TOPIC {i}/{len(TOPICS)}")
        print(f"{'='*60}")
        
        session = run_topic_session(bot, nav, topic, num_videos=5)
        all_sessions.append(session)
        
        # Brief pause between topics
        if i < len(TOPICS):
            print(f"\n⏸️  Pausing 10s before next topic...")
            time.sleep(10)
    
    campaign_time = time.time() - campaign_start
    
    # Generate report
    print("\n\n" + "="*60)
    print("📊 CAMPAIGN COMPLETE")
    print("="*60)
    
    total_success = sum(s["success_count"] for s in all_sessions)
    total_videos = sum(s["total_videos"] for s in all_sessions)
    
    print(f"\n✅ Total videos engaged: {total_success}/{total_videos}")
    print(f"⏱️  Total time: {campaign_time/60:.1f} minutes")
    print(f"⚡ Average per video: {campaign_time/total_success:.1f} seconds")
    
    report = {
        "campaign_time": campaign_time,
        "total_success": total_success,
        "total_videos": total_videos,
        "sessions": all_sessions
    }
    
    # Print detailed results
    print("\n" + "="*60)
    print("DETAILED RESULTS")
    print("="*60)
    
    for session in all_sessions:
        topic = session["topic"]
        success = session["success_count"]
        total = session["total_videos"]
        print(f"\n📍 {topic.upper()}: {success}/{total} videos")
        
        for result in session["results"]:
            if result.get("success"):
                video_id = result.get('video_id', 'unknown')
                print(f"  ✅ Video {result['video_num']} ({video_id}): {result['comment'][:45]}...")
    
    return report


if __name__ == "__main__":
    report = main()
