#!/usr/bin/env python3
"""Complete TikTok engagement session - working version!"""

import sys
sys.path.insert(0, '/Users/mladjanantic/.openclaw/workspace/androidSkill')

from src.bot.android.tiktok_android_bot import TikTokAndroidBot
from src.bot.android.tiktok_navigation import TikTokNavigation
import time
import random
import os
import subprocess
from datetime import datetime

# Car/racing topics
TOPICS = ["dragy", "laptimer", "circuit", "racebox", "acceleration", "trackday"]

def main():
    print("\n" + "="*60)
    print("🚀 COMPLETE TIKTOK ENGAGEMENT SESSION")
    print("="*60)
    
    # Get device ID from environment or use first connected device
    device_id = os.environ.get("ANDROID_DEVICE_ID")
    if not device_id:
        # Auto-detect first connected device
        result = subprocess.run(["adb", "devices"], capture_output=True, text=True)
        devices = [line.split()[0] for line in result.stdout.split("\n")[1:] if line.strip() and "device" in line]
        if not devices:
            print("❌ No Android device found. Connect device and enable USB debugging.")
            return
        device_id = devices[0]
        print(f"📱 Using device: {device_id}")
    
    bot = TikTokAndroidBot(device_id=device_id)
    nav = TikTokNavigation(bot)
    
    session_report = []
    num_videos = 3  # Start with 3 videos
    
    # Pick random topic
    topic = random.choice(TOPICS)
    print(f"\n📍 Search topic: {topic}")
    print("="*60)
    
    # Step 1: Launch and navigate to search
    print("\n1️⃣  Launching TikTok...")
    bot.launch_tiktok()
    bot.wait_for_feed()
    
    print("2️⃣  Going to Home...")
    nav.go_to_home()
    time.sleep(1)
    
    print("3️⃣  Opening search...")
    nav.tap_search_icon()
    
    print(f"4️⃣  Searching for '{topic}'...")
    nav.search_query(topic)
    
    # Take screenshot of search results
    print("5️⃣  Capturing search results...")
    bot.take_screenshot(f"data/session_search_results_{topic}.png")
    print(f"✓ Search results showing for: {topic}")
    
    # Loop through videos
    for video_num in range(1, num_videos + 1):
        print("\n" + "="*60)
        print(f"📹 VIDEO {video_num}/{num_videos}")
        print("="*60)
        
        try:
            # Tap video from grid (positions 1-4)
            position = ((video_num - 1) % 4) + 1
            print(f"\n  Tapping video #{position} from grid...")
            nav.tap_video_from_grid(position)
            
            # Take screenshot of video
            screenshot_path = f"data/session_video_{video_num}.png"
            bot.take_screenshot(screenshot_path)
            print(f"  ✓ Screenshot: {screenshot_path}")
            
            # Analyze video (placeholder - you'll use Claude Vision here)
            print(f"\n  💭 Analyzing video...")
            video_analysis = {
                "topic": f"{topic} related content",
                "screenshot": screenshot_path
            }
            
            # Generate comment based on topic
            print(f"  💬 Generating comment...")
            comments_by_topic = {
                "dragy": [
                    "That 60ft time is insane! What mods are you running?",
                    "Dragy never lies! What was the trap speed on this run?",
                    "Those are some serious numbers! Street tires or slicks?"
                ],
                "laptimer": [
                    "That sector time is incredible! What track is this?",
                    "Sub-2-minute lap! What's your setup?",
                    "Those times are getting better! Track day soon?"
                ],
                "circuit": [
                    "That racing line through turn 3 is perfect!",
                    "Circuit looks fast! Which track is this?",
                    "That apex speed is impressive! What car?"
                ],
                "racebox": [
                    "Racebox data looks solid! What's your best 0-60?",
                    "Those GPS numbers are accurate! Nice run!",
                    "Racebox Pro or Mini? Love the data!"
                ],
                "acceleration": [
                    "That launch was perfect! AWD or RWD?",
                    "0-60 in how many seconds? Looks fast!",
                    "Clean pull! What power are you making?"
                ],
                "trackday": [
                    "Track day vibes! Which circuit?",
                    "That looks like a perfect day! More sessions coming?",
                    "Love the track content! What times did you hit?"
                ]
            }
            
            comment = random.choice(comments_by_topic.get(topic, [
                "Amazing content! Love the car stuff!",
                "That's impressive! What's the setup?",
                "Clean run! More videos coming?"
            ]))
            
            print(f"  ✓ Comment: {comment}")
            
            # Post comment
            print(f"\n  📝 Posting comment...")
            success = bot.post_comment(comment)
            
            if success:
                print(f"  ✅ Comment posted!")
                
                # Log for report
                session_report.append({
                    "video_num": video_num,
                    "topic": topic,
                    "position": position,
                    "comment": comment,
                    "screenshot": screenshot_path,
                    "success": True,
                    "timestamp": datetime.now().strftime("%H:%M:%S")
                })
            else:
                print(f"  ❌ Comment failed")
                session_report.append({
                    "video_num": video_num,
                    "topic": topic,
                    "success": False
                })
            
            # Go back to search results for next video
            if video_num < num_videos:
                print(f"\n  ↩️  Returning to search results...")
                bot.go_back()
                time.sleep(2)
                
                # If we used all 4 visible videos, scroll down for more
                if video_num % 4 == 0:
                    print(f"  📜 Scrolling for more videos...")
                    bot.scroll_down()
                    time.sleep(1)
        
        except Exception as e:
            print(f"\n  ❌ Error on video {video_num}: {e}")
            import traceback
            traceback.print_exc()
            # Try to recover by going back
            bot.go_back()
            time.sleep(2)
    
    # Session complete
    print("\n" + "="*60)
    print("✅ SESSION COMPLETE")
    print("="*60)
    
    # Print report
    print(f"\n📊 SUMMARY:")
    print(f"  Topic: {topic}")
    print(f"  Videos engaged: {len([r for r in session_report if r.get('success')])}/{num_videos}")
    
    for item in session_report:
        if item.get('success'):
            print(f"\n  Video {item['video_num']}:")
            print(f"    Grid position: #{item['position']}")
            print(f"    Comment: {item['comment']}")
            print(f"    Time: {item['timestamp']}")
    
    print("\n" + "="*60)
    return session_report


if __name__ == "__main__":
    report = main()
