---
name: tiktok-android-bot
description: Automate TikTok engagement on Android using ADB. Search topics, comment with AI or templates, includes setup wizard. Use for TikTok automation campaigns and building social presence through strategic commenting.
---

# TikTok Android Bot

Automate TikTok engagement on Android using ADB. No web scraping, no CAPTCHA, 100% success rate.

## What It Does

- **Interactive Setup** - Wizard guides first-time configuration
- **Two Comment Modes** - Static templates (fast) or AI-generated (smart)
- **Two Operation Modes** - Search specific topics or explore For You feed
- **Duplicate Prevention** - Never comments twice on same video
- **Flexible Configuration** - User defines topics and comment style

## Prerequisites

- Android device with USB debugging enabled
- ADB (Android Debug Bridge) installed
- TikTok app logged in on device
- Python 3.9+
- USB cable

## First-Time Setup

The skill includes an interactive setup wizard that runs automatically:

```bash
python3 tiktok_bot.py search --topics fitness --videos 5
```

**Or run setup manually:**

```bash
python3 setup.py
```

The wizard asks:

1. **Topics** - What to engage with (e.g., "fitness,cooking,travel")
2. **Comment Style:**
   - **Static** - Predefined templates (fast, free, no API)
   - **AI** - Claude/GPT Vision analyzes videos (smart, ~$0.01-0.05/comment)
3. **Configuration:**
   - Static: Enter 6-8 comment variations per topic
   - AI: Choose provider (Anthropic/OpenAI/OpenRouter) + API key

Setup saves to `config.py` and `.env` (both gitignored).

## Usage

### Search Mode - Target Specific Topics

Search for topics and comment on related videos:

```bash
# Single topic, 5 videos
python3 tiktok_bot.py search --topics fitness --videos 5

# Multiple topics, 3 videos each  
python3 tiktok_bot.py search --topics "fitness,cooking,travel" --videos 3

# Specify device (optional)
python3 tiktok_bot.py search --topics gaming --videos 5 --device 001431538002547
```

**Flow:**
1. Searches each topic
2. Opens videos from search results grid (2x2 layout)
3. Generates comment (AI analyzes or uses template)
4. Posts comment
5. Returns to search results for next video

### Explore Mode - For You Feed

Comment on random videos from For You feed:

```bash
# Comment on 10 random videos
python3 tiktok_bot.py explore --videos 10
```

**Flow:**
1. Starts on For You feed
2. Analyzes current video (if AI) or uses generic comment
3. Posts comment
4. Scrolls to next video

## Comment Styles

### Static Templates

Fast, reliable, no API costs. User provides 6-8 variations per topic.

**Example config:**
```python
COMMENT_STYLE = "static"

COMMENTS_BY_TOPIC = {
    "fitness": [
        "That form looks perfect! What's your workout routine?",
        "Impressive progress! How long training?",
        # ... more variations
    ]
}
```

### AI-Generated

Claude Vision or GPT-4 Vision analyzes video screenshots and generates contextual comments.

**Example config:**
```python
COMMENT_STYLE = "ai"
AI_PROVIDER = "anthropic"
AI_MODEL = "claude-3-5-sonnet-20241022"
```

**API key in .env:**
```bash
ANTHROPIC_API_KEY=sk-ant-...
```

**Cost:** $0.01-0.05 per comment depending on provider.

## Configuration Files

After setup, you'll have:

- `config.py` - Topics, comment style, templates/AI settings
- `.env` - API key (if AI mode)
- `.bot_settings.json` - Preferences

All gitignored by default.

## Device Setup

### Enable USB Debugging

```
Settings → About Phone → Tap "Build Number" 7 times
Settings → Developer Options → Enable "USB Debugging"
```

Connect device via USB and authorize computer.

### Verify Connection

```bash
adb devices
# Should show: <device_id>  device

adb shell wm size
# Note screen resolution (e.g., 1080x2392)
```

## Troubleshooting

### "No Android device found"

```bash
adb kill-server
adb start-server
adb devices
```

Re-authorize on device if needed.

### Search icon tap misses

Coordinates are optimized for 1080x2392 screens. For different sizes:

1. Take screenshot: `adb shell screencap -p /sdcard/screen.png && adb pull /sdcard/screen.png`
2. Find search icon pixel location (top-right)
3. Update in `src/bot/android/tiktok_navigation.py`:
   ```python
   search_icon_x = 995  # Your X
   search_icon_y = 205  # Your Y
   ```

See `references/COORDINATES.md` for detailed coordinate guide.

### AI generation fails

Check:
1. API key in `.env` file
2. API key is valid and has credits
3. Model name is correct
4. Falls back to generic comments automatically

### Post button not working

Ensure keyboard is dismissed before tapping Post. The bot does this automatically with `KEYCODE_BACK`.

## Performance

### Timing

- **Static mode:** ~25 seconds per video
- **AI mode:** ~30 seconds per video (adds 5s for analysis)
- **Full search session (5 videos):** 2-2.5 minutes
- **Explore session (10 videos):** 4-5 minutes

### Success Rate

- **100%** with working coordinates
- **0%** if tap coordinates miss targets

### Cost (AI Mode)

- Claude Vision: $0.01-0.02 per comment
- GPT-4 Vision: $0.02-0.05 per comment
- 100 comments: $1-5

## Best Practices

### Comment Quality

✅ **Good:**
- Specific observations or questions
- 10-25 words
- Genuine enthusiasm
- No emojis (sounds more real)

❌ **Bad:**
- Generic praise ("nice video!")
- Spam or self-promotion
- Too short ("first!")
- Low-value ("🔥🔥🔥")

### Rate Limits

- **25-30 comments/day max** per account
- **Space sessions:** Once daily, vary times
- **Take breaks:** Skip 1-2 days per week
- **Monitor:** Watch for shadowban signs

### Account Safety

- **Age accounts:** 7+ days before automating
- **Manual activity first:** Like, follow, browse naturally
- **Vary behavior:** Different topics, times, comment styles
- **Start small:** Test with 3-5 videos first

## Advanced

### Scheduling with OpenClaw Cron

```bash
openclaw cron add \
  --name "Daily TikTok" \
  --schedule "0 10 * * *" \
  --tz "Your/Timezone" \
  --payload '{"kind":"agentTurn","message":"cd /path/to/skill && python3 tiktok_bot.py search --topics fitness,gaming --videos 5"}'
```

### Custom AI Prompt

Edit `config.py`:

```python
AI_COMMENT_PROMPT = """
Analyze this video and generate a comment.
Topic: {topic}

Your custom guidelines here...
- Be enthusiastic
- Ask specific questions
- Reference visible elements
"""
```

### Multiple Devices

Set `ANDROID_DEVICE_ID` environment variable:

```bash
ANDROID_DEVICE_ID=device1 python3 tiktok_bot.py search --topics fitness --videos 5
```

Or use `--device` flag:

```bash
python3 tiktok_bot.py search --topics fitness --videos 5 --device device1
```

## Files Included

```
tiktok-android-bot/
├── SKILL.md                    # This file
├── README.md                   # Comprehensive docs
├── setup.py                    # Interactive setup wizard
├── tiktok_bot.py              # Main script (CLI)
├── config.example.py          # Example configuration
├── requirements.txt           # Python dependencies
├── scripts/
│   ├── run_full_campaign.py   # Legacy: 25-video campaign
│   └── run_complete_session.py # Legacy: 3-video session
├── src/
│   ├── bot/android/
│   │   ├── tiktok_android_bot.py    # Core automation
│   │   └── tiktok_navigation.py     # Navigation flows
│   ├── ai_comments.py         # AI comment generation
│   └── logger.py              # Logging utility
└── references/
    └── COORDINATES.md         # Tap coordinate guide
```

## Requirements

```
loguru>=0.7.0
anthropic>=0.18.0  # If using AI mode
openai>=1.12.0     # If using AI mode
```

ADB must be installed and in PATH.

## License

MIT - Use responsibly. Automated commenting may violate TikTok's ToS.

## See Also

- `README.md` - Full documentation
- `references/COORDINATES.md` - Coordinate customization guide
- Main repository: https://github.com/mladjan/androidSkill
