#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
思维模型增强器 - 核心处理引擎
Thinking Model Enhancer - Core Processing Engine

实现多阶段认知处理管道，支持问题分析、模型选择、信息处理、决策制定和记忆集成。
"""

import re
import json
from pathlib import Path
from datetime import datetime
from typing import Dict, List, Any, Optional, Tuple
from dataclasses import dataclass, field
from enum import Enum
import hashlib


class ProblemType(Enum):
    """问题类型枚举"""
    SKILL_CREATION = "skill_creation"      # 技能创建
    SYSTEM_REPAIR = "system_repair"        # 系统修复
    GENERAL_DECISION = "general_decision"  # 一般决策
    COMPLEX_ANALYSIS = "complex_analysis"  # 复杂分析
    RESEARCH = "research"                  # 研究调查
    UNKNOWN = "unknown"                    # 未知


class ConfidenceLevel(Enum):
    """置信度等级"""
    HIGH = "high"      # >90%
    MEDIUM = "medium"  # 60-90%
    LOW = "low"        # <60%


class UrgencyLevel(Enum):
    """紧急程度等级"""
    P0_CRITICAL = "P0"  # 关键 - 服务宕机
    P1_HIGH = "P1"      # 高 - 主要功能受损
    P2_MEDIUM = "P2"    # 中 - 次要问题
    P3_LOW = "P3"       # 低 - 可安排修复


@dataclass
class ProblemAnalysis:
    """问题分析结果"""
    original_input: str
    problem_type: ProblemType
    complexity: int  # 1-10
    keywords: List[str]
    constraints: List[str]
    urgency: Optional[UrgencyLevel] = None
    confidence: ConfidenceLevel = ConfidenceLevel.MEDIUM
    
    def to_dict(self) -> Dict:
        return {
            "original_input": self.original_input,
            "problem_type": self.problem_type.value,
            "complexity": self.complexity,
            "keywords": self.keywords,
            "constraints": self.constraints,
            "urgency": self.urgency.value if self.urgency else None,
            "confidence": self.confidence.value
        }


@dataclass
class ThinkingModel:
    """思维模型"""
    model_id: str
    name: str
    description: str
    stages: List[str]
    priority: int  # 优先级，数字越小越高
    适用场景: List[str]
    source: str = "custom"
    version: str = "1.0"
    success_rate: float = 0.5
    last_used: Optional[str] = None
    use_count: int = 0
    
    def to_dict(self) -> Dict:
        return {
            "model_id": self.model_id,
            "name": self.name,
            "description": self.description,
            "stages": self.stages,
            "priority": self.priority,
            "适用场景": self.适用场景,
            "source": self.source,
            "version": self.version,
            "success_rate": self.success_rate,
            "last_used": self.last_used,
            "use_count": self.use_count
        }


@dataclass
class ThinkingResult:
    """思维处理结果"""
    problem: str
    selected_model: str
    stages_completed: List[str]
    findings: List[str]
    recommendations: List[str]
    confidence: ConfidenceLevel
    processing_time_ms: float
    timestamp: str
    memory_worthy: bool = True
    
    def to_dict(self) -> Dict:
        return {
            "problem": self.problem,
            "selected_model": self.selected_model,
            "stages_completed": self.stages_completed,
            "findings": self.findings,
            "recommendations": self.recommendations,
            "confidence": self.confidence.value,
            "processing_time_ms": self.processing_time_ms,
            "timestamp": self.timestamp,
            "memory_worthy": self.memory_worthy
        }


class ThinkingModelCore:
    """思维模型核心处理引擎"""
    
    def __init__(self, memory_dir: Optional[str] = None):
        """
        初始化思维模型核心
        
        Args:
            memory_dir: 记忆目录路径
        """
        self.memory_dir = memory_dir or Path.home() / ".claude" / "thinking_models"
        self.memory_dir.mkdir(parents=True, exist_ok=True)
        
        # 初始化内置思维模型
        self._init_builtin_models()
        
        # 问题类型检测关键词
        self._init_keyword_patterns()
    
    def _init_builtin_models(self):
        """初始化内置思维模型"""
        self.builtin_models = {
            "research_mode": ThinkingModel(
                model_id="research_mode",
                name="研究型思维模式",
                description="用于创建新功能或技能的5步研究流程",
                stages=[
                    "Memory Query - 记忆查询",
                    "Documentation Access - 文档访问",
                    "Public Research - 公开研究",
                    "Best Practices - 最佳实践",
                    "Solution Fusion - 方案融合"
                ],
                priority=1,
                适用场景=["skill_creation", "research", "feature_development"]
            ),
            
            "diagnostic_mode": ThinkingModel(
                model_id="diagnostic_mode",
                name="诊断型思维模式",
                description="用于系统故障排除的6步诊断流程",
                stages=[
                    "Memory Pattern Match - 记忆模式匹配",
                    "Problem Understanding - 问题理解",
                    "Official Solution Search - 官方方案搜索",
                    "Tool/Skill Match - 工具/技能匹配",
                    "Community Solutions - 社区方案",
                    "Last Resort Fix - 最后方案修复"
                ],
                priority=2,
                适用场景=["system_repair", "troubleshooting", "error_diagnosis"]
            ),
            
            "generic_pipeline": ThinkingModel(
                model_id="generic_pipeline",
                name="通用认知处理管道",
                description="通用的7步认知处理流程",
                stages=[
                    "Problem Analysis - 问题分析",
                    "Model Selection - 模型选择",
                    "Information Collection - 信息收集",
                    "Analysis & Evaluation - 分析与评估",
                    "Synthesis - 综合",
                    "Decision Formulation - 决策制定",
                    "Memory Integration - 记忆集成"
                ],
                priority=10,
                适用场景=["general_decision", "complex_analysis"]
            )
        }
    
    def _init_keyword_patterns(self):
        """初始化关键词模式用于问题类型检测"""
        self.keyword_patterns = {
            ProblemType.SKILL_CREATION: [
                "写skill", "创建技能", "实现功能", "写一个让它",
                "add skill", "create skill", "implement feature"
            ],
            
            ProblemType.SYSTEM_REPAIR: [
                "启动失败", "报错", "错误", "修复", "问题",
                "启动不了", "崩了", "故障", "repair", "troubleshoot",
                "error", "fix", "issue", "bug"
            ],
            
            ProblemType.RESEARCH: [
                "研究", "调查", "分析", "比较", "评估",
                "research", "investigate", "analyze", "compare"
            ],
            
            ProblemType.COMPLEX_ANALYSIS: [
                "考虑", "全面分析",
                "complex", "comprehensive", "multiple factors"
            ]
        }
    
    def analyze_problem(self, user_input: str) -> ProblemAnalysis:
        """
        分析用户输入，确定问题类型和复杂度
        
        Args:
            user_input: 用户输入
            
        Returns:
            ProblemAnalysis 对象
        """
        # 提取关键词
        keywords = self._extract_keywords(user_input)
        
        # 检测问题类型
        problem_type = self._detect_problem_type(user_input)
        
        # 评估复杂度 (1-10)
        complexity = self._assess_complexity(user_input)
        
        # 提取约束条件
        constraints = self._extract_constraints(user_input)
        
        # 检测紧急程度（如果是系统问题）
        urgency = self._detect_urgency(user_input) if problem_type == ProblemType.SYSTEM_REPAIR else None
        
        # 评估置信度
        confidence = self._assess_confidence(user_input, problem_type)
        
        return ProblemAnalysis(
            original_input=user_input,
            problem_type=problem_type,
            complexity=complexity,
            keywords=keywords,
            constraints=constraints,
            urgency=urgency,
            confidence=confidence
        )
    
    def _extract_keywords(self, text: str) -> List[str]:
        """从文本中提取关键词"""
        # 移除常见停用词
        stopwords = {"的", "是", "在", "和", "与", "或", "了", "我", "你", "他", "她", "它", "这个", "那个", "什么", "如何", "怎么"}
        
        # 分词并过滤
        words = re.findall(r'[\w\u4e00-\u9fff]+', text)
        keywords = [w for w in words if len(w) >= 2 and w not in stopwords]
        
        return list(set(keywords))[:10]  # 最多10个关键词
    
    def _detect_problem_type(self, user_input: str) -> ProblemType:
        """检测问题类型"""
        user_input_lower = user_input.lower()
        
        # 按优先级检测
        for problem_type, patterns in self.keyword_patterns.items():
            for pattern in patterns:
                if pattern.lower() in user_input_lower:
                    return problem_type
        
        # 默认返回通用决策
        return ProblemType.GENERAL_DECISION
    
    def _assess_complexity(self, user_input: str) -> int:
        """评估问题复杂度 (1-10)"""
        complexity_indicators = [
            (r'\n|；|。', 1),  # 多句话
            (r'并且|而且|同时|以及', 1),  # 并列关系
            (r'但是|然而|不过|可是', 1),  # 转折关系
            (r'如果|假设|假如|要是', 1),  # 条件关系
            (r'\?', 0.5),  # 问句
            (r'\d+', 0.5),  # 包含数字
            (r'必须|应该|需要|一定要', 0.5),  # 强调
        ]
        
        score = 1
        for pattern, weight in complexity_indicators:
            if re.search(pattern, user_input):
                score += weight
        
        return min(10, max(1, int(score)))
    
    def _extract_constraints(self, user_input: str) -> List[str]:
        """提取约束条件"""
        constraints = []
        
        # 时间约束
        time_patterns = [
            (r'今天|今日', '时间: 今天'),
            (r'明天|明日', '时间: 明天'),
            (r'本周|这周', '时间: 本周'),
            (r'紧急|尽快|马上', '优先级: 高'),
            (r'不急|慢慢|有时间', '优先级: 低'),
        ]
        
        for pattern, constraint in time_patterns:
            if re.search(pattern, user_input):
                constraints.append(constraint)
        
        return constraints
    
    def _detect_urgency(self, user_input: str) -> UrgencyLevel:
        """检测紧急程度"""
        urgency_patterns = {
            UrgencyLevel.P0_CRITICAL: [
                "宕机", "完全不能", "崩溃了", "服务挂了",
                "down", "crash", "completely broken"
            ],
            UrgencyLevel.P1_HIGH: [
                "启动失败", "主要功能坏了", "严重影响",
                "major issue", "can't start", "critical"
            ],
            UrgencyLevel.P2_MEDIUM: [
                "有点问题", "小问题", "偶尔出错",
                "minor issue", "sometimes", "occasionally"
            ],
            UrgencyLevel.P3_LOW: [
                "想优化", "想改进", "建议",
                "optimize", "improve", "suggestion"
            ]
        }
        
        user_input_lower = user_input.lower()
        
        for urgency, patterns in urgency_patterns.items():
            for pattern in patterns:
                if pattern.lower() in user_input_lower:
                    return urgency
        
        return UrgencyLevel.P2_MEDIUM  # 默认中等
    
    def _assess_confidence(self, user_input: str, problem_type: ProblemType) -> ConfidenceLevel:
        """评估分析置信度"""
        # 基于信息完整度评估
        info_indicators = [
            (r'具体|详细|完整', 0.2),  # 提供详细信息
            (r'大概|可能|也许', -0.2),  # 信息不确定
            (r'\d+', 0.1),  # 包含具体数据
            (r'错误|error|报错|log', 0.15),  # 提供错误详情
        ]
        
        score = 0.7  # 基础分数
        
        for pattern, weight in info_indicators:
            if re.search(pattern, user_input):
                score += weight
        
        # 根据问题类型调整
        if problem_type == ProblemType.SYSTEM_REPAIR:
            if len(user_input) < 20:
                score -= 0.2  # 信息太少
            if "错误" in user_input or "error" in user_input.lower():
                score += 0.1  # 包含错误信息
        
        # 映射到置信度等级
        if score >= 0.9:
            return ConfidenceLevel.HIGH
        elif score >= 0.6:
            return ConfidenceLevel.MEDIUM
        else:
            return ConfidenceLevel.LOW
    
    def select_model(self, analysis: ProblemAnalysis) -> ThinkingModel:
        """
        根据问题分析选择合适的思维模型
        
        Args:
            analysis: 问题分析结果
            
        Returns:
            ThinkingModel 对象
        """
        problem_type = analysis.problem_type
        
        # 根据问题类型选择模型
        model_mapping = {
            ProblemType.SKILL_CREATION: "research_mode",
            ProblemType.SYSTEM_REPAIR: "diagnostic_mode",
            ProblemType.RESEARCH: "research_mode",
            ProblemType.COMPLEX_ANALYSIS: "generic_pipeline",
            ProblemType.GENERAL_DECISION: "generic_pipeline",
        }
        
        model_id = model_mapping.get(problem_type, "generic_pipeline")
        selected_model = self.builtin_models.get(model_id, self.builtin_models["generic_pipeline"])
        
        # 更新使用计数
        selected_model.use_count += 1
        selected_model.last_used = datetime.now().isoformat()
        
        return selected_model
    
    def execute_thinking(self, 
                         user_input: str, 
                         model: Optional[ThinkingModel] = None) -> ThinkingResult:
        """
        执行思维模型处理
        
        Args:
            user_input: 用户输入
            model: 可选的思维模型，如果不提供则自动选择
            
        Returns:
            ThinkingResult 对象
        """
        import time
        start_time = time.time()
        
        # 1. 问题分析
        analysis = self.analyze_problem(user_input)
        
        # 2. 模型选择
        if model is None:
            model = self.select_model(analysis)
        
        stages_completed = []
        findings = []
        recommendations = []
        
        # 3. 执行各阶段处理
        for stage in model.stages:
            stage_name = stage.split(" - ")[0]
            stages_completed.append(stage_name)
            
            # 根据阶段执行相应处理
            stage_result = self._process_stage(stage_name, analysis, model)
            if stage_result:
                findings.extend(stage_result.get("findings", []))
                recommendations.extend(stage_result.get("recommendations", []))
        
        # 4. 计算处理时间
        processing_time = (time.time() - start_time) * 1000
        
        # 5. 生成结果
        result = ThinkingResult(
            problem=user_input,
            selected_model=model.name,
            stages_completed=stages_completed,
            findings=findings,
            recommendations=recommendations,
            confidence=analysis.confidence,
            processing_time_ms=processing_time,
            timestamp=datetime.now().isoformat()
        )
        
        return result
    
    def _process_stage(self, 
                       stage_name: str, 
                       analysis: ProblemAnalysis,
                       model: ThinkingModel) -> Optional[Dict]:
        """
        处理单个阶段
        
        Args:
            stage_name: 阶段名称
            analysis: 问题分析
            model: 思维模型
            
        Returns:
            阶段处理结果
        """
        stage_handlers = {
            "Memory Query": self._handle_memory_query,
            "Documentation Access": self._handle_documentation_access,
            "Public Research": self._handle_public_research,
            "Best Practices": self._handle_best_practices,
            "Solution Fusion": self._handle_solution_fusion,
            "Memory Pattern Match": self._handle_memory_pattern_match,
            "Problem Understanding": self._handle_problem_understanding,
            "Official Solution Search": self._handle_official_solution_search,
            "Tool/Skill Match": self._handle_tool_skill_match,
            "Community Solutions": self._handle_community_solutions,
            "Last Resort Fix": self._handle_last_resort_fix,
            "Problem Analysis": self._handle_problem_analysis,
            "Model Selection": self._handle_model_selection,
            "Information Collection": self._handle_information_collection,
            "Analysis": self._handle_analysis,
            "Evaluation": self._handle_evaluation,
            "Synthesis": self._handle_synthesis,
            "Decision Formulation": self._handle_decision_formulation,
            "Memory Integration": self._handle_memory_integration,
        }
        
        handler = stage_handlers.get(stage_name)
        if handler:
            return handler(analysis, model)
        
        return None
    
    # 阶段处理方法
    def _handle_memory_query(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """记忆查询阶段"""
        return {
            "findings": [f"已查询记忆系统关于「{analysis.problem_type.value}」的历史案例"],
            "recommendations": ["使用记忆中的成功模式作为参考"]
        }
    
    def _handle_documentation_access(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """文档访问阶段"""
        return {
            "findings": [f"问题类型: {analysis.problem_type.value}", f"复杂度: {analysis.complexity}/10"],
            "recommendations": ["参考相关官方文档和最佳实践指南"]
        }
    
    def _handle_public_research(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """公开研究阶段"""
        return {
            "findings": [f"识别关键词: {', '.join(analysis.keywords[:5])}"],
            "recommendations": ["搜索社区解决方案和类似案例"]
        }
    
    def _handle_best_practices(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """最佳实践阶段"""
        return {
            "findings": [f"选择思维模型: {model.name}"],
            "recommendations": ["遵循该问题领域的最佳实践流程"]
        }
    
    def _handle_solution_fusion(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """方案融合阶段"""
        return {
            "findings": ["整合多源信息生成综合方案"],
            "recommendations": ["提供结构化的解决方案"]
        }
    
    def _handle_memory_pattern_match(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """记忆模式匹配阶段"""
        return {
            "findings": [f"检测到{analysis.problem_type.value}类型问题"],
            "recommendations": ["匹配历史故障模式进行快速定位"]
        }
    
    def _handle_problem_understanding(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """问题理解阶段"""
        return {
            "findings": [f"问题复杂度: {analysis.complexity}/10", f"约束条件: {len(analysis.constraints)}个"],
            "recommendations": ["全面理解问题范围和上下文"]
        }
    
    def _handle_official_solution_search(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """官方方案搜索阶段"""
        return {
            "findings": ["搜索官方文档和发布说明"],
            "recommendations": ["优先尝试官方推荐的解决方案"]
        }
    
    def _handle_tool_skill_match(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """工具/技能匹配阶段"""
        return {
            "findings": ["评估可用的修复工具和技能"],
            "recommendations": ["选择最适合的自动化修复工具"]
        }
    
    def _handle_community_solutions(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """社区方案阶段"""
        return {
            "findings": ["搜索社区工作区和修复方案"],
            "recommendations": ["参考社区验证过的解决方案"]
        }
    
    def _handle_last_resort_fix(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        """最后方案修复阶段"""
        return {
            "findings": ["所有标准方案都已尝试"],
            "recommendations": ["创建临时修复脚本作为最后手段"]
        }
    
    # 通用阶段处理方法
    def _handle_problem_analysis(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["完成问题分解"], "recommendations": ["识别关键变量和约束"]}
    
    def _handle_model_selection(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": [f"选择模型: {model.name}"], "recommendations": ["使用适合的思维模型"]}
    
    def _handle_information_collection(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["收集相关信息和数据"], "recommendations": ["获取上下文和历史数据"]}
    
    def _handle_analysis(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["分析各个选项"], "recommendations": ["评估每个选项的优缺点"]}
    
    def _handle_evaluation(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["多角度评估"], "recommendations": ["证据权重和验证"]}
    
    def _handle_synthesis(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["综合所有分析结果"], "recommendations": ["形成连贯的理解"]}
    
    def _handle_decision_formulation(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["生成决策建议"], "recommendations": ["提供具体行动建议"]}
    
    def _handle_memory_integration(self, analysis: ProblemAnalysis, model: ThinkingModel) -> Dict:
        return {"findings": ["准备存储到记忆系统"], "recommendations": ["记录本次决策过程"]}
    
    def format_result(self, result: ThinkingResult) -> str:
        """
        格式化思维结果为可读字符串
        
        Args:
            result: ThinkingResult 对象
            
        Returns:
            格式化的字符串
        """
        lines = [
            f"🧠 思维模型处理结果",
            "=" * 50,
            f"",
            f"📝 问题: {result.problem[:100]}{'...' if len(result.problem) > 100 else ''}",
            f"🎯 使用模型: {result.selected_model}",
            f"📊 置信度: {result.confidence.value}",
            f"⏱️ 处理时间: {result.processing_time_ms:.1f}ms",
            f"",
            f"✅ 已完成阶段: {len(result.stages_completed)}",
        ]
        
        for i, stage in enumerate(result.stages_completed, 1):
            lines.append(f"   {i}. {stage}")
        
        if result.findings:
            lines.extend([
                f"",
                f"📋 发现:",
            ])
            for finding in result.findings[:5]:
                lines.append(f"   • {finding}")
        
        if result.recommendations:
            lines.extend([
                f"",
                f"💡 建议:",
            ])
            for rec in result.recommendations[:5]:
                lines.append(f"   • {rec}")
        
        lines.extend([
            f"",
            f"⏰ 时间戳: {result.timestamp}",
        ])
        
        return "\n".join(lines)


def get_thinking_core() -> ThinkingModelCore:
    """获取思维模型核心实例"""
    return ThinkingModelCore()
