---
name: thingsboard
description: Manage ThingsBoard devices, dashboards, telemetry, and users via the ThingsBoard REST API.
homepage: https://thingsboard.io
metadata: {"clawdbot":{"emoji":"📊","requires":{"bins":["jq","curl"],"env":["TB_URL","TB_USERNAME","TB_PASSWORD"]}}}
---

# ThingsBoard Skill

Manage ThingsBoard IoT platform resources including devices, dashboards, telemetry data, and users.

## Setup

1. Configure your ThingsBoard server in `credentials.json`:
   ```json
   [
     {
       "name": "Server Thingsboard",
       "url": "http://localhost:8080",
       "account": [
         {
           "sysadmin": {
             "email": "sysadmin@thingsboard.org",
             "password": "sysadmin"
           }
         },
         {
           "tenant": {
             "email": "tenant@thingsboard.org",
             "password": "tenant"
           }
         }
       ]
     }
   ]
   ```

2. Set environment variables:
   ```bash
   export TB_URL="http://localhost:8080"
   export TB_USERNAME="tenant@thingsboard.org"
   export TB_PASSWORD="tenant"
   ```

3. Get authentication token:
   ```bash
   export TB_TOKEN=$(curl -s -X POST "$TB_URL/api/auth/login" \
     -H "Content-Type: application/json" \
     -d "{\"username\":\"$TB_USERNAME\",\"password\":\"$TB_PASSWORD\"}" | jq -r '.token')
   ```

## Usage

All commands use curl to interact with the ThingsBoard REST API.

### Authentication

**Login and get token:**
```bash
curl -s -X POST "$TB_URL/api/auth/login" \
  -H "Content-Type: application/json" \
  -d "{\"username\":\"$TB_USERNAME\",\"password\":\"$TB_PASSWORD\"}" | jq -r '.token'
```

**Refresh token (when expired):**
```bash
curl -s -X POST "$TB_URL/api/auth/token" \
  -H "Content-Type: application/json" \
  -d "{\"refreshToken\":\"$TB_REFRESH_TOKEN\"}" | jq -r '.token'
```

**Get current user info:**
```bash
curl -s "$TB_URL/api/auth/user" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

### Device Management

**List all tenant devices:**
```bash
curl -s "$TB_URL/api/tenant/devices?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {name, id: .id.id, type}'
```

**Get device by ID:**
```bash
curl -s "$TB_URL/api/device/{deviceId}" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get device credentials:**
```bash
curl -s "$TB_URL/api/device/{deviceId}/credentials" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

### Telemetry & Attributes

**Get telemetry keys:**
```bash
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/keys/timeseries" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get latest telemetry:**
```bash
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/values/timeseries?keys=temperature,humidity" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get timeseries data with time range:**
```bash
START_TS=$(($(date +%s)*1000 - 3600000))  # 1 hour ago
END_TS=$(($(date +%s)*1000))              # now
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/values/timeseries?keys=temperature&startTs=$START_TS&endTs=$END_TS&limit=100" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get attribute keys:**
```bash
# Client scope
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/keys/attributes/CLIENT_SCOPE" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq

# Shared scope
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/keys/attributes/SHARED_SCOPE" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq

# Server scope
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/keys/attributes/SERVER_SCOPE" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get attributes by scope:**
```bash
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/values/attributes/CLIENT_SCOPE?keys=attribute1,attribute2" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Save device attributes:**
```bash
curl -s -X POST "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/attributes/SERVER_SCOPE" \
  -H "X-Authorization: Bearer $TB_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"attribute1":"value1","attribute2":"value2"}' | jq
```

**Delete timeseries keys:**
```bash
curl -s -X DELETE "$TB_URL/api/plugins/telemetry/DEVICE/{deviceId}/timeseries/delete?keys=oldKey1,oldKey2&deleteAllDataForKeys=true" \
  -H "X-Authorization: Bearer $TB_TOKEN"
```

### Dashboard Management

**List all dashboards:**
```bash
curl -s "$TB_URL/api/tenant/dashboards?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {name, id: .id.id}'
```

**Get dashboard info:**
```bash
curl -s "$TB_URL/api/dashboard/{dashboardId}" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Make dashboard public:**
```bash
curl -s -X POST "$TB_URL/api/customer/public/dashboard/{dashboardId}" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

**Get public dashboard info (no auth required):**
```bash
curl -s "$TB_URL/api/dashboard/info/{publicDashboardId}" | jq
```

**Remove public access:**
```bash
curl -s -X DELETE "$TB_URL/api/customer/public/dashboard/{dashboardId}" \
  -H "X-Authorization: Bearer $TB_TOKEN"
```

### User Management

**List tenant users:**
```bash
curl -s "$TB_URL/api/tenant/users?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {email, firstName, lastName, id: .id.id}'
```

**List customers:**
```bash
curl -s "$TB_URL/api/customers?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {title, id: .id.id}'
```

**Get customer users:**
```bash
curl -s "$TB_URL/api/customer/{customerId}/users?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[]'
```

### Assets

**List all assets:**
```bash
curl -s "$TB_URL/api/tenant/assets?pageSize=100&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {name, type, id: .id.id}'
```

**Get asset by ID:**
```bash
curl -s "$TB_URL/api/asset/{assetId}" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```

## Notes

- **Authentication**: JWT tokens expire after a configured period (default: 2 hours). Re-authenticate when you receive 401 errors.
- **Device/Dashboard IDs**: Entity IDs are in the format `{entityType: "DEVICE", id: "uuid"}`. Use the `id` field for API calls.
- **Pagination**: Most list endpoints support `pageSize` and `page` parameters (default: 100 items per page, max: 1000).
- **Attribute Scopes**:
  - `CLIENT_SCOPE`: Client-side attributes (set by devices)
  - `SHARED_SCOPE`: Shared between server and devices
  - `SERVER_SCOPE`: Server-side only (not visible to devices)
- **Timestamps**: Use milliseconds since epoch for `startTs` and `endTs` parameters.
- **Rate Limits**: Check your ThingsBoard server configuration for API rate limits.
- **HTTPS**: For production, use HTTPS URLs (e.g., `https://demo.thingsboard.io`).

## Examples

```bash
# Complete workflow: Login, list devices, get telemetry
export TB_URL="http://localhost:8080"
export TB_USERNAME="tenant@thingsboard.org"
export TB_PASSWORD="tenant"

# Get token
export TB_TOKEN=$(curl -s -X POST "$TB_URL/api/auth/login" \
  -H "Content-Type: application/json" \
  -d "{\"username\":\"$TB_USERNAME\",\"password\":\"$TB_PASSWORD\"}" | jq -r '.token')

# List all devices
curl -s "$TB_URL/api/tenant/devices?pageSize=10&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq '.data[] | {name, type, id: .id.id}'

# Get first device ID
DEVICE_ID=$(curl -s "$TB_URL/api/tenant/devices?pageSize=1&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq -r '.data[0].id.id')

# Get telemetry keys for device
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/$DEVICE_ID/keys/timeseries" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq

# Get latest telemetry values
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/$DEVICE_ID/values/timeseries?keys=temperature,humidity" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq

# Get historical data (last hour)
START_TS=$(($(date +%s)*1000 - 3600000))
END_TS=$(($(date +%s)*1000))
curl -s "$TB_URL/api/plugins/telemetry/DEVICE/$DEVICE_ID/values/timeseries?keys=temperature&startTs=$START_TS&endTs=$END_TS&limit=100" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq

# List dashboards and make first one public
DASHBOARD_ID=$(curl -s "$TB_URL/api/tenant/dashboards?pageSize=1&page=0" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq -r '.data[0].id.id')

curl -s -X POST "$TB_URL/api/customer/public/dashboard/$DASHBOARD_ID" \
  -H "X-Authorization: Bearer $TB_TOKEN" | jq
```
