use anchor_lang::prelude::*;
use anchor_spl::token::{self, Mint, Token, TokenAccount, Transfer};

declare_id!("7rSMKhD3ve2NcR4qdYK5xcbMHfGtEjTgoKCS5Mgx9ECX");

// Game constants
const ENTRY_FEE: u64 = 1_000_000; // 1 USDC (6 decimals)
const TOTAL_FLIPS: u32 = 14;
const OPERATOR_FEE_BPS: u64 = 100; // 1% to operator
const BPS_BASE: u64 = 10000;
const BUFFER_SIZE: usize = 256;
// Jackpot gets the rest: 99%

#[program]
pub mod the_flip {
    use super::*;

    /// Initialize a new game. Creates the game state PDA and the USDC vault.
    pub fn initialize_game(ctx: Context<InitializeGame>) -> Result<()> {
        let game = &mut ctx.accounts.game;
        game.authority = ctx.accounts.authority.key();
        game.usdc_mint = ctx.accounts.usdc_mint.key();
        game.vault = ctx.accounts.vault.key();
        game.bump = ctx.bumps.game;
        game.vault_bump = ctx.bumps.vault;
        game.global_flip = 0;
        game.flip_results = [0u8; BUFFER_SIZE];
        game.jackpot_pool = 0;
        game.operator_pool = 0;
        game.total_entries = 0;
        game.total_wins = 0;

        msg!("THE FLIP initialized. Vault: {}", game.vault);
        Ok(())
    }

    /// Player enters the game. Transfers 1 USDC to the vault and creates a ticket PDA.
    /// Always open — no rounds, no gates.
    pub fn enter(ctx: Context<Enter>, predictions: [u8; 14]) -> Result<()> {
        let game = &mut ctx.accounts.game;

        // Validate predictions: each byte must be 1 (H) or 2 (T)
        for &p in predictions.iter() {
            require!(p == 1 || p == 2, FlipError::InvalidPrediction);
        }

        // Transfer 1 USDC from player to vault
        let cpi_ctx = CpiContext::new(
            ctx.accounts.token_program.to_account_info(),
            Transfer {
                from: ctx.accounts.player_token_account.to_account_info(),
                to: ctx.accounts.vault.to_account_info(),
                authority: ctx.accounts.player.to_account_info(),
            },
        );
        token::transfer(cpi_ctx, ENTRY_FEE)?;

        // Split into pools: 1% operator, 99% jackpot
        let operator_amount = ENTRY_FEE * OPERATOR_FEE_BPS / BPS_BASE;
        let jackpot_amount = ENTRY_FEE - operator_amount;
        game.operator_pool += operator_amount;
        game.jackpot_pool += jackpot_amount;
        game.total_entries += 1;

        // Initialize ticket
        let ticket = &mut ctx.accounts.ticket;
        ticket.game = game.key();
        ticket.player = ctx.accounts.player.key();
        ticket.start_flip = game.global_flip;
        ticket.predictions = predictions;
        ticket.winner = false;
        ticket.collected = false;
        ticket.bump = ctx.bumps.ticket;

        msg!(
            "Player {} entered at flip #{}. Total entries: {}",
            ctx.accounts.player.key(),
            game.global_flip,
            game.total_entries
        );
        Ok(())
    }

    /// Execute the next global flip. Permissionless — anyone can call.
    pub fn flip(ctx: Context<FlipCtx>) -> Result<()> {
        let game = &mut ctx.accounts.game;

        let idx = game.global_flip as usize % BUFFER_SIZE;

        // Randomness from slot + timestamp + game key + global flip number
        let clock = Clock::get()?;
        let mut seed: u8 = (game.global_flip & 0xFF) as u8;
        for b in clock.slot.to_le_bytes() { seed ^= b; }
        for b in clock.unix_timestamp.to_le_bytes() { seed ^= b; }
        for b in game.key().to_bytes() { seed ^= b; }

        // Even = H (1), Odd = T (2)
        let result: u8 = if seed % 2 == 0 { 1 } else { 2 };

        game.flip_results[idx] = result;
        game.global_flip += 1;

        let result_str = if result == 1 { "HEADS" } else { "TAILS" };
        msg!("Flip #{}: {}", game.global_flip, result_str);

        Ok(())
    }

    /// Verify 14/14 predictions match and pay the entire jackpot. Permissionless.
    pub fn claim(ctx: Context<Claim>) -> Result<()> {
        let game = &mut ctx.accounts.game;
        let ticket = &mut ctx.accounts.ticket;

        // All 14 flips must be revealed
        require!(
            game.global_flip >= ticket.start_flip + TOTAL_FLIPS,
            FlipError::FlipsNotRevealed
        );

        // Must be within circular buffer window (256 flips)
        require!(
            game.global_flip - ticket.start_flip <= BUFFER_SIZE as u32,
            FlipError::BufferExpired
        );

        require!(!ticket.collected, FlipError::AlreadyCollected);

        // Verify all 14 predictions match flip results
        for i in 0..TOTAL_FLIPS {
            let idx = (ticket.start_flip + i) as usize % BUFFER_SIZE;
            require!(
                ticket.predictions[i as usize] == game.flip_results[idx],
                FlipError::NotAWinner
            );
        }

        ticket.winner = true;
        ticket.collected = true;
        game.total_wins += 1;

        // Winner takes the entire jackpot
        let payout = game.jackpot_pool;

        // PDA signer seeds for vault
        let authority_key = game.authority;
        let seeds = &[
            b"vault" as &[u8],
            authority_key.as_ref(),
            &[game.vault_bump],
        ];
        let signer_seeds = &[&seeds[..]];

        let cpi_ctx = CpiContext::new_with_signer(
            ctx.accounts.token_program.to_account_info(),
            Transfer {
                from: ctx.accounts.vault.to_account_info(),
                to: ctx.accounts.player_token_account.to_account_info(),
                authority: ctx.accounts.vault.to_account_info(),
            },
            signer_seeds,
        );
        token::transfer(cpi_ctx, payout)?;

        game.jackpot_pool = 0;

        msg!(
            "WINNER! {} claimed {} USDC (win #{})",
            ticket.player,
            payout,
            game.total_wins
        );
        Ok(())
    }

    /// Authority withdraws accumulated operator fees from the vault.
    pub fn withdraw_fees(ctx: Context<WithdrawFees>, amount: u64) -> Result<()> {
        let game = &mut ctx.accounts.game;

        require!(amount <= game.operator_pool, FlipError::InsufficientFees);

        let authority_key = game.authority;
        let seeds = &[
            b"vault" as &[u8],
            authority_key.as_ref(),
            &[game.vault_bump],
        ];
        let signer_seeds = &[&seeds[..]];

        let cpi_ctx = CpiContext::new_with_signer(
            ctx.accounts.token_program.to_account_info(),
            Transfer {
                from: ctx.accounts.vault.to_account_info(),
                to: ctx.accounts.authority_token_account.to_account_info(),
                authority: ctx.accounts.vault.to_account_info(),
            },
            signer_seeds,
        );
        token::transfer(cpi_ctx, amount)?;

        game.operator_pool -= amount;
        msg!("Withdrew {} USDC units in operator fees", amount);
        Ok(())
    }

    /// Close old game PDA for migration. Authority only.
    pub fn close_game_v1(ctx: Context<CloseGameV1>) -> Result<()> {
        let game_info = ctx.accounts.game.to_account_info();
        let authority_info = ctx.accounts.authority.to_account_info();

        // Return lamports to authority
        let lamports = game_info.lamports();
        **game_info.try_borrow_mut_lamports()? = 0;
        **authority_info.try_borrow_mut_lamports()? += lamports;

        // Zero out data
        let mut data = game_info.try_borrow_mut_data()?;
        for byte in data.iter_mut() {
            *byte = 0;
        }

        msg!("Old game PDA closed.");
        Ok(())
    }
}

// ─── Account Contexts ───────────────────────────────────────────────────────

#[derive(Accounts)]
pub struct InitializeGame<'info> {
    #[account(mut)]
    pub authority: Signer<'info>,

    #[account(
        init,
        payer = authority,
        space = 8 + Game::INIT_SPACE,
        seeds = [b"game", authority.key().as_ref()],
        bump,
    )]
    pub game: Account<'info, Game>,

    pub usdc_mint: Account<'info, Mint>,

    #[account(
        init_if_needed,
        payer = authority,
        token::mint = usdc_mint,
        token::authority = vault,
        seeds = [b"vault", authority.key().as_ref()],
        bump,
    )]
    pub vault: Account<'info, TokenAccount>,

    pub system_program: Program<'info, System>,
    pub token_program: Program<'info, Token>,
    pub rent: Sysvar<'info, Rent>,
}

#[derive(Accounts)]
pub struct Enter<'info> {
    #[account(mut)]
    pub player: Signer<'info>,

    #[account(
        mut,
        seeds = [b"game", game.authority.as_ref()],
        bump = game.bump,
    )]
    pub game: Account<'info, Game>,

    #[account(
        init,
        payer = player,
        space = 8 + Ticket::INIT_SPACE,
        seeds = [b"ticket", game.key().as_ref(), player.key().as_ref(), &game.global_flip.to_le_bytes()],
        bump,
    )]
    pub ticket: Account<'info, Ticket>,

    #[account(
        mut,
        constraint = player_token_account.owner == player.key(),
        constraint = player_token_account.mint == game.usdc_mint,
    )]
    pub player_token_account: Account<'info, TokenAccount>,

    #[account(
        mut,
        address = game.vault,
    )]
    pub vault: Account<'info, TokenAccount>,

    pub system_program: Program<'info, System>,
    pub token_program: Program<'info, Token>,
}

/// Permissionless — anyone can execute the next flip.
#[derive(Accounts)]
pub struct FlipCtx<'info> {
    pub caller: Signer<'info>,

    #[account(
        mut,
        seeds = [b"game", game.authority.as_ref()],
        bump = game.bump,
    )]
    pub game: Account<'info, Game>,
}

#[derive(Accounts)]
pub struct Claim<'info> {
    pub claimer: Signer<'info>,

    #[account(
        mut,
        seeds = [b"game", game.authority.as_ref()],
        bump = game.bump,
    )]
    pub game: Account<'info, Game>,

    #[account(
        mut,
        constraint = ticket.game == game.key() @ FlipError::TicketGameMismatch,
        constraint = ticket.player == player.key() @ FlipError::PlayerMismatch,
    )]
    pub ticket: Account<'info, Ticket>,

    /// CHECK: Verified via ticket.player constraint
    pub player: UncheckedAccount<'info>,

    #[account(
        mut,
        constraint = player_token_account.owner == player.key(),
        constraint = player_token_account.mint == game.usdc_mint,
    )]
    pub player_token_account: Account<'info, TokenAccount>,

    #[account(
        mut,
        address = game.vault,
    )]
    pub vault: Account<'info, TokenAccount>,

    pub token_program: Program<'info, Token>,
}

#[derive(Accounts)]
pub struct WithdrawFees<'info> {
    #[account(
        constraint = authority.key() == game.authority @ FlipError::Unauthorized,
    )]
    pub authority: Signer<'info>,

    #[account(
        mut,
        seeds = [b"game", game.authority.as_ref()],
        bump = game.bump,
    )]
    pub game: Account<'info, Game>,

    #[account(
        mut,
        constraint = authority_token_account.owner == authority.key(),
        constraint = authority_token_account.mint == game.usdc_mint,
    )]
    pub authority_token_account: Account<'info, TokenAccount>,

    #[account(
        mut,
        address = game.vault,
    )]
    pub vault: Account<'info, TokenAccount>,

    pub token_program: Program<'info, Token>,
}

#[derive(Accounts)]
pub struct CloseGameV1<'info> {
    #[account(mut)]
    pub authority: Signer<'info>,

    /// CHECK: Old game PDA being closed. Verified by seeds.
    #[account(
        mut,
        seeds = [b"game", authority.key().as_ref()],
        bump,
    )]
    pub game: UncheckedAccount<'info>,
}

// ─── State ──────────────────────────────────────────────────────────────────

#[account]
#[derive(InitSpace)]
pub struct Game {
    pub authority: Pubkey,           // 32
    pub usdc_mint: Pubkey,           // 32
    pub vault: Pubkey,               // 32
    pub bump: u8,                    // 1
    pub vault_bump: u8,              // 1
    pub global_flip: u32,            // 4  — total flips ever executed
    pub flip_results: [u8; 256],     // 256 — circular buffer (index = global_flip % 256)
    pub jackpot_pool: u64,           // 8
    pub operator_pool: u64,          // 8
    pub total_entries: u32,          // 4
    pub total_wins: u32,             // 4  — lifetime winners
}

#[account]
#[derive(InitSpace)]
pub struct Ticket {
    pub game: Pubkey,                // 32
    pub player: Pubkey,              // 32
    pub start_flip: u32,             // 4  — which global flip this ticket starts at
    pub predictions: [u8; 14],       // 14
    pub winner: bool,                // 1
    pub collected: bool,             // 1
    pub bump: u8,                    // 1
}

// ─── Errors ─────────────────────────────────────────────────────────────────

#[error_code]
pub enum FlipError {
    #[msg("Invalid prediction: must be 1 (H) or 2 (T)")]
    InvalidPrediction,
    #[msg("Not a 14/14 winner")]
    NotAWinner,
    #[msg("Already collected payout")]
    AlreadyCollected,
    #[msg("Unauthorized")]
    Unauthorized,
    #[msg("Ticket/game mismatch")]
    TicketGameMismatch,
    #[msg("Player mismatch")]
    PlayerMismatch,
    #[msg("Insufficient operator fees")]
    InsufficientFees,
    #[msg("Not all 14 flips have been revealed yet")]
    FlipsNotRevealed,
    #[msg("Ticket expired — circular buffer overwritten (claim within 256 flips)")]
    BufferExpired,
}
