/**
 * THE FLIP — Continuous Game Demo & Operations Script
 *
 * No rounds. No entry windows. The game never stops.
 * Flips happen continuously on a global counter.
 * Enter anytime, pick 14 H/T predictions, ride the next 14 global flips.
 * Get all 14 right → take the entire jackpot.
 *
 * Works with deployed program: 7rSMKhD3ve2NcR4qdYK5xcbMHfGtEjTgoKCS5Mgx9ECX
 *
 * Usage:
 *   node app/demo.mjs init                    Initialize game + vault
 *   node app/demo.mjs enter <HHTHTT...>       Enter with 14 H/T predictions
 *   node app/demo.mjs flip                    Execute one coin flip (permissionless)
 *   node app/demo.mjs claim <player> <start>  Claim jackpot (verify 14/14 + pay)
 *   node app/demo.mjs status                  Show game state
 *   node app/demo.mjs ticket <player> <start> Show a player's ticket
 *   node app/demo.mjs withdraw-fees [amount]  Withdraw operator fees
 *   node app/demo.mjs close-game-v1           Close old game PDA for migration
 */

import { Connection, Keypair, PublicKey, SystemProgram } from '@solana/web3.js';
import {
  getAssociatedTokenAddress,
  TOKEN_PROGRAM_ID
} from '@solana/spl-token';
import * as anchor from '@coral-xyz/anchor';
import fs from 'fs';
import path from 'path';

// --- Config ---
const DEVNET_URL = 'https://api.devnet.solana.com';
const USDC_MINT = new PublicKey('4zMMC9srt5Ri5X14GAgXhaHii3GnPAEERYPJgZJDncDU');
const PROGRAM_ID = new PublicKey('7rSMKhD3ve2NcR4qdYK5xcbMHfGtEjTgoKCS5Mgx9ECX');
const TOTAL_FLIPS = 14;
const BUFFER_SIZE = 256;

// Load IDL
const IDL_PATH = path.join(import.meta.dirname, '..', 'target', 'idl', 'the_flip.json');

// Load wallet
function loadWallet(keyPath) {
  const raw = JSON.parse(fs.readFileSync(keyPath || process.env.ANCHOR_WALLET ||
    path.join(process.env.HOME, '.config', 'solana', 'id.json'), 'utf8'));
  return Keypair.fromSecretKey(Uint8Array.from(raw));
}

// PDA derivation
function getGamePDA(authority) {
  return PublicKey.findProgramAddressSync(
    [Buffer.from('game'), authority.toBuffer()], PROGRAM_ID
  );
}

function getVaultPDA(authority) {
  return PublicKey.findProgramAddressSync(
    [Buffer.from('vault'), authority.toBuffer()], PROGRAM_ID
  );
}

function getTicketPDA(game, player, startFlip) {
  const buf = Buffer.alloc(4);
  buf.writeUInt32LE(startFlip);
  return PublicKey.findProgramAddressSync(
    [Buffer.from('ticket'), game.toBuffer(), player.toBuffer(), buf],
    PROGRAM_ID
  );
}

// Parse predictions string (HHTHTT...) to array of u8 (1=H, 2=T)
function parsePredictions(str) {
  if (str.length !== 14) throw new Error('Must be exactly 14 predictions (H or T)');
  const result = [];
  for (let i = 0; i < 14; i++) {
    const c = str[i].toUpperCase();
    if (c === 'H') result.push(1);
    else if (c === 'T') result.push(2);
    else throw new Error('Invalid char: ' + c + ' (must be H or T)');
  }
  return result;
}

// Format flip result (1=H, 2=T)
function flipToStr(r) { return r === 1 ? 'H' : r === 2 ? 'T' : '?'; }

// Format USDC amount (6 decimals)
function fmtUsdc(raw) {
  const n = typeof raw === 'number' ? raw : Number(raw.toString());
  return (n / 1_000_000).toFixed(6);
}

async function main() {
  const cmd = process.argv[2];
  if (!cmd) {
    console.log('Usage: node app/demo.mjs <command> [args]');
    console.log('Commands: init, enter, flip, claim, status, ticket, withdraw-fees, close-game-v1');
    process.exit(1);
  }

  const connection = new Connection(DEVNET_URL, 'confirmed');
  const wallet = loadWallet();
  const provider = new anchor.AnchorProvider(
    connection,
    new anchor.Wallet(wallet),
    { commitment: 'confirmed' }
  );
  anchor.setProvider(provider);

  if (!fs.existsSync(IDL_PATH)) {
    console.error('IDL not found at', IDL_PATH);
    console.error('Run anchor build first or copy the IDL.');
    process.exit(1);
  }
  const idl = JSON.parse(fs.readFileSync(IDL_PATH, 'utf8'));
  const program = new anchor.Program(idl, provider);

  const [gamePDA] = getGamePDA(wallet.publicKey);
  const [vaultPDA] = getVaultPDA(wallet.publicKey);

  switch (cmd) {
    case 'init': {
      console.log('Initializing THE FLIP (continuous game)...');
      console.log('  Authority:', wallet.publicKey.toBase58());
      console.log('  Game PDA: ', gamePDA.toBase58());
      console.log('  Vault PDA:', vaultPDA.toBase58());
      console.log('  Program:  ', PROGRAM_ID.toBase58());

      try {
        const tx = await program.methods.initializeGame().accounts({
          authority: wallet.publicKey,
          game: gamePDA,
          usdcMint: USDC_MINT,
          vault: vaultPDA,
          systemProgram: SystemProgram.programId,
          tokenProgram: TOKEN_PROGRAM_ID,
          rent: anchor.web3.SYSVAR_RENT_PUBKEY,
        }).rpc();
        console.log('Game initialized! TX:', tx);
      } catch (e) {
        if (e.message?.includes('already in use')) {
          console.log('Game already initialized.');
        } else {
          throw e;
        }
      }
      break;
    }

    case 'enter': {
      const preds = process.argv[3];
      const playerKeyPath = process.argv[4];
      if (!preds) { console.error('Usage: enter <HHTHTT...> [player_keypair_path]'); process.exit(1); }

      const parsed = parsePredictions(preds);
      const player = playerKeyPath ? loadWallet(playerKeyPath) : wallet;

      const playerProvider = new anchor.AnchorProvider(
        connection,
        new anchor.Wallet(player),
        { commitment: 'confirmed' }
      );
      const playerProgram = new anchor.Program(idl, playerProvider);

      const game = await program.account.game.fetch(gamePDA);
      const startFlip = game.globalFlip;
      const [ticketPDA] = getTicketPDA(gamePDA, player.publicKey, startFlip);
      const playerATA = await getAssociatedTokenAddress(USDC_MINT, player.publicKey);

      console.log('Entering at flip #' + startFlip + ' with player ' + player.publicKey.toBase58());
      console.log('Predictions: ' + preds.toUpperCase());
      console.log('Your flips:  #' + startFlip + ' – #' + (startFlip + 13));

      const tx = await playerProgram.methods.enter(parsed).accounts({
        player: player.publicKey,
        game: gamePDA,
        ticket: ticketPDA,
        playerTokenAccount: playerATA,
        vault: vaultPDA,
        systemProgram: SystemProgram.programId,
        tokenProgram: TOKEN_PROGRAM_ID,
      }).rpc();

      console.log('Entry accepted! TX:', tx);
      console.log('Ticket PDA:', ticketPDA.toBase58());
      break;
    }

    case 'flip': {
      console.log('Executing next flip...');
      const tx = await program.methods.flip().accounts({
        caller: wallet.publicKey,
        game: gamePDA,
      }).rpc();

      const game = await program.account.game.fetch(gamePDA);
      const flipNum = game.globalFlip;
      const idx = (flipNum - 1) % BUFFER_SIZE;
      const result = flipToStr(game.flipResults[idx]);
      console.log('Flip #' + flipNum + ': ' + (result === 'H' ? 'HEADS' : 'TAILS') + '  TX: ' + tx);
      break;
    }

    case 'claim': {
      const playerPubkey = process.argv[3];
      const startFlipArg = process.argv[4];
      if (!playerPubkey || !startFlipArg) {
        console.error('Usage: claim <player_pubkey> <start_flip>');
        process.exit(1);
      }

      const player = new PublicKey(playerPubkey);
      const startFlip = parseInt(startFlipArg);
      const [ticketPDA] = getTicketPDA(gamePDA, player, startFlip);
      const playerATA = await getAssociatedTokenAddress(USDC_MINT, player);

      console.log('Claiming for player ' + playerPubkey + ' starting at flip #' + startFlip + '...');
      const tx = await program.methods.claim().accounts({
        claimer: wallet.publicKey,
        game: gamePDA,
        ticket: ticketPDA,
        player: player,
        playerTokenAccount: playerATA,
        vault: vaultPDA,
        tokenProgram: TOKEN_PROGRAM_ID,
      }).rpc();
      console.log('Claim TX:', tx);

      const game = await program.account.game.fetch(gamePDA);
      const ticket = await program.account.ticket.fetch(ticketPDA);
      if (ticket.winner) {
        console.log('WINNER! Jackpot collected. New jackpot:', fmtUsdc(game.jackpotPool), 'USDC');
      } else {
        console.log('Not a winner.');
      }
      break;
    }

    case 'status': {
      try {
        const game = await program.account.game.fetch(gamePDA);
        console.log('=== THE FLIP — Continuous Game Status ===');
        console.log('Program:       ' + PROGRAM_ID.toBase58());
        console.log('Authority:     ' + game.authority.toBase58());
        console.log('Vault:         ' + game.vault.toBase58());
        console.log('');
        console.log('Global flip:   ' + game.globalFlip);
        console.log('Total entries: ' + game.totalEntries);
        console.log('Total wins:    ' + game.totalWins);
        console.log('');
        console.log('Jackpot pool:  ' + fmtUsdc(game.jackpotPool) + ' USDC');
        console.log('Operator pool: ' + fmtUsdc(game.operatorPool) + ' USDC');
        console.log('');

        // Show last N flip results from circular buffer
        const show = Math.min(game.globalFlip, 20);
        if (show > 0) {
          const results = [];
          for (let i = game.globalFlip - show; i < game.globalFlip; i++) {
            const idx = i % BUFFER_SIZE;
            results.push('#' + (i + 1) + ':' + flipToStr(game.flipResults[idx]));
          }
          console.log('Recent flips:  ' + results.join('  '));
        }
        console.log('='.repeat(42));
      } catch (e) {
        console.log('Game not initialized. Run: node app/demo.mjs init');
      }
      break;
    }

    case 'ticket': {
      const playerPubkey = process.argv[3];
      const startFlipArg = process.argv[4];
      if (!playerPubkey) { console.error('Usage: ticket <player_pubkey> [start_flip]'); process.exit(1); }

      const player = new PublicKey(playerPubkey);
      const game = await program.account.game.fetch(gamePDA);

      // If start_flip not provided, try to find it
      let startFlip;
      if (startFlipArg) {
        startFlip = parseInt(startFlipArg);
      } else {
        // Try recent flip numbers (last 20)
        let found = false;
        for (let sf = game.globalFlip; sf >= Math.max(0, game.globalFlip - 20); sf--) {
          try {
            const [pda] = getTicketPDA(gamePDA, player, sf);
            await program.account.ticket.fetch(pda);
            startFlip = sf;
            found = true;
            break;
          } catch (e) { /* not found, try next */ }
        }
        if (!found) {
          console.log('No recent ticket found for this player. Try: ticket <player> <start_flip>');
          break;
        }
      }

      const [ticketPDA] = getTicketPDA(gamePDA, player, startFlip);

      try {
        const ticket = await program.account.ticket.fetch(ticketPDA);
        const endFlip = ticket.startFlip + TOTAL_FLIPS - 1;
        console.log('=== Ticket for ' + player.toBase58() + ' ===');
        console.log('Start flip:    #' + ticket.startFlip);
        console.log('End flip:      #' + endFlip);
        console.log('Predictions:   ' + ticket.predictions.map(p => p === 1 ? 'H' : 'T').join(''));

        // Compare predictions vs results
        const flipsRevealed = Math.min(TOTAL_FLIPS, Math.max(0, game.globalFlip - ticket.startFlip));
        let alive = true;
        let score = 0;
        let diedAt = null;
        const comparison = [];

        for (let i = 0; i < TOTAL_FLIPS; i++) {
          const flipNum = ticket.startFlip + i;
          const predicted = flipToStr(ticket.predictions[i]);
          if (flipNum < game.globalFlip) {
            const idx = flipNum % BUFFER_SIZE;
            const actual = flipToStr(game.flipResults[idx]);
            const match = predicted === actual;
            if (alive && match) {
              score = i + 1;
            } else if (alive && !match) {
              alive = false;
              diedAt = i + 1;
            }
            comparison.push('#' + (flipNum + 1) + ':' + predicted + (match ? '=' : '≠') + actual);
          } else {
            comparison.push('#' + (flipNum + 1) + ':' + predicted + '/?');
          }
        }

        console.log('Flips:         ' + flipsRevealed + '/' + TOTAL_FLIPS + ' revealed');

        if (ticket.winner && ticket.collected) {
          console.log('Status:        WINNER (collected)');
        } else if (ticket.winner) {
          console.log('Status:        WINNER');
        } else if (!alive) {
          console.log('Status:        ELIMINATED at flip ' + diedAt);
        } else if (flipsRevealed === TOTAL_FLIPS) {
          console.log('Status:        ALL CORRECT — claim available!');
        } else {
          console.log('Status:        ALIVE (' + score + '/' + flipsRevealed + ' correct)');
        }

        console.log('');
        console.log('Comparison:');
        // Print 7 per line
        for (let i = 0; i < comparison.length; i += 7) {
          console.log('  ' + comparison.slice(i, i + 7).join('  '));
        }
      } catch (e) {
        console.log('No ticket found for this player at start_flip ' + startFlip + '.');
      }
      break;
    }

    case 'withdraw-fees': {
      const game = await program.account.game.fetch(gamePDA);
      const amount = process.argv[3] ? parseInt(process.argv[3]) : Number(game.operatorPool.toString());
      if (amount <= 0) { console.log('No operator fees to withdraw.'); return; }

      const authorityATA = await getAssociatedTokenAddress(USDC_MINT, wallet.publicKey);
      console.log('Withdrawing ' + fmtUsdc(amount) + ' USDC in operator fees...');
      const tx = await program.methods.withdrawFees(new anchor.BN(amount)).accounts({
        authority: wallet.publicKey,
        game: gamePDA,
        authorityTokenAccount: authorityATA,
        vault: vaultPDA,
        tokenProgram: TOKEN_PROGRAM_ID,
      }).rpc();
      console.log('Fees withdrawn! TX:', tx);
      break;
    }

    case 'close-game-v1': {
      console.log('Closing old game PDA...');
      const tx = await program.methods.closeGameV1().accounts({
        authority: wallet.publicKey,
        game: gamePDA,
      }).rpc();
      console.log('Old game PDA closed! TX:', tx);
      break;
    }

    default:
      console.error('Unknown command:', cmd);
      console.log('Commands: init, enter, flip, claim, status, ticket, withdraw-fees, close-game-v1');
      process.exit(1);
  }
}

main().catch(e => {
  console.error('Error:', e.message || e);
  process.exit(1);
});
