# 优化的金融分析技能使用指南

## 概述

优化的金融分析技能整合了滚动窗口调仓逻辑和回测功能，提供更真实的投资组合分析。

## 核心功能

### 1. 滚动窗口风险平价分析
- **避免未来数据**：使用历史一年数据计算波动率
- **动态调仓**：每月根据最新波动率重新计算权重
- **风险平价**：各资产对组合风险贡献相等
- **回测验证**：完整的回测流程，验证策略效果

### 2. 数据处理
- **CSV数据导入**：支持中国市场数据
- **百分比转换**：自动将百分比数据转换为实际收益率
- **异常值处理**：识别和处理异常值
- **缺失值处理**：删除包含NaN的行

### 3. 分析指标
- **总收益率**：整个回测期间的总收益
- **年化收益率**：年化后的收益率
- **年化波动率**：年化后的波动率
- **最大回撤**：最大回撤幅度
- **夏普比率**：风险调整后的收益

## 安装依赖

```bash
pip install pandas numpy matplotlib seaborn
```

## 使用方法

### 1. 基本使用

#### 运行回测分析
```bash
python optimized_main.py --csv "C:\path\to\marketdata.csv" --output ./backtest_output
```

#### 使用默认数据路径
```bash
python optimized_main.py
```

#### 自定义输出目录
```bash
python optimized_main.py --output ./my_backtest
```

### 2. 查看分析结果

#### 显示分析报告
```bash
python optimized_main.py --show-report
```

#### 显示分析指标
```bash
python optimized_main.py --show-metrics
```

#### 显示资产权重
```bash
python optimized_main.py --show-weights
```

### 3. 直接使用技能类

```python
from optimized_risk_parity_skill import OptimizedRiskParitySkill

# 创建技能实例
skill = OptimizedRiskParitySkill('C:\path\to\marketdata.csv')

# 运行回测
result = skill.run_backtest('C:\path\to\marketdata.csv', './backtest_output')

# 获取报告
report = skill.get_report()
print(report)

# 获取指标
metrics = skill.get_metrics()
print(f"总收益率: {metrics['total_return']*100:.2f}%")
print(f"夏普比率: {metrics['sharpe_ratio']:.2f}")

# 获取权重
weights = skill.get_weights()
for asset, weight in weights.items():
    print(f"{asset}: {weight*100:.2f}%")
```

## 数据格式要求

### CSV文件格式
- 第0行：资产名称（如：CFFEX5五年期国债期货）
- 第1行：数据类型（如：涨跌幅(%)）
- 第2行开始：实际数据
  - 第1列：日期
  - 第2-5列：各资产收益率（百分比格式）

### 示例数据
```
日期,TF.CFE,T.CFE,CU.SHF,AU.SHF
2015-03-20,0.1377,NaN,0.767,-0.1472
2015-03-23,-0.1834,-0.0721,3.9248,1.1373
2015-03-24,-0.0510,-0.1031,0.3662,0.4165
```

## 分析流程

### 1. 数据加载
- 读取CSV文件
- 提取实际数据（跳过前两行）
- 将百分比数据转换为实际收益率（除以100）
- 删除缺失值

### 2. 滚动波动率计算
- 使用expanding窗口计算历史波动率
- 最小窗口大小：252个交易日（约1年）
- 年化波动率：波动率 × √252

### 3. 权重计算
- 基于波动率倒数加权：`权重 = (1/波动率) / ∑(1/波动率)`
- 每月调仓，动态调整权重
- 使用前一天的权重计算当日收益率

### 4. 投资组合收益率计算
- 使用滚动权重计算每日收益率
- 累计收益：`(1 + 每日收益率).cumprod()`

### 5. 绩效评估
- 总收益率：累计收益 - 1
- 年化收益率：`(1 + 总收益率)^(252/天数) - 1`
- 年化波动率：每日收益率标准差 × √252
- 最大回撤：累计收益最大值与当前值的差值
- 夏普比率：`(年化收益率 - 无风险利率) / 年化波动率`

### 6. 可视化
- 收益曲线图：累计收益随时间变化
- 资产配置饼图：各资产平均权重
- 相关性热力图：资产间相关性
- 资产收益对比图：各资产累计收益
- 滚动权重变化图：权重随时间变化

## 输出文件

### 分析报告
- **文件名**: `rolling_risk_parity_report.txt`
- **内容**: 完整的分析报告，包括配置、收益、风险、建议等

### 详细数据
- **文件名**: `rolling_risk_parity_data.json`
- **内容**: 
  - 投资组合配置
  - 分析指标
  - 资产映射
  - 滚动权重数据

### 可视化图表
1. **收益曲线图**: `rolling_risk_parity_returns.png`
2. **资产配置饼图**: `rolling_risk_parity_allocation.png`
3. **相关性热力图**: `rolling_risk_parity_correlation.png`
4. **资产收益对比图**: `rolling_asset_returns_comparison.png`
5. **滚动权重变化图**: `rolling_weight_changes.png`

## 示例输出

### 分析报告
```
中国市场滚动风险平价组合分析报告
============================================================
生成时间: 2026-02-26 10:20:04
数据来源: C:\Users\wu_zhuoran\.openclaw\workspace\data\marketdata.csv
数据时间范围: 2015-03-23 至 2026-02-25
数据点数: 2656
滚动窗口: 252个交易日（约1年）
调仓频率: 每月

平均投资组合配置:
  五年期国债 (TF.CFE): 46.04% (波动率: 2.46%)
  十年期国债 (T.CFE): 32.55% (波动率: 3.61%)
  沪铜 (CU.SHF): 9.33% (波动率: 17.87%)
  沪金 (AU.SHF): 12.09% (波动率: 14.88%)

收益指标:
  总收益率: 44.78%
  年化收益率: 3.57%

风险指标:
  年化波动率: 3.76%
  最大回撤: -9.48%
  夏普比率: 0.42
```

## 优化特点

### 1. 避免未来数据
- 使用expanding窗口而非rolling窗口
- 确保不使用未来数据
- 最小窗口大小为252个交易日

### 2. 动态调仓
- 每月根据最新波动率重新计算权重
- 模拟真实投资中的调仓操作
- 适应市场变化

### 3. 风险平价
- 各资产对组合风险贡献相等
- 降低单一资产风险暴露
- 适合长期持有

### 4. 回测验证
- 完整的回测流程
- 避免数据泄露
- 提供详细的绩效指标

## 注意事项

### 数据质量
- 确保数据质量，处理缺失值和异常值
- 检查数据格式是否正确
- 验证数据时间范围

### 参数调整
- 滚动窗口大小可根据实际情况调整
- 调仓频率可根据市场情况调整
- 考虑交易成本对回测结果的影响

### 结果解读
- 夏普比率 > 1: 投资组合表现良好
- 夏普比率 0-1: 投资组合表现一般
- 夏普比率 < 0: 投资组合表现不佳
- 最大回撤 > 20%: 风险较高

## 常见问题

### Q: 如何处理异常值？
A: 脚本会自动检查数据中的异常值，但建议在分析前检查数据质量。

### Q: 如何调整调仓频率？
A: 修改`rolling_window_risk_parity.py`中的`rebalance_frequency`参数。

### Q: 如何调整滚动窗口大小？
A: 修改`rolling_window_risk_parity.py`中的`window_size`参数。

### Q: 如何添加更多资产？
A: 在`asset_mapping`和`asset_types`中添加新资产信息。

### Q: 如何比较不同策略？
A: 运行多次分析，比较不同参数下的结果。

## 技术细节

### 滚动窗口计算
```python
# 使用expanding窗口计算波动率
rolling_vol = returns_data[asset].expanding(min_periods=window_size).std() * np.sqrt(252)
```

### 权重计算
```python
# 基于波动率倒数加权
total_vol = sum(1/v for v in current_vols.values())
weight = (1/current_vols[asset]) / total_vol
```

### 投资组合收益率
```python
# 使用前一天的权重计算当日收益率
portfolio_return = sum(daily_returns[asset] * weights[asset] for asset in assets)
```

## 总结

优化的金融分析技能提供了完整的滚动窗口风险平价组合分析功能，避免了未来数据问题，提供了更真实的回测结果。通过动态调仓和风险平价策略，为投资者提供有价值的参考。