#!/bin/bash

# Telegram 群组语音消息发送脚本

# 检查依赖
if ! command -v edge-tts &> /dev/null; then
    echo "错误: 未找到 edge-tts，请先安装: pip install edge-tts"
    exit 1
fi

if ! command -v ffmpeg &> /dev/null; then
    echo "错误: 未找到 ffmpeg，请先安装"
    exit 1
fi

if [ $# -lt 2 ]; then
    echo "用法: $0 \"语音内容\" \"群组会话键\" [语音类型] [语速]"
    echo "示例: $0 \"你好，这是一条测试消息\" \"agent:main:telegram:group:-1001234567890\" \"zh-CN-XiaoxiaoNeural\" \"+5%\""
    exit 1
fi

TEXT="$1"
SESSION_KEY="$2"
VOICE="${3:-zh-CN-XiaoxiaoNeural}"  # 默认使用晓晓声线
RATE="${4:-+5%}"                    # 默认语速+5%

# 生成临时文件名
TEMP_FILE=$(mktemp -p /tmp voice_XXXXXXXX)
TEMP_MP3="${TEMP_FILE}.mp3"
TEMP_OGG="${TEMP_FILE}.ogg"

# 清理函数
cleanup() {
    rm -f "$TEMP_MP3" "$TEMP_OGG"
}
trap cleanup EXIT

echo "正在生成语音..."
edge-tts --voice "$VOICE" --rate="$RATE" --text "$TEXT" --write-media "$TEMP_MP3"

if [ $? -ne 0 ]; then
    echo "错误: 生成语音失败"
    exit 1
fi

echo "正在转换为Telegram兼容格式..."
ffmpeg -y -i "$TEMP_MP3" -c:a libopus -b:a 48k -ac 1 -ar 48000 -application voip "$TEMP_OGG" >/dev/null 2>&1

if [ $? -ne 0 ]; then
    echo "错误: 转换音频格式失败"
    exit 1
fi

echo "正在发送到群组..."

# 使用 OpenClaw 的 sessions_send 功能发送语音
node -e "
const { sessions_send } = require('@openclaw/core');
sessions_send({
  sessionKey: '$SESSION_KEY',
  message: 'MEDIA:$TEMP_OGG'
}).then(result => {
  console.log('语音消息发送成功');
  console.log(result);
}).catch(err => {
  console.error('发送失败:', err);
});
"

echo "操作完成"