#!/bin/bash
# Rename Telegram forum topic (with optional icon)
# 重命名 Telegram 论坛话题（可选更换图标）
#
# Usage / 用法:
#   ./rename-topic.sh <chat_id> <thread_id> <new_name> [icon]
#   ./rename-topic.sh --icons  # Show available icons / 显示可用图标
#
# Icon can be / 图标可以是:
#   - Emoji (from the map below / 下面的映射表)
#   - Full custom_emoji_id (19-digit number / 19位数字)
#   - Omitted to keep current icon / 省略则保持当前图标
#
# Requires / 需要: TELEGRAM_BOT_TOKEN environment variable
#
# Data source: Telegram Bot API getForumTopicIconStickers (2026-02-07)

# Get token from environment
BOT_TOKEN="${TELEGRAM_BOT_TOKEN:-}"

if [ -z "$BOT_TOKEN" ] && [ "$1" != "--icons" ]; then
  echo "Error: TELEGRAM_BOT_TOKEN not set"
  echo "错误：未设置 TELEGRAM_BOT_TOKEN"
  echo ""
  echo "Set it in your environment or OpenClaw config:"
  echo "在环境变量或 OpenClaw 配置中设置："
  echo "  export TELEGRAM_BOT_TOKEN=\"your-bot-token\""
  exit 1
fi

# Hardcoded emoji → ID mapping (112 icons, verified from API)
declare -A ICON_MAP=(
  ["📰"]="5434144690511290129"
  ["💡"]="5312536423851630001"
  ["⚡️"]="5312016608254762256"
  ["🎙"]="5377544228505134960"
  ["🔝"]="5418085807791545980"
  ["🗣"]="5370870893004203704"
  ["🆒"]="5420216386448270341"
  ["❗️"]="5379748062124056162"
  ["📝"]="5373251851074415873"
  ["📆"]="5433614043006903194"
  ["📁"]="5357315181649076022"
  ["🔎"]="5309965701241379366"
  ["📣"]="5309984423003823246"
  ["🔥"]="5312241539987020022"
  ["❤️"]="5312138559556164615"
  ["❓"]="5377316857231450742"
  ["📈"]="5350305691942788490"
  ["📉"]="5350713563512052787"
  ["💎"]="5309958691854754293"
  ["💰"]="5350452584119279096"
  ["💸"]="5309929258443874898"
  ["🪙"]="5377690785674175481"
  ["💱"]="5310107765874632305"
  ["⁉️"]="5377438129928020693"
  ["🎮"]="5309950797704865693"
  ["💻"]="5350554349074391003"
  ["📱"]="5409357944619802453"
  ["🚗"]="5312322066328853156"
  ["🏠"]="5312486108309757006"
  ["💘"]="5310029292527164639"
  ["🎉"]="5310228579009699834"
  ["‼️"]="5377498341074542641"
  ["🏆"]="5312315739842026755"
  ["🏁"]="5408906741125490282"
  ["🎬"]="5368653135101310687"
  ["🎵"]="5310045076531978942"
  ["🔞"]="5420331611830886484"
  ["📚"]="5350481781306958339"
  ["👑"]="5357107601584693888"
  ["⚽️"]="5375159220280762629"
  ["🏀"]="5384327463629233871"
  ["📺"]="5350513667144163474"
  ["👀"]="5357121491508928442"
  ["🫦"]="5357185426392096577"
  ["🍓"]="5310157398516703416"
  ["💄"]="5310262535021142850"
  ["👠"]="5368741306484925109"
  ["✈️"]="5348436127038579546"
  ["🧳"]="5357120306097956843"
  ["🏖"]="5310303848311562896"
  ["⛅️"]="5350424168615649565"
  ["🦄"]="5413625003218313783"
  ["🛍"]="5350699789551935589"
  ["👜"]="5377478880577724584"
  ["🛒"]="5431492767249342908"
  ["🚂"]="5350497316203668441"
  ["🛥"]="5350422527938141909"
  ["🏔"]="5418196338774907917"
  ["🏕"]="5350648297189023928"
  ["🤖"]="5309832892262654231"
  ["🪩"]="5350751634102166060"
  ["🎟"]="5377624166436445368"
  ["🏴‍☠️"]="5386395194029515402"
  ["🗳"]="5350387571199319521"
  ["🎓"]="5357419403325481346"
  ["🔭"]="5368585403467048206"
  ["🔬"]="5377580546748588396"
  ["🎶"]="5377317729109811382"
  ["🎤"]="5382003830487523366"
  ["🕺"]="5357298525765902091"
  ["💃"]="5357370526597653193"
  ["🪖"]="5357188789351490453"
  ["💼"]="5348227245599105972"
  ["🧪"]="5411138633765757782"
  ["👨‍👩‍👧‍👦"]="5386435923204382258"
  ["👶"]="5377675010259297233"
  ["🤰"]="5386609083400856174"
  ["💅"]="5368808634392257474"
  ["🏛"]="5350548830041415279"
  ["🧮"]="5355127101970194557"
  ["🖨"]="5386379624773066504"
  ["👮‍♂️"]="5377494501373780436"
  ["🩺"]="5350307998340226571"
  ["💊"]="5310094636159607472"
  ["💉"]="5310139157790596888"
  ["🧼"]="5377468357907849200"
  ["🪪"]="5418115271267197333"
  ["🛃"]="5372819184658949787"
  ["🍽"]="5350344462612570293"
  ["🐟"]="5384574037701696503"
  ["🎨"]="5310039132297242441"
  ["🎭"]="5350658016700013471"
  ["🎩"]="5357504778685392027"
  ["🔮"]="5350367161514732241"
  ["🍹"]="5350520238444126134"
  ["🎂"]="5310132165583840589"
  ["☕️"]="5350392020785437399"
  ["🍣"]="5350406176997646350"
  ["🍔"]="5350403544182694064"
  ["🍕"]="5350444672789519765"
  ["🦠"]="5312424913615723286"
  ["💬"]="5417915203100613993"
  ["🎄"]="5312054580060625569"
  ["🎃"]="5309744892677727325"
  ["✍️"]="5238156910363950406"
  ["⭐️"]="5235579393115438657"
  ["✅"]="5237699328843200968"
  ["🎖"]="5238027455754680851"
  ["🤡"]="5238234236955148254"
  ["🧠"]="5237889595894414384"
  ["🦮"]="5237999392438371490"
  ["🐈"]="5235912661102773458"
)

show_icons() {
  echo "Available icons (112) / 可用图标 (112个):"
  echo ""
  echo "【News/Status 新闻/状态】📰 📣 ❗️ ‼️ ⁉️ ❓ 🔝 🆒 🔞 ⭐️ ✅"
  echo "【Energy/Weather 能量/天气】🔥 ⚡️ ⛅️ 🏔"
  echo "【Ideas 创意/想法】💡 🔮 🎨"
  echo "【Work/Office 工作/办公】📝 📆 📁 🔎 📈 📉 💼 🖨 ✍️"
  echo "【Tech/Code 科技/编程】💻 📱 🤖 🧠 🧮"
  echo "【Science 科学/研究】🔬 🔭 🧪 🦠"
  echo "【Education 学习/教育】📚 🎓"
  echo "【Communication 沟通/社交】💬 🗣 🎙 🎤"
  echo "【Finance/Shopping 金融/购物】💰 💸 🪙 💱 💎 🛒 🛍 👜 👠"
  echo "【Entertainment 娱乐/游戏】🎮 🎬 🎵 🎶 📺 🎭 🪩 🎟"
  echo "【Sports 运动】⚽️ 🏀 🏁 💃 🕺"
  echo "【Travel 旅行/交通】✈️ 🚗 🚂 🛥 🧳 🛃 🏖 🏕"
  echo "【Family/Life 家庭/生活】🏠 👨‍👩‍👧‍👦 👶 🤰"
  echo "【Emotions 情感】❤️ 💘"
  echo "【Food/Drinks 食物/饮料】☕️ 🍔 🍕 🍣 🍹 🍽 🍓 🎂"
  echo "【Health 健康/医疗】🩺 💊 💉 🧼"
  echo "【Celebration 节日/庆祝】🎉 🏆 🎖 🎄 🎃"
  echo "【Characters 人物/角色】👑 👀 🎩 🤡 👮‍♂️ 🏴‍☠️ 🪖"
  echo "【Animals 动物】🐈 🐟 🦄 🦮"
  echo "【Beauty 美容】💄 💅 🫦"
  echo "【Other 其他】🏛 🗳 🪪"
}

# Check for --icons flag
if [ "$1" == "--icons" ]; then
  show_icons
  exit 0
fi

CHAT_ID="$1"
THREAD_ID="$2"
NAME="$3"
ICON="$4"

if [ -z "$CHAT_ID" ] || [ -z "$THREAD_ID" ] || [ -z "$NAME" ]; then
  echo "Usage / 用法: $0 <chat_id> <thread_id> <new_name> [icon]"
  echo "              $0 --icons  # Show available icons / 显示可用图标"
  echo ""
  echo "Examples / 示例:"
  echo "  $0 288536750 145417 'Test Topic' 💻"
  echo "  $0 288536750 145417 '测试话题' 5350554349074391003"
  exit 1
fi

# Check name length
CHAR_COUNT=$(echo -n "$NAME" | wc -m)
if [ "$CHAR_COUNT" -gt 10 ]; then
  echo "Warning: Name is $CHAR_COUNT chars, exceeds 10 char limit"
  echo "警告：名称 $CHAR_COUNT 字符，超过 10 字符限制"
fi

# Build curl arguments
CURL_ARGS=(-s "https://api.telegram.org/bot${BOT_TOKEN}/editForumTopic")
CURL_ARGS+=(-d "chat_id=${CHAT_ID}")
CURL_ARGS+=(-d "message_thread_id=${THREAD_ID}")
CURL_ARGS+=(--data-urlencode "name=${NAME}")

# Add icon if provided
if [ -n "$ICON" ]; then
  # Check if it's a 19-digit ID
  if [[ "$ICON" =~ ^[0-9]{19}$ ]]; then
    ICON_ID="$ICON"
  elif [ -n "${ICON_MAP[$ICON]}" ]; then
    ICON_ID="${ICON_MAP[$ICON]}"
  else
    echo "Warning: Icon '$ICON' not found, skipping icon change"
    echo "警告：图标 '$ICON' 未找到，跳过图标更换"
    ICON_ID=""
  fi
  
  if [ -n "$ICON_ID" ]; then
    CURL_ARGS+=(-d "icon_custom_emoji_id=${ICON_ID}")
    echo "Using icon ID / 使用图标 ID: $ICON_ID"
  fi
fi

# Execute
curl "${CURL_ARGS[@]}" | jq .
