#!/usr/bin/env node

const fs = require('fs');
const { execSync } = require('child_process');
const path = require('path');

// 获取命令行参数中的机器人令牌
const BOT_TOKEN = process.argv[2];

if (!BOT_TOKEN) {
    console.error('错误: 请提供机器人令牌作为参数');
    console.error('用法: node deploy.js <BOT_TOKEN>');
    process.exit(1);
}

console.log('开始部署 Telegram 配对代码自动批准机器人...');

try {
    // 1. 创建机器人脚本
    const botScript = `const { Telegraf } = require('telegraf');
const { exec } = require('child_process');

// 使用提供的机器人令牌
const BOT_TOKEN = '${BOT_TOKEN}';

// 创建机器人实例
const bot = new Telegraf(BOT_TOKEN);

// 中间件：记录所有传入的消息
bot.use((ctx, next) => {
    console.log('Received message:', ctx.message);
    return next();
});

// 监听文本消息
bot.on('text', (ctx) => {
    const message = ctx.message.text.trim();
    const chatId = ctx.message.chat.id;
    const userId = ctx.message.from.id;

    console.log(\`Received message from user \${userId} in chat \${chatId}: \${message}\`);

    // 检查消息是否匹配配对代码格式
    const patterns = [
        /^([A-Z0-9]{8})$/, // NDW4JDJ4 格式
        /^code:\\s*([A-Z0-9]{8})$/i, // code: NDW4JDJ4 格式
        /^Pairing\\s+code:\\s*([A-Z0-9]{8})$/i // Pairing code: NDW4JDJ4 格式
    ];

    let pairingCode = null;

    for (const pattern of patterns) {
        const match = message.match(pattern);
        if (match) {
            pairingCode = match[1];
            console.log(\`Found pairing code: \${pairingCode}\`);
            break;
        }
    }

    if (pairingCode) {
        console.log(\`Approving pairing code: \${pairingCode} for user: \${userId}\`);

        // 执行配对批准命令
        const command = \`openclaw pairing approve telegram \${pairingCode}\`;
        
        exec(command, (error, stdout, stderr) => {
            if (error) {
                console.error(\`Error executing command: \${error}\`);
                ctx.reply(\`配对失败: \${error.message}\`);
                return;
            }
            
            if (stderr) {
                console.error(\`stderr: \${stderr}\`);
                ctx.reply(\`配对过程中出现警告: \${stderr}\`);
                return;
            }

            console.log(\`Successfully executed: \${command}\`);
            ctx.reply(\`配对成功！用户 \${userId} 现在已被授权.\`);
        });
    } else {
        // 如果消息不符合配对代码格式，则发送提示
        console.log(\`Message does not match pairing code format: \${message}\`);
        ctx.reply(\`请发送配对代码以获得访问权限。

支持格式: 
NDW4JDJ4
code: NDW4JDJ4  
Pairing code: NDW4JDJ4\`);
    }
});

// 启动机器人
bot.launch()
    .then(() => {
        console.log('Telegram Pairing Approval Bot is running...');
    })
    .catch(error => {
        console.error('Failed to launch bot:', error);
    });

// 启用优雅关闭
process.once('SIGINT', () => bot.stop('SIGINT'));
process.once('SIGTERM', () => bot.stop('SIGTERM'));
`;

    const botScriptPath = path.join(__dirname, '../simple_telegram_bot.js');
    fs.writeFileSync(botScriptPath, botScript);
    console.log('✓ 机器人脚本已创建');

    // 2. 创建 package.json 文件
    const packageJson = {
        name: "telegram-pairing-approver",
        version: "1.0.0",
        description: "Telegram bot for automatic approval of pairing codes",
        main: "simple_telegram_bot.js",
        scripts: {
            start: "node simple_telegram_bot.js"
        },
        dependencies: {
            "telegraf": "^4.12.2"
        },
        keywords: ["telegram", "bot", "pairing", "approval"],
        author: "OpenClaw",
        license: "MIT"
    };

    const packageJsonPath = path.join(__dirname, '../package.json');
    fs.writeFileSync(packageJsonPath, JSON.stringify(packageJson, null, 2));
    console.log('✓ package.json 文件已创建');

    // 3. 安装依赖
    console.log('正在安装依赖...');
    execSync('npm install', { cwd: path.dirname(botScriptPath), stdio: 'inherit' });
    console.log('✓ 依赖已安装');

    // 4. 创建系统服务文件
    const serviceContent = `[Unit]
Description=Telegram Pairing Code Approval Bot
After=network.target

[Service]
Type=simple
User=root
WorkingDirectory=${path.dirname(botScriptPath)}
ExecStart=/usr/bin/node ${botScriptPath}
Restart=always
RestartSec=10
Environment=NODE_ENV=production

[Install]
WantedBy=multi-user.target`;

    const serviceFilePath = '/etc/systemd/system/telegram-pairing-bot.service';
    fs.writeFileSync(serviceFilePath, serviceContent);
    console.log('✓ 系统服务文件已创建');

    // 5. 重新加载systemd守护进程
    execSync('systemctl daemon-reload');
    console.log('✓ Systemd 守护进程已重新加载');

    // 6. 如果服务已存在，先停止它
    try {
        execSync('systemctl stop telegram-pairing-bot.service', { stdio: 'ignore' });
    } catch (e) {
        // 服务可能不存在，忽略错误
    }

    // 7. 启用并启动服务
    execSync('systemctl enable telegram-pairing-bot.service');
    execSync('systemctl start telegram-pairing-bot.service');
    
    console.log('✓ 服务已启用并启动');

    // 8. 检查服务状态
    const status = execSync('systemctl is-active telegram-pairing-bot.service').toString().trim();
    console.log('✓ 服务状态:', status);

    console.log('\\n✓ Telegram 配对代码自动批准机器人服务部署完成！');
    console.log('✓ 服务将在系统启动时自动运行，并具备自动重启功能');
    
} catch (error) {
    console.error('部署过程中发生错误:', error.message);
    process.exit(1);
}