/**
 * WebSocket Relay Server for telebiz-tt MCP Bridge
 *
 * This relay server connects:
 * - Browser (executor): has the authenticated Telegram session
 * - MCP clients: send tool execution requests
 *
 * Protocol:
 * - Executor sends: { type: 'register', role: 'executor' }
 * - Client sends: { type: 'register', role: 'client' }
 * - Client sends: { id, type: 'execute', tool, args }
 * - Relay forwards to executor, returns response to client
 *
 * Features:
 * - Connection tracking with metrics
 * - Heartbeat/ping support
 * - Graceful shutdown
 * - Status reporting
 */

import { WebSocketServer, WebSocket } from 'ws';

const PORT = parseInt(process.env.TELEBIZ_PORT || '9716', 10);

interface McpRequest {
  id?: string;
  type: 'execute' | 'list_tools' | 'list_skills' | 'ping' | 'register' | 'status';
  tool?: string;
  args?: Record<string, unknown>;
  role?: 'executor' | 'client';
}

interface McpResponse {
  id?: string;
  type: 'result' | 'tools' | 'skills' | 'pong' | 'error' | 'register';
  role?: 'executor' | 'client';
  success?: boolean;
  data?: unknown;
  error?: string;
  tools?: Array<{
    name: string;
    description: string;
    inputSchema: {
      type: 'object';
      properties: Record<string, unknown>;
      required: string[];
    };
  }>;
}

// Connection state
let executor: WebSocket | null = null;
const clients = new Set<WebSocket>();
const pendingRequests = new Map<string, {
  client: WebSocket;
  resolve: (response: McpResponse) => void;
  reject: (error: Error) => void;
  timeout: ReturnType<typeof setTimeout>;
}>();

const REQUEST_TIMEOUT = 60000; // 60 seconds
const HEARTBEAT_INTERVAL = 30000; // 30 seconds

// Metrics
const metrics = {
  startTime: Date.now(),
  totalRequests: 0,
  successfulRequests: 0,
  failedRequests: 0,
  executorConnects: 0,
  executorDisconnects: 0,
  clientConnects: 0,
  clientDisconnects: 0,
};

function generateId(): string {
  return `${Date.now()}-${Math.random().toString(36).slice(2, 11)}`;
}

function getStatus(): Record<string, unknown> {
  return {
    uptime: Math.floor((Date.now() - metrics.startTime) / 1000),
    executorConnected: executor !== null && executor.readyState === WebSocket.OPEN,
    clientCount: clients.size,
    pendingRequests: pendingRequests.size,
    metrics: { ...metrics },
  };
}

function log(msg: string, ...args: unknown[]) {
  console.log(`[relay] ${msg}`, ...args);
}

function sendToExecutor(request: McpRequest): Promise<McpResponse> {
  return new Promise((resolve, reject) => {
    if (!executor || executor.readyState !== WebSocket.OPEN) {
      reject(new Error('Executor not connected'));
      return;
    }

    const id = request.id || generateId();
    const requestWithId = { ...request, id };

    const timeout = setTimeout(() => {
      pendingRequests.delete(id);
      reject(new Error('Request timeout'));
    }, REQUEST_TIMEOUT);

    pendingRequests.set(id, {
      client: null as any, // We'll handle response directly
      resolve,
      reject,
      timeout,
    });

    executor.send(JSON.stringify(requestWithId));
  });
}

const wss = new WebSocketServer({ port: PORT });

log(`Starting relay server on port ${PORT}`);

wss.on('connection', (ws: WebSocket) => {
  log('New connection');

  ws.on('message', async (data: Buffer) => {
    let message: McpRequest;
    try {
      message = JSON.parse(data.toString());
    } catch (e) {
      ws.send(JSON.stringify({ type: 'error', error: 'Invalid JSON' }));
      return;
    }

    // Handle registration
    if (message.type === 'register') {
      if (message.role === 'executor') {
        if (executor) {
          log('Replacing existing executor');
          executor.close();
        }
        executor = ws;
        metrics.executorConnects++;
        log('Executor connected');
        ws.send(JSON.stringify({ type: 'register', success: true }));
      } else if (message.role === 'client') {
        clients.add(ws);
        metrics.clientConnects++;
        log(`Client connected (${clients.size} total)`);
        ws.send(JSON.stringify({ type: 'register', success: true }));
      }
      return;
    }

    // Handle status request
    if (message.type === 'status') {
      ws.send(JSON.stringify({ type: 'status', ...getStatus() }));
      return;
    }

    // Handle responses from executor
    if (ws === executor && message.id) {
      const pending = pendingRequests.get(message.id);
      if (pending) {
        clearTimeout(pending.timeout);
        pendingRequests.delete(message.id);
        pending.resolve(message as McpResponse);
      }
      return;
    }

    // Handle requests from clients
    if (clients.has(ws)) {
      try {
        const response = await sendToExecutor(message);
        ws.send(JSON.stringify(response));
      } catch (error) {
        ws.send(JSON.stringify({
          id: message.id,
          type: 'error',
          success: false,
          error: error instanceof Error ? error.message : 'Unknown error',
        }));
      }
      return;
    }

    // Unknown connection - reject
    ws.send(JSON.stringify({
      type: 'error',
      error: 'Please register first with { type: "register", role: "executor"|"client" }',
    }));
  });

  ws.on('close', () => {
    if (ws === executor) {
      log('Executor disconnected');
      executor = null;
      metrics.executorDisconnects++;
    } else if (clients.has(ws)) {
      clients.delete(ws);
      metrics.clientDisconnects++;
      log(`Client disconnected (${clients.size} remaining)`);
    }
  });

  ws.on('error', (error) => {
    log('WebSocket error:', error.message);
  });
});

// Handle graceful shutdown
process.on('SIGINT', () => {
  log('Shutting down...');
  wss.close(() => {
    process.exit(0);
  });
});

// Status endpoint via stdin (for debugging)
process.stdin.on('data', (data) => {
  const cmd = data.toString().trim();
  if (cmd === 'status') {
    console.log({
      executorConnected: executor !== null,
      clientCount: clients.size,
      pendingRequests: pendingRequests.size,
    });
  }
});

log(`Relay server running on ws://localhost:${PORT}`);
log('Waiting for executor (browser) and clients to connect...');
